/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * GTK Calendar Widget
 * Copyright (C) 1998 Cesar Miquel, Shawn T. Amundson and Mattias Groenlund
 * 
 * lib_date routines
 * Copyright (c) 1995, 1996, 1997, 1998 by Steffen Beyer
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-2000.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE__NL_TIME_FIRST_WEEKDAY
#include <langinfo.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include <gdk/gdkkeysyms.h>
#include "gtkkimicalendar.h"

#define _gtk_marshal_VOID__VOID	g_cclosure_marshal_VOID__VOID
#define GTK_PARAM_READABLE G_PARAM_READABLE|G_PARAM_STATIC_NAME|G_PARAM_STATIC_NICK|G_PARAM_STATIC_BLURB
#define GTK_PARAM_WRITABLE G_PARAM_WRITABLE|G_PARAM_STATIC_NAME|G_PARAM_STATIC_NICK|G_PARAM_STATIC_BLURB
#define GTK_PARAM_READWRITE G_PARAM_READWRITE|G_PARAM_STATIC_NAME|G_PARAM_STATIC_NICK|G_PARAM_STATIC_BLURB

/***************************************************************************/
/* The following date routines are taken from the lib_date package. 
 * They have been minimally edited to avoid conflict with types defined
 * in win32 headers.
 */

static const guint month_length[2][13] =
{
  { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 },
  { 0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 }
};

static const guint days_in_months[2][14] =
{
  { 0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
  { 0, 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }
};

static glong  calc_days(guint year, guint mm, guint dd);
static guint  day_of_week(guint year, guint mm, guint dd);
static glong  dates_difference(guint year1, guint mm1, guint dd1,
			       guint year2, guint mm2, guint dd2);
static guint  weeks_in_year(guint year);

static gboolean 
leap (guint year)
{
  return((((year % 4) == 0) && ((year % 100) != 0)) || ((year % 400) == 0));
}

static guint 
day_of_week (guint year, guint mm, guint dd)
{
  glong  days;
  
  days = calc_days(year, mm, dd);
  if (days > 0L)
    {
      days--;
      days %= 7L;
      days++;
    }
  return( (guint) days );
}

static guint weeks_in_year(guint year)
{
  return(52 + ((day_of_week(year,1,1)==4) || (day_of_week(year,12,31)==4)));
}

static gboolean 
check_date(guint year, guint mm, guint dd)
{
  if (year < 1) return FALSE;
  if ((mm < 1) || (mm > 12)) return FALSE;
  if ((dd < 1) || (dd > month_length[leap(year)][mm])) return FALSE;
  return TRUE;
}

static guint 
week_number(guint year, guint mm, guint dd)
{
  guint first;
  
  first = day_of_week(year,1,1) - 1;
  return( (guint) ( (dates_difference(year,1,1, year,mm,dd) + first) / 7L ) +
	  (first < 4) );
}

static glong 
year_to_days(guint year)
{
  return( year * 365L + (year / 4) - (year / 100) + (year / 400) );
}


static glong 
calc_days(guint year, guint mm, guint dd)
{
  gboolean lp;
  
  if (year < 1) return(0L);
  if ((mm < 1) || (mm > 12)) return(0L);
  if ((dd < 1) || (dd > month_length[(lp = leap(year))][mm])) return(0L);
  return( year_to_days(--year) + days_in_months[lp][mm] + dd );
}

static gboolean 
week_of_year(guint *week, guint *year, guint mm, guint dd)
{
  if (check_date(*year,mm,dd))
    {
      *week = week_number(*year,mm,dd);
      if (*week == 0) 
	*week = weeks_in_year(--(*year));
      else if (*week > weeks_in_year(*year))
	{
	  *week = 1;
	  (*year)++;
	}
      return TRUE;
    }
  return FALSE;
}

static glong 
dates_difference(guint year1, guint mm1, guint dd1,
		 guint year2, guint mm2, guint dd2)
{
  return( calc_days(year2, mm2, dd2) - calc_days(year1, mm1, dd1) );
}

/*** END OF lib_date routines ********************************************/

/* Spacing around day/week headers and main area, inside those windows */
#define CALENDAR_MARGIN		 0
/* Spacing around day/week headers and main area, outside those windows */
#define INNER_BORDER		 0
/* Separation between day headers and main area */
#define CALENDAR_YSEP		 2
/* Separation between week headers and main area */
#define CALENDAR_XSEP		 2

#define DAY_XSEP		 0 /* not really good for small calendar */
#define DAY_YSEP		 0 /* not really good for small calendar */

#define SCROLL_DELAY_FACTOR      5

#define STEP 2     /* space day/day */
#define PLUS_H 202 /* + max height pix */

/* Color usage */
#define HEADER_FG_COLOR(widget)		 (& (widget)->style->fg[GTK_WIDGET_STATE (widget)])
#define HEADER_BG_COLOR(widget)		 (& (widget)->style->bg[GTK_WIDGET_STATE (widget)])
#define SELECTED_BG_COLOR(widget)	 (& (widget)->style->base[GTK_WIDGET_HAS_FOCUS (widget) ? GTK_STATE_SELECTED : GTK_STATE_ACTIVE])
#define SELECTED_FG_COLOR(widget)	 (& (widget)->style->text[GTK_WIDGET_HAS_FOCUS (widget) ? GTK_STATE_SELECTED : GTK_STATE_ACTIVE])
#define NORMAL_DAY_COLOR(widget)	 (& (widget)->style->text[GTK_WIDGET_STATE (widget)])
#define PREV_MONTH_COLOR(widget)	 (& (widget)->style->mid[GTK_WIDGET_STATE (widget)])
#define NEXT_MONTH_COLOR(widget)	 (& (widget)->style->mid[GTK_WIDGET_STATE (widget)])
#define MARKED_COLOR(widget)		 (& (widget)->style->text[GTK_WIDGET_STATE (widget)])
#define BACKGROUND_COLOR(widget)	 (& (widget)->style->base[GTK_WIDGET_STATE (widget)])
#define HIGHLIGHT_BACK_COLOR(widget)	 (& (widget)->style->mid[GTK_WIDGET_STATE (widget)])

enum {
  MONTH_PREV,
  MONTH_CURRENT,
  MONTH_NEXT
};

enum {
  MONTH_CHANGED_SIGNAL,
  DAY_SELECTED_SIGNAL,
  DAY_SELECTED_DOUBLE_CLICK_SIGNAL,
  DAY_SELECTED_CLICK_SIGNAL,
  PREV_MONTH_SIGNAL,
  NEXT_MONTH_SIGNAL,
  PREV_YEAR_SIGNAL,
  NEXT_YEAR_SIGNAL,
  WEEK_SELECTED_CLICK_SIGNAL,
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_YEAR,
  PROP_MONTH,
  PROP_DAY,
  PROP_SHOW_DAY_NAMES,
  PROP_NO_MONTH_CHANGE,
  PROP_SHOW_WEEK_NUMBERS,
  PROP_SHOW_DETAILS,
  PROP_DETAIL_WIDTH_CHARS,
  PROP_DETAIL_HEIGHT_ROWS
};

static guint gtk_kimi_calendar_signals[LAST_SIGNAL] = { 0 };

typedef struct _GtkKimiCalendarPrivate GtkKimiCalendarPrivate;
struct _GtkKimiCalendarPrivate
{
  GdkWindow *day_name_win;
  GdkWindow *main_win;
  GdkWindow *week_win;

  guint day_name_h;
  guint main_h;

  guint	     max_month_width;
  guint	     max_year_width;
  
  gint click_x;
  gint click_y;

  guint day_width;
  guint week_width;

  guint min_day_width;
  guint max_day_char_width;
  guint max_day_char_ascent;
  guint max_day_char_descent;
  guint max_label_char_ascent;
  guint max_label_char_descent;
  guint max_week_char_width;
  
  /* flags */
  guint year_before : 1;

  guint in_drag : 1;
  guint drag_highlight : 1;

  gint click_child;

  gint week_start;

  gint drag_start_x;
  gint drag_start_y;

  /* Optional callback, used to display extra information for each day. */

  GtkKimiCalendarDetailFunc detail_func;

  gpointer              detail_func_user_data;
  GDestroyNotify        detail_func_destroy;

  /* Size requistion for details provided by the hook. */
  gint detail_height_rows;
  gint detail_width_chars;
  gint detail_overflow[6];

  /* background image */
  GdkPixbuf* bg_pixbuf;

  double rect_day_transparency;
};

#define GTK_KIMI_CALENDAR_GET_PRIVATE(widget)  ((GtkKimiCalendarPrivate*)(GTK_KIMI_CALENDAR (widget)->priv))

static void gtk_kimi_calendar_finalize     (GObject      *calendar);
static void gtk_kimi_calendar_destroy      (GtkObject    *calendar);
static void gtk_kimi_calendar_set_property (GObject      *object,
				       guint         prop_id,
				       const GValue *value,
				       GParamSpec   *pspec);
static void gtk_kimi_calendar_get_property (GObject      *object,
				       guint         prop_id,
				       GValue       *value,
				       GParamSpec   *pspec);

static void     gtk_kimi_calendar_realize        (GtkWidget        *widget);
static void     gtk_kimi_calendar_unrealize      (GtkWidget        *widget);
static void     gtk_kimi_calendar_size_request   (GtkWidget        *widget,
					     GtkRequisition   *requisition);
static void     gtk_kimi_calendar_size_allocate  (GtkWidget        *widget,
					     GtkAllocation    *allocation);
static gboolean gtk_kimi_calendar_expose         (GtkWidget        *widget,
					     GdkEventExpose   *event);
static gboolean gtk_kimi_calendar_button_press   (GtkWidget        *widget,
					     GdkEventButton   *event);
static gboolean gtk_kimi_calendar_button_release (GtkWidget        *widget,
					     GdkEventButton   *event);
static gboolean gtk_kimi_calendar_enter_notify   (GtkWidget        *widget,
					     GdkEventCrossing *event);
static gboolean gtk_kimi_calendar_leave_notify   (GtkWidget        *widget,
					     GdkEventCrossing *event);
static gboolean gtk_kimi_calendar_key_press      (GtkWidget        *widget,
					     GdkEventKey      *event);
static gboolean gtk_kimi_calendar_focus_out      (GtkWidget        *widget,
					     GdkEventFocus    *event);
static void     gtk_kimi_calendar_grab_notify    (GtkWidget        *widget,
					     gboolean          was_grabbed);
static void     gtk_kimi_calendar_state_changed  (GtkWidget        *widget,
					     GtkStateType      previous_state);
static void     gtk_kimi_calendar_style_set      (GtkWidget        *widget,
					     GtkStyle         *previous_style);

static void     gtk_kimi_calendar_drag_data_get      (GtkWidget        *widget,
						 GdkDragContext   *context,
						 GtkSelectionData *selection_data,
						 guint             info,
						 guint             time);
static void     gtk_kimi_calendar_drag_data_received (GtkWidget        *widget,
						 GdkDragContext   *context,
						 gint              x,
						 gint              y,
						 GtkSelectionData *selection_data,
						 guint             info,
						 guint             time);
static gboolean gtk_kimi_calendar_drag_motion        (GtkWidget        *widget,
						 GdkDragContext   *context,
						 gint              x,
						 gint              y,
						 guint             time);
static void     gtk_kimi_calendar_drag_leave         (GtkWidget        *widget,
						 GdkDragContext   *context,
						 guint             time);
static gboolean gtk_kimi_calendar_drag_drop          (GtkWidget        *widget,
						 GdkDragContext   *context,
						 gint              x,
						 gint              y,
						 guint             time);

static void calendar_invalidate_day     (GtkKimiCalendar *widget,
					 gint       row,
					 gint       col);
static void calendar_invalidate_day_num (GtkKimiCalendar *widget,
					 gint       day);

static void calendar_compute_days      (GtkKimiCalendar *calendar);
     
static char    *default_abbreviated_dayname[7];
static char    *default_monthname[12];

G_DEFINE_TYPE (GtkKimiCalendar, gtk_kimi_calendar, GTK_TYPE_WIDGET)

static void
gtk_kimi_calendar_class_init (GtkKimiCalendarClass *class)
{
  GObjectClass   *gobject_class;
  GtkObjectClass   *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass*)  class;
  object_class = (GtkObjectClass*)  class;
  widget_class = (GtkWidgetClass*) class;
  
  gobject_class->set_property = gtk_kimi_calendar_set_property;
  gobject_class->get_property = gtk_kimi_calendar_get_property;
  gobject_class->finalize = gtk_kimi_calendar_finalize;

  object_class->destroy = gtk_kimi_calendar_destroy;

  widget_class->realize = gtk_kimi_calendar_realize;
  widget_class->unrealize = gtk_kimi_calendar_unrealize;
  widget_class->expose_event = gtk_kimi_calendar_expose;
  widget_class->size_request = gtk_kimi_calendar_size_request;
  widget_class->size_allocate = gtk_kimi_calendar_size_allocate;
  widget_class->button_press_event = gtk_kimi_calendar_button_press;
  widget_class->button_release_event = gtk_kimi_calendar_button_release;
  widget_class->key_press_event = gtk_kimi_calendar_key_press;
  widget_class->style_set = gtk_kimi_calendar_style_set;
  widget_class->state_changed = gtk_kimi_calendar_state_changed;
  widget_class->grab_notify = gtk_kimi_calendar_grab_notify;
  widget_class->focus_out_event = gtk_kimi_calendar_focus_out;
 
  widget_class->drag_data_get = gtk_kimi_calendar_drag_data_get;
  widget_class->drag_motion = gtk_kimi_calendar_drag_motion;
  widget_class->drag_leave = gtk_kimi_calendar_drag_leave;
  widget_class->drag_drop = gtk_kimi_calendar_drag_drop;
  widget_class->drag_data_received = gtk_kimi_calendar_drag_data_received;
  
  /**
   * GtkKimiCalendar:year:
   *
   * The selected year. 
   * This property gets initially set to the current year.
   */  
  g_object_class_install_property (gobject_class,
                                   PROP_YEAR,
                                   g_param_spec_int ("year",
						     P_("Year"),
						     P_("The selected year"),
						     0, G_MAXINT >> 9, 0,
						     GTK_PARAM_READWRITE));

  /**
   * GtkKimiCalendar:month:
   *
   * The selected month (as a number between 0 and 11). 
   * This property gets initially set to the current month.
   */
  g_object_class_install_property (gobject_class,
                                   PROP_MONTH,
                                   g_param_spec_int ("month",
						     P_("Month"),
						     P_("The selected month (as a number between 0 and 11)"),
						     0, 11, 0,
						     GTK_PARAM_READWRITE));

  /**
   * GtkKimiCalendar:day:
   *
   * The selected day (as a number between 1 and 31, or 0 
   * to unselect the currently selected day).
   * This property gets initially set to the current day.
   */
  g_object_class_install_property (gobject_class,
                                   PROP_DAY,
                                   g_param_spec_int ("day",
						     P_("Day"),
						     P_("The selected day (as a number between 1 and 31, or 0 to unselect the currently selected day)"),
						     0, 31, 0,
						     GTK_PARAM_READWRITE));


/**
 * GtkKimiCalendar:show-day-names:
 *
 * Determines whether day names are displayed.
 *
 * Since: 2.4
 */
  g_object_class_install_property (gobject_class,
                                   PROP_SHOW_DAY_NAMES,
                                   g_param_spec_boolean ("show-day-names",
							 P_("Show Day Names"),
							 P_("If TRUE, day names are displayed"),
							 TRUE,
							 GTK_PARAM_READWRITE));
/**
 * GtkKimiCalendar:no-month-change:
 *
 * Determines whether the selected month can be changed.
 *
 * Since: 2.4
 */
  g_object_class_install_property (gobject_class,
                                   PROP_NO_MONTH_CHANGE,
                                   g_param_spec_boolean ("no-month-change",
							 P_("No Month Change"),
							 P_("If TRUE, the selected month cannot be changed"),
							 FALSE,
							 GTK_PARAM_READWRITE));

/**
 * GtkKimiCalendar:show-week-numbers:
 *
 * Determines whether week numbers are displayed.
 *
 * Since: 2.4
 */
  g_object_class_install_property (gobject_class,
                                   PROP_SHOW_WEEK_NUMBERS,
                                   g_param_spec_boolean ("show-week-numbers",
							 P_("Show Week Numbers"),
							 P_("If TRUE, week numbers are displayed"),
							 FALSE,
							 GTK_PARAM_READWRITE));

/**
 * GtkKimiCalendar:detail-width-chars:
 *
 * Width of a detail cell, in characters.
 * A value of 0 allows any width. See gtk_kimi_calendar_set_detail_func().
 *
 * Since: 2.14
 */
  g_object_class_install_property (gobject_class,
                                   PROP_DETAIL_WIDTH_CHARS,
                                   g_param_spec_int ("detail-width-chars",
						     P_("Details Width"),
						     P_("Details width in characters"),
						     0, 127, 0,
						     GTK_PARAM_READWRITE));

/**
 * GtkKimiCalendar:detail-height-rows:
 *
 * Height of a detail cell, in rows.
 * A value of 0 allows any width. See gtk_kimi_calendar_set_detail_func().
 *
 * Since: 2.14
 */
  g_object_class_install_property (gobject_class,
                                   PROP_DETAIL_HEIGHT_ROWS,
                                   g_param_spec_int ("detail-height-rows",
						     P_("Details Height"),
						     P_("Details height in rows"),
						     0, 127, 0,
						     GTK_PARAM_READWRITE));

/**
 * GtkKimiCalendar:show-details:
 *
 * Determines whether details are shown directly in the widget, or if they are
 * available only as tooltip. When this property is set days with details are
 * marked.
 *
 * Since: 2.14
 */
  g_object_class_install_property (gobject_class,
                                   PROP_SHOW_DETAILS,
                                   g_param_spec_boolean ("show-details",
							 P_("Show Details"),
							 P_("If TRUE, details are shown"),
							 TRUE,
							 GTK_PARAM_READWRITE));

  gtk_kimi_calendar_signals[MONTH_CHANGED_SIGNAL] =
    g_signal_new (I_("month-changed"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, month_changed),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[DAY_SELECTED_SIGNAL] =
    g_signal_new (I_("day-selected"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, day_selected),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[DAY_SELECTED_DOUBLE_CLICK_SIGNAL] =
    g_signal_new (I_("day-selected-double-click"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, day_selected_double_click),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[DAY_SELECTED_CLICK_SIGNAL] =
    g_signal_new (I_("day-selected-click"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, day_selected_click),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[WEEK_SELECTED_CLICK_SIGNAL] =
    g_signal_new (I_("week-selected-click"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, week_selected_click),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[PREV_MONTH_SIGNAL] =
    g_signal_new (I_("prev-month"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, prev_month),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[NEXT_MONTH_SIGNAL] =
    g_signal_new (I_("next-month"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, next_month),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[PREV_YEAR_SIGNAL] =
    g_signal_new (I_("prev-year"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, prev_year),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  gtk_kimi_calendar_signals[NEXT_YEAR_SIGNAL] =
    g_signal_new (I_("next-year"),
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkKimiCalendarClass, next_year),
		  NULL, NULL,
		  _gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  
  g_type_class_add_private (gobject_class, sizeof (GtkKimiCalendarPrivate));
}

static void
gtk_kimi_calendar_init (GtkKimiCalendar *calendar)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  time_t secs;
  struct tm *tm;
  gint i;
#ifdef G_OS_WIN32
  wchar_t wbuffer[100];
#else
  char buffer[255];
  time_t tmp_time;
#endif
  GtkKimiCalendarPrivate *priv;
  gchar *year_before;
#ifdef HAVE__NL_TIME_FIRST_WEEKDAY
  union { unsigned int word; char *string; } langinfo;
  gint week_1stday = 0;
  gint first_weekday = 1;
  guint week_origin;
#else
  gchar *week_start;
#endif

  priv = calendar->priv = G_TYPE_INSTANCE_GET_PRIVATE (calendar,
						       GTK_TYPE_KIMI_CALENDAR,
						       GtkKimiCalendarPrivate);

  GTK_WIDGET_SET_FLAGS (widget, GTK_CAN_FOCUS);
  
  if (!default_abbreviated_dayname[0])
    for (i=0; i<7; i++)
      {
#ifndef G_OS_WIN32
	tmp_time= (i+3)*86400;
	strftime ( buffer, sizeof (buffer), "%a", gmtime (&tmp_time));
	default_abbreviated_dayname[i] = g_locale_to_utf8 (buffer, -1, NULL, NULL, NULL);
#else
	if (!GetLocaleInfoW (GetThreadLocale (), LOCALE_SABBREVDAYNAME1 + (i+6)%7,
			     wbuffer, G_N_ELEMENTS (wbuffer)))
	  default_abbreviated_dayname[i] = g_strdup_printf ("(%d)", i);
	else
	  default_abbreviated_dayname[i] = g_utf16_to_utf8 (wbuffer, -1, NULL, NULL, NULL);
#endif
      }
  
  if (!default_monthname[0])
    for (i=0; i<12; i++)
      {
#ifndef G_OS_WIN32
	tmp_time=i*2764800;
	strftime ( buffer, sizeof (buffer), "%B", gmtime (&tmp_time));
	default_monthname[i] = g_locale_to_utf8 (buffer, -1, NULL, NULL, NULL);
#else
	if (!GetLocaleInfoW (GetThreadLocale (), LOCALE_SMONTHNAME1 + i,
			     wbuffer, G_N_ELEMENTS (wbuffer)))
	  default_monthname[i] = g_strdup_printf ("(%d)", i);
	else
	  default_monthname[i] = g_utf16_to_utf8 (wbuffer, -1, NULL, NULL, NULL);
#endif
      }
  
  /* Set defaults */
  secs = time (NULL);
  tm = localtime (&secs);
  calendar->month = tm->tm_mon;
  calendar->year  = 1900 + tm->tm_year;

  for (i=0;i<31;i++)
  calendar->marked_date[i] = FALSE;
  calendar->num_marked_dates = 0;
  calendar->selected_day = tm->tm_mday;
  
  calendar->display_flags = (GTK_KIMI_CALENDAR_SHOW_HEADING |
			     GTK_KIMI_CALENDAR_SHOW_DAY_NAMES |
			     GTK_KIMI_CALENDAR_SHOW_DETAILS);
  
  calendar->highlight_row = -1;
  calendar->highlight_col = -1;
  
  calendar->focus_row = -1;
  calendar->focus_col = -1;

  priv->max_year_width = 0;
  priv->max_month_width = 0;
  priv->max_day_char_width = 0;
  priv->max_week_char_width = 0;

  priv->max_day_char_ascent = 0;
  priv->max_day_char_descent = 0;
  priv->max_label_char_ascent = 0;
  priv->max_label_char_descent = 0;

  priv->click_child = -1;

  priv->in_drag = 0;
  priv->drag_highlight = 0;

  gtk_drag_dest_set (widget, 0, NULL, 0, GDK_ACTION_COPY);
  gtk_drag_dest_add_text_targets (widget);

  priv->year_before = 0;

  /* Translate to calendar:YM if you want years to be displayed
   * before months; otherwise translate to calendar:MY.
   * Do *not* translate it to anything else, if it
   * it isn't calendar:YM or calendar:MY it will not work.
   *
   * Note that the ordering described here is logical order, which is
   * further influenced by BIDI ordering. Thus, if you have a default
   * text direction of RTL and specify "calendar:YM", then the year
   * will appear to the right of the month.
   */
  year_before = _("calendar:MY");
  if (strcmp (year_before, "calendar:YM") == 0)
    priv->year_before = 1;
  else if (strcmp (year_before, "calendar:MY") != 0)
    g_warning ("Whoever translated calendar:MY did so wrongly.\n");

#ifdef G_OS_WIN32
  priv->week_start = 0;
  week_start = NULL;

  if (GetLocaleInfoW (GetThreadLocale (), LOCALE_IFIRSTDAYOFWEEK,
		      wbuffer, G_N_ELEMENTS (wbuffer)))
    week_start = g_utf16_to_utf8 (wbuffer, -1, NULL, NULL, NULL);
      
  if (week_start != NULL)
    {
      priv->week_start = (week_start[0] - '0' + 1) % 7;
      g_free(week_start);
    }
#else
#ifdef HAVE__NL_TIME_FIRST_WEEKDAY
  langinfo.string = nl_langinfo (_NL_TIME_FIRST_WEEKDAY);
  first_weekday = langinfo.string[0];
  langinfo.string = nl_langinfo (_NL_TIME_WEEK_1STDAY);
  week_origin = langinfo.word;
  if (week_origin == 19971130) /* Sunday */
    week_1stday = 0;
  else if (week_origin == 19971201) /* Monday */
    week_1stday = 1;
  else
    g_warning ("Unknown value of _NL_TIME_WEEK_1STDAY.\n");

  priv->week_start = (week_1stday + first_weekday - 1) % 7;
#else
  /* Translate to calendar:week_start:0 if you want Sunday to be the
   * first day of the week to calendar:week_start:1 if you want Monday
   * to be the first day of the week, and so on.
   */  
  week_start = _("calendar:week_start:0");

  if (strncmp (week_start, "calendar:week_start:", 20) == 0)
    priv->week_start = *(week_start + 20) - '0';
  else 
    priv->week_start = -1;
  
  if (priv->week_start < 0 || priv->week_start > 6)
    {
      g_warning ("Whoever translated calendar:week_start:0 did so wrongly.\n");
      priv->week_start = 0;
    }
#endif
#endif

  priv->rect_day_transparency = 1.0;
  
  calendar_compute_days (calendar);
}

/****************************************
 *          Utility Functions           *
 ****************************************/

static void
calendar_queue_refresh (GtkKimiCalendar *calendar)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);

  if (!(priv->detail_func) ||
      !(calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DETAILS) ||
       (priv->detail_width_chars && priv->detail_height_rows))
    gtk_widget_queue_draw (GTK_WIDGET (calendar));
  else
    gtk_widget_queue_resize (GTK_WIDGET (calendar));
}

static void
calendar_set_month_next (GtkKimiCalendar *calendar)
{
  gint month_len;

  if (calendar->display_flags & GTK_KIMI_CALENDAR_NO_MONTH_CHANGE)
    return;
  
  
  if (calendar->month == 11)
    {
      calendar->month = 0;
      calendar->year++;
    } 
  else 
    calendar->month++;
  
  calendar_compute_days (calendar);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[NEXT_MONTH_SIGNAL],
		 0);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[MONTH_CHANGED_SIGNAL],
		 0);
  
  month_len = month_length[leap (calendar->year)][calendar->month + 1];
  
  if (month_len < calendar->selected_day)
    {
      calendar->selected_day = 0;
      gtk_kimi_calendar_select_day (calendar, month_len);
    }
  else
    gtk_kimi_calendar_select_day (calendar, calendar->selected_day);

  calendar_queue_refresh (calendar);
}

static void
calendar_set_year_prev (GtkKimiCalendar *calendar)
{
  gint month_len;

  calendar->year--;
  calendar_compute_days (calendar);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[PREV_YEAR_SIGNAL],
		 0);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[MONTH_CHANGED_SIGNAL],
		 0);
  
  month_len = month_length[leap (calendar->year)][calendar->month + 1];
  
  if (month_len < calendar->selected_day)
    {
      calendar->selected_day = 0;
      gtk_kimi_calendar_select_day (calendar, month_len);
    }
  else
    gtk_kimi_calendar_select_day (calendar, calendar->selected_day);
  
  calendar_queue_refresh (calendar);
}

static void
calendar_set_year_next (GtkKimiCalendar *calendar)
{
  gint month_len;

  calendar->year++;
  calendar_compute_days (calendar);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[NEXT_YEAR_SIGNAL],
		 0);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[MONTH_CHANGED_SIGNAL],
		 0);
  
  month_len = month_length[leap (calendar->year)][calendar->month + 1];
  
  if (month_len < calendar->selected_day)
    {
      calendar->selected_day = 0;
      gtk_kimi_calendar_select_day (calendar, month_len);
    }
  else
    gtk_kimi_calendar_select_day (calendar, calendar->selected_day);
  
  calendar_queue_refresh (calendar);
}

static void
calendar_compute_days (GtkKimiCalendar *calendar)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (GTK_WIDGET (calendar));
  gint month;
  gint year;
  gint ndays_in_month;
  gint ndays_in_prev_month;
  gint first_day;
  gint row;
  gint col;
  gint day;

  year = calendar->year;
  month = calendar->month + 1;
  
  ndays_in_month = month_length[leap (year)][month];
  
  first_day = day_of_week (year, month, 1);
  first_day = (first_day + 7 - priv->week_start) % 7;
  
  /* Compute days of previous month */
  if (month > 1)
    ndays_in_prev_month = month_length[leap (year)][month-1];
  else
    ndays_in_prev_month = month_length[leap (year)][12];
  day = ndays_in_prev_month - first_day + 1;
  
  row = 0;
  if (first_day > 0)
    {
      for (col = 0; col < first_day; col++)
	{
	  calendar->day[row][col] = day;
	  calendar->day_month[row][col] = MONTH_PREV;
	  day++;
	}
    }
  
  /* Compute days of current month */
  col = first_day;
  for (day = 1; day <= ndays_in_month; day++)
    {
      calendar->day[row][col] = day;
      calendar->day_month[row][col] = MONTH_CURRENT;
      
      col++;
      if (col == 7)
	{
	  row++;
	  col = 0;
	}
    }
  
  /* Compute days of next month */
  day = 1;
  for (; row <= 5; row++)
    {
      for (; col <= 6; col++)
	{
	  calendar->day[row][col] = day;
	  calendar->day_month[row][col] = MONTH_NEXT;
	  day++;
	}
      col = 0;
    }
}

static void
calendar_select_and_focus_day (GtkKimiCalendar *calendar,
			       guint        day)
{
  gint old_focus_row = calendar->focus_row;
  gint old_focus_col = calendar->focus_col;
  gint row;
  gint col;
  
  for (row = 0; row < 6; row ++)
    for (col = 0; col < 7; col++)
      {
	if (calendar->day_month[row][col] == MONTH_CURRENT 
	    && calendar->day[row][col] == day)
	  {
	    calendar->focus_row = row;
	    calendar->focus_col = col;
	  }
      }

  if (old_focus_row != -1 && old_focus_col != -1)
    calendar_invalidate_day (calendar, old_focus_row, old_focus_col);
  
  gtk_kimi_calendar_select_day (calendar, day);
}

/****************************************
 *     Layout computation utilities     *
 ****************************************/

static gint
calendar_row_height (GtkKimiCalendar *calendar)
{
  return (GTK_KIMI_CALENDAR_GET_PRIVATE (calendar)->main_h - CALENDAR_MARGIN
	  - ((calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
	     ? CALENDAR_YSEP : CALENDAR_MARGIN)) / 6;
}

/* calendar_left_x_for_column: returns the x coordinate
 * for the left of the column */
static gint
calendar_left_x_for_column (GtkKimiCalendar *calendar,
			    gint	 column)
{
  gint width;
  gint x_left;
  
  if (gtk_widget_get_direction (GTK_WIDGET (calendar)) == GTK_TEXT_DIR_RTL)
    column = 6 - column;

  width = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar)->day_width;
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
    x_left = CALENDAR_XSEP + (width + DAY_XSEP) * column;
  else
    x_left = CALENDAR_MARGIN + (width + DAY_XSEP) * column;
  
  return x_left;
}

/* column_from_x: returns the column 0-6 that the
 * x pixel of the xwindow is in */
static gint
calendar_column_from_x (GtkKimiCalendar *calendar,
			gint	     event_x)
{
  gint c, column;
  gint x_left, x_right;
  
  column = -1;
  
  for (c = 0; c < 7; c++)
    {
      x_left = calendar_left_x_for_column (calendar, c) + STEP;
      x_right = x_left + GTK_KIMI_CALENDAR_GET_PRIVATE (calendar)->day_width;
      
      if (event_x >= x_left && event_x < x_right)
	{
	  column = c;
	  break;
	}
    }
  
  return column;
}

/* calendar_top_y_for_row: returns the y coordinate
 * for the top of the row */
static gint
calendar_top_y_for_row (GtkKimiCalendar *calendar,
			gint	     row)
{
  
  return (GTK_KIMI_CALENDAR_GET_PRIVATE (calendar)->main_h 
	  - (CALENDAR_MARGIN + (6 - row)
	     * calendar_row_height (calendar)));
}

/* row_from_y: returns the row 0-5 that the
 * y pixel of the xwindow is in */
static gint
calendar_row_from_y (GtkKimiCalendar *calendar,
		     gint	  event_y)
{
  gint r, row;
  gint height;
  gint y_top, y_bottom;
  
  height = calendar_row_height (calendar);
  row = -1;
  
  for (r = 0; r < 6; r++)
    {
      y_top = calendar_top_y_for_row (calendar, r) + STEP;
      y_bottom = y_top + height;
      
      if (event_y >= y_top && event_y < y_bottom)
	{
	  row = r;
	  break;
	}
    }
  
  return row;
}

static void
calendar_day_rectangle (GtkKimiCalendar  *calendar,
			gint          row,
			gint          col,
			GdkRectangle *rect)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);

  rect->x = calendar_left_x_for_column (calendar, col);
  rect->y = calendar_top_y_for_row (calendar, row);
  rect->height = calendar_row_height (calendar);
  rect->width = priv->day_width;
}

static void
calendar_set_month_prev (GtkKimiCalendar *calendar)
{
  gint month_len;
  
  if (calendar->display_flags & GTK_KIMI_CALENDAR_NO_MONTH_CHANGE)
    return;
  
  if (calendar->month == 0)
    {
      calendar->month = 11;
      calendar->year--;
    } 
  else 
    calendar->month--;
  
  month_len = month_length[leap (calendar->year)][calendar->month + 1];
  
  calendar_compute_days (calendar);
  
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[PREV_MONTH_SIGNAL],
		 0);
  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[MONTH_CHANGED_SIGNAL],
		 0);
  
  if (month_len < calendar->selected_day)
    {
      calendar->selected_day = 0;
      gtk_kimi_calendar_select_day (calendar, month_len);
    }
  else
    {
      if (calendar->selected_day < 0)
	calendar->selected_day = calendar->selected_day + 1 + month_length[leap (calendar->year)][calendar->month + 1];
      gtk_kimi_calendar_select_day (calendar, calendar->selected_day);
    }

  calendar_queue_refresh (calendar);
}

/****************************************
 *           Basic object methods       *
 ****************************************/

static void
gtk_kimi_calendar_finalize (GObject *object)
{
  G_OBJECT_CLASS (gtk_kimi_calendar_parent_class)->finalize (object);
}

static void
gtk_kimi_calendar_destroy (GtkObject *object)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (object);
  
  /* Call the destroy function for the extra display callback: */
  if (priv->detail_func_destroy && priv->detail_func_user_data)
    {
      priv->detail_func_destroy (priv->detail_func_user_data);
      priv->detail_func_user_data = NULL;
      priv->detail_func_destroy = NULL;
    }

  GTK_OBJECT_CLASS (gtk_kimi_calendar_parent_class)->destroy (object);
}


static void
calendar_set_display_option (GtkKimiCalendar              *calendar,
			     GtkKimiCalendarDisplayOptions flag,
			     gboolean                  setting)
{
  GtkKimiCalendarDisplayOptions flags;
  if (setting) 
    flags = calendar->display_flags | flag;
  else
    flags = calendar->display_flags & ~flag; 
  gtk_kimi_calendar_set_display_options (calendar, flags);
}

static gboolean
calendar_get_display_option (GtkKimiCalendar              *calendar,
			     GtkKimiCalendarDisplayOptions flag)
{
  return (calendar->display_flags & flag) != 0;
}

static void 
gtk_kimi_calendar_set_property (GObject      *object,
			   guint         prop_id,
			   const GValue *value,
			   GParamSpec   *pspec)
{
  GtkKimiCalendar *calendar;

  calendar = GTK_KIMI_CALENDAR (object);

  switch (prop_id) 
    {
    case PROP_YEAR:
      gtk_kimi_calendar_select_month (calendar,
				 calendar->month,
				 g_value_get_int (value));
      break;
    case PROP_MONTH:
      gtk_kimi_calendar_select_month (calendar,
				 g_value_get_int (value),
				 calendar->year);
      break;
    case PROP_DAY:
      gtk_kimi_calendar_select_day (calendar,
			       g_value_get_int (value));
      break;
    case PROP_SHOW_DAY_NAMES:
      calendar_set_display_option (calendar,
				   GTK_KIMI_CALENDAR_SHOW_DAY_NAMES,
				   g_value_get_boolean (value));
      break;
    case PROP_NO_MONTH_CHANGE:
      calendar_set_display_option (calendar,
				   GTK_KIMI_CALENDAR_NO_MONTH_CHANGE,
				   g_value_get_boolean (value));
      break;
    case PROP_SHOW_WEEK_NUMBERS:
      calendar_set_display_option (calendar,
				   GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS,
				   g_value_get_boolean (value));
      break;
    case PROP_SHOW_DETAILS:
      calendar_set_display_option (calendar,
				   GTK_KIMI_CALENDAR_SHOW_DETAILS,
				   g_value_get_boolean (value));
      break;
    case PROP_DETAIL_WIDTH_CHARS:
      gtk_kimi_calendar_set_detail_width_chars (calendar,
                                           g_value_get_int (value));
      break;
    case PROP_DETAIL_HEIGHT_ROWS:
      gtk_kimi_calendar_set_detail_height_rows (calendar,
                                           g_value_get_int (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void 
gtk_kimi_calendar_get_property (GObject      *object,
			   guint         prop_id,
			   GValue       *value,
			   GParamSpec   *pspec)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (object);
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (object);

  switch (prop_id) 
    {
    case PROP_YEAR:
      g_value_set_int (value, calendar->year);
      break;
    case PROP_MONTH:
      g_value_set_int (value, calendar->month);
      break;
    case PROP_DAY:
      g_value_set_int (value, calendar->selected_day);
      break;
    case PROP_SHOW_DAY_NAMES:
      g_value_set_boolean (value, calendar_get_display_option (calendar,
							       GTK_KIMI_CALENDAR_SHOW_DAY_NAMES));
      break;
    case PROP_NO_MONTH_CHANGE:
      g_value_set_boolean (value, calendar_get_display_option (calendar,
							       GTK_KIMI_CALENDAR_NO_MONTH_CHANGE));
      break;
    case PROP_SHOW_WEEK_NUMBERS:
      g_value_set_boolean (value, calendar_get_display_option (calendar,
							       GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS));
      break;
    case PROP_SHOW_DETAILS:
      g_value_set_boolean (value, calendar_get_display_option (calendar,
							       GTK_KIMI_CALENDAR_SHOW_DETAILS));
      break;
    case PROP_DETAIL_WIDTH_CHARS:
      g_value_set_int (value, priv->detail_width_chars);
      break;
    case PROP_DETAIL_HEIGHT_ROWS:
      g_value_set_int (value, priv->detail_height_rows);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

/****************************************
 *             Realization              *
 ****************************************/

static void
calendar_realize_day_names (GtkKimiCalendar *calendar)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  GdkWindowAttr attributes;
  gint attributes_mask;
  
  /* Day names	window --------------------------------- */
  if ( calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
    {
      attributes.wclass = GDK_INPUT_OUTPUT;
      attributes.window_type = GDK_WINDOW_CHILD;
      attributes.visual = gtk_widget_get_visual (widget);
      attributes.colormap = gtk_widget_get_colormap (widget);
      attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;
      attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
      attributes.x = (widget->style->xthickness + INNER_BORDER);
      attributes.y = (widget->style->ythickness 
					   + INNER_BORDER);
      attributes.width = (widget->allocation.width 
			  - (widget->style->xthickness + INNER_BORDER) 
			  * 2);
      attributes.height = priv->day_name_h;
      priv->day_name_win = gdk_window_new (widget->window,
					   &attributes, 
					   attributes_mask);
      gdk_window_set_background (priv->day_name_win, 
				 BACKGROUND_COLOR ( GTK_WIDGET ( calendar)));
      gdk_window_show (priv->day_name_win);
      gdk_window_set_user_data (priv->day_name_win, widget);
    }
  else
    {
      priv->day_name_win = NULL;
    }
}

static void
calendar_realize_week_numbers (GtkKimiCalendar *calendar)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  GdkWindowAttr attributes;
  gint attributes_mask;
  
  /* Week number window -------------------------------- */
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
    {
      attributes.wclass = GDK_INPUT_OUTPUT;
      attributes.window_type = GDK_WINDOW_CHILD;
      attributes.visual = gtk_widget_get_visual (widget);
      attributes.colormap = gtk_widget_get_colormap (widget);
      attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;
      
      attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
      if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR) 
	attributes.x = widget->style->xthickness + INNER_BORDER;
      else 
	attributes.x = widget->allocation.width - priv->week_width - (widget->style->xthickness + INNER_BORDER);
      attributes.y = (priv->day_name_h 
		      + (widget->style->ythickness + INNER_BORDER));
      attributes.width = priv->week_width;
      attributes.height = priv->main_h;
      priv->week_win = gdk_window_new (widget->window,
				       &attributes, attributes_mask);
      gdk_window_set_background (priv->week_win,  
				 BACKGROUND_COLOR (GTK_WIDGET (calendar)));
      gdk_window_show (priv->week_win);
      gdk_window_set_user_data (priv->week_win, widget);
    } 
  else
    {
      priv->week_win = NULL;
    }
}

static void
gtk_kimi_calendar_realize (GtkWidget *widget)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  GdkWindowAttr attributes;
  gint attributes_mask;

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask =  (gtk_widget_get_events (widget) 
			    | GDK_EXPOSURE_MASK |GDK_KEY_PRESS_MASK | GDK_SCROLL_MASK);
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window = gdk_window_new (widget->parent->window,
				   &attributes, attributes_mask);
  
  widget->style = gtk_style_attach (widget->style, widget->window);
  
  /* Day names	window --------------------------------- */
  calendar_realize_day_names (calendar);
  /* Week number window -------------------------------- */
  calendar_realize_week_numbers (calendar);
  /* Main Window --------------------------------------	 */
  attributes.event_mask =  (gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK
			    | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
			    | GDK_POINTER_MOTION_MASK | GDK_LEAVE_NOTIFY_MASK);
  
  if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR) 
    attributes.x = priv->week_width + (widget->style->ythickness + INNER_BORDER);
  else
    attributes.x = widget->style->ythickness + INNER_BORDER;

  attributes.y = (priv->day_name_h 
		  + (widget->style->ythickness + INNER_BORDER));
  attributes.width = (widget->allocation.width - attributes.x 
		      - (widget->style->xthickness + INNER_BORDER));
  if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL)
    attributes.width -= priv->week_width;

  attributes.height = priv->main_h;
  priv->main_win = gdk_window_new (widget->window,
				   &attributes, attributes_mask);
  gdk_window_set_background (priv->main_win, 
			     BACKGROUND_COLOR ( GTK_WIDGET ( calendar)));

  gdk_window_show (priv->main_win);
  gdk_window_set_user_data (priv->main_win, widget);
  gdk_window_set_background (widget->window, BACKGROUND_COLOR (widget));
  gdk_window_show (widget->window);
  gdk_window_set_user_data (widget->window, widget);
}

static void
gtk_kimi_calendar_unrealize (GtkWidget *widget)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  gint i;
  
  if (priv->week_win)
    {
      gdk_window_set_user_data (priv->week_win, NULL);
      gdk_window_destroy (priv->week_win);
      priv->week_win = NULL;      
    }
  
  if (priv->main_win)
    {
      gdk_window_set_user_data (priv->main_win, NULL);
      gdk_window_destroy (priv->main_win);
      priv->main_win = NULL;      
    }
  if (priv->day_name_win)
    {
      gdk_window_set_user_data (priv->day_name_win, NULL);
      gdk_window_destroy (priv->day_name_win);
      priv->day_name_win = NULL;      
    }

  GTK_WIDGET_CLASS (gtk_kimi_calendar_parent_class)->unrealize (widget);
}

static gchar*
gtk_kimi_calendar_get_detail (GtkKimiCalendar *calendar,
                         gint         row,
                         gint         column)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  gint year, month;

  year = calendar->year;
  month = calendar->month + calendar->day_month[row][column] - MONTH_CURRENT;

  if (month < 0)
    {
      month += 12;
      year -= 1;
    }
  else if (month > 11)
    {
      month -= 12;
      year += 1;
    }

  return priv->detail_func (calendar,
                            year, month,
                            calendar->day[row][column],
                            priv->detail_func_user_data);
}


/****************************************
 *       Size Request and Allocate      *
 ****************************************/

static void
gtk_kimi_calendar_size_request (GtkWidget	  *widget,
			   GtkRequisition *requisition)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  PangoLayout *layout;
  PangoRectangle logical_rect;

  gint height;
  gint i, r, c;
  gint calendar_margin = CALENDAR_MARGIN;
  gint main_width;
  gint focus_width;
  gint focus_padding;
  gint max_detail_height;

  gtk_widget_style_get (GTK_WIDGET (widget),
			"focus-line-width", &focus_width,
			"focus-padding", &focus_padding,
			NULL);

  layout = gtk_widget_create_pango_layout (widget, NULL);
  
  /*
   * Calculate the requisition	width for the widget.
   */
  
  /* Header width */
  
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_HEADING)
    {
      priv->max_month_width = 0;
      for (i = 0; i < 12; i++)
	{
	  pango_layout_set_text (layout, default_monthname[i], -1);
	  pango_layout_get_pixel_extents (layout, NULL, &logical_rect);
	  priv->max_month_width = MAX (priv->max_month_width,
					       logical_rect.width + 8);
	}

      priv->max_year_width = 0;
      /* Translators:  This is a text measurement template.
       * Translate it to the widest year text
       *
       * If you don't understand this, leave it as "2000"
       */
      pango_layout_set_text (layout, C_("year measurement template", "2000"), -1);	  
      pango_layout_get_pixel_extents (layout, NULL, &logical_rect);
      priv->max_year_width = MAX (priv->max_year_width,
				  logical_rect.width + 8);
    } 
  else 
    {
      priv->max_month_width = 0;
      priv->max_year_width = 0;
    }
  
  /* Mainwindow labels width */
  
  priv->max_day_char_width = 0;
  priv->max_day_char_ascent = 0;
  priv->max_day_char_descent = 0;
  priv->min_day_width = 0;

  for (i = 0; i < 9; i++)
    {
      gchar buffer[32];
      g_snprintf (buffer, sizeof (buffer), C_("calendar:day:digits", "%d"), i * 11);
      pango_layout_set_text (layout, buffer, -1);	  
      pango_layout_get_pixel_extents (layout, NULL, &logical_rect);
      priv->min_day_width = MAX (priv->min_day_width,
					 logical_rect.width);

      priv->max_day_char_ascent = MAX (priv->max_day_char_ascent,
					       PANGO_ASCENT (logical_rect));
      priv->max_day_char_descent = MAX (priv->max_day_char_descent, 
						PANGO_DESCENT (logical_rect));
    }
  
  priv->max_label_char_ascent = 0;
  priv->max_label_char_descent = 0;
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
    for (i = 0; i < 7; i++)
      {
	pango_layout_set_text (layout, default_abbreviated_dayname[i], -1);
	pango_layout_line_get_pixel_extents (pango_layout_get_lines_readonly (layout)->data, NULL, &logical_rect);

	priv->min_day_width = MAX (priv->min_day_width, logical_rect.width);
	priv->max_label_char_ascent = MAX (priv->max_label_char_ascent,
						   PANGO_ASCENT (logical_rect));
	priv->max_label_char_descent = MAX (priv->max_label_char_descent, 
						    PANGO_DESCENT (logical_rect));
      }
  
  priv->max_week_char_width = 0;
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
    for (i = 0; i < 9; i++)
      {
	gchar buffer[32];
	g_snprintf (buffer, sizeof (buffer), C_("calendar:week:digits", "%d"), i * 11);
	pango_layout_set_text (layout, buffer, -1);	  
	pango_layout_get_pixel_extents (layout, NULL, &logical_rect);
	priv->max_week_char_width = MAX (priv->max_week_char_width,
					   logical_rect.width / 2);
      }
  
  /* Calculate detail extents. Do this as late as possible since
   * pango_layout_set_markup is called which alters font settings. */
  max_detail_height = 0;

  if (priv->detail_func && (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DETAILS))
    {
      gchar *markup, *tail;

      if (priv->detail_width_chars || priv->detail_height_rows)
        {
          gint rows = MAX (1, priv->detail_height_rows) - 1;
          gsize len = priv->detail_width_chars + rows + 16;

          markup = tail = g_alloca (len);

          memcpy (tail,     "<small>", 7);
          tail += 7;

          memset (tail, 'm', priv->detail_width_chars);
          tail += priv->detail_width_chars;

          memset (tail, '\n', rows);
          tail += rows;

          memcpy (tail,     "</small>", 9);
          tail += 9;

          g_assert (len == (tail - markup));

          pango_layout_set_markup (layout, markup, -1);
          pango_layout_get_pixel_extents (layout, NULL, &logical_rect);

          if (priv->detail_width_chars)
            priv->min_day_width = MAX (priv->min_day_width, logical_rect.width);
          if (priv->detail_height_rows)
            max_detail_height = MAX (max_detail_height, logical_rect.height);
        }

      if (!priv->detail_width_chars || !priv->detail_height_rows)
        for (r = 0; r < 6; r++)
          for (c = 0; c < 7; c++)
            {
              gchar *detail = gtk_kimi_calendar_get_detail (calendar, r, c);

              if (detail)
                {
                  markup = g_strconcat ("<small>", detail, "</small>", NULL);
                  pango_layout_set_markup (layout, markup, -1);

                  if (priv->detail_width_chars)
                    {
                      pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);
                      pango_layout_set_width (layout, PANGO_SCALE * priv->min_day_width);
                    }

                  pango_layout_get_pixel_extents (layout, NULL, &logical_rect);

                  if (!priv->detail_width_chars)
                    priv->min_day_width = MAX (priv->min_day_width, logical_rect.width);
                  if (!priv->detail_height_rows)
                    max_detail_height = MAX (max_detail_height, logical_rect.height);

                  g_free (markup);
                  g_free (detail);
                }
            }
    }

  /* We add one to max_day_char_width to be able to make the marked day "bold" */
  priv->max_day_char_width = priv->min_day_width / 2 + 1;

  main_width = (7 * (priv->min_day_width + (focus_padding + focus_width) * 2) + (DAY_XSEP * 6) + CALENDAR_MARGIN * 2
		+ (priv->max_week_char_width
		   ? priv->max_week_char_width * 2 + (focus_padding + focus_width) * 2 + CALENDAR_XSEP * 2
		   : 0));
  
  
  requisition->width = main_width + INNER_BORDER * 2 + widget->style->xthickness * 2;
  
  /*
   * Calculate the requisition height for the widget.
   */
  
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
    {
      priv->day_name_h = (priv->max_label_char_ascent
				  + priv->max_label_char_descent
				  + 2 * (focus_padding + focus_width) + calendar_margin);
      calendar_margin = CALENDAR_YSEP;
    } 
  else
    {
      priv->day_name_h = 0;
    }

  priv->main_h = (CALENDAR_MARGIN + calendar_margin
			  + 6 * (priv->max_day_char_ascent
				 + priv->max_day_char_descent 
                                 + max_detail_height
				 + 2 * (focus_padding + focus_width))
			  + DAY_YSEP * 5) + PLUS_H;

  height = (priv->day_name_h 
	    + priv->main_h);
  
  requisition->height = height + (widget->style->ythickness + INNER_BORDER) * 2;

  g_object_unref (layout);
}

static void
gtk_kimi_calendar_size_allocate (GtkWidget	  *widget,
			    GtkAllocation *allocation)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  gint xthickness = widget->style->xthickness;
  gint ythickness = widget->style->xthickness;
  guint i;
  
  widget->allocation = *allocation;
    
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
    {
      priv->day_width = (priv->min_day_width
			 * ((allocation->width - (xthickness + INNER_BORDER) * 2
			     - (CALENDAR_MARGIN * 2) -  (DAY_XSEP * 6) - CALENDAR_XSEP * 2))
			 / (7 * priv->min_day_width + priv->max_week_char_width * 2));
      priv->week_width = ((allocation->width - (xthickness + INNER_BORDER) * 2
			   - (CALENDAR_MARGIN * 2) - (DAY_XSEP * 6) - CALENDAR_XSEP * 2 )
			  - priv->day_width * 7 + CALENDAR_MARGIN + CALENDAR_XSEP);
    } 
  else 
    {
      priv->day_width = (allocation->width
			 - (xthickness + INNER_BORDER) * 2
			 - (CALENDAR_MARGIN * 2)
			 - (DAY_XSEP * 6))/7;
      priv->week_width = 0;
    }
  
  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);
          
      if (priv->day_name_win)
	gdk_window_move_resize (priv->day_name_win,
				xthickness + INNER_BORDER,
				(widget->style->ythickness + INNER_BORDER),
				allocation->width - (xthickness + INNER_BORDER) * 2,
				priv->day_name_h);
      if (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR) 
	{
	  if (priv->week_win)
	    gdk_window_move_resize (priv->week_win,
				    (xthickness + INNER_BORDER),
				    priv->day_name_h
				    + (widget->style->ythickness + INNER_BORDER),
				    priv->week_width,
				    priv->main_h);
	  gdk_window_move_resize (priv->main_win,
				  priv->week_width + (xthickness + INNER_BORDER),
				  priv->day_name_h
				  + (widget->style->ythickness + INNER_BORDER),
				  allocation->width 
				  - priv->week_width 
				  - (xthickness + INNER_BORDER) * 2,
				  priv->main_h);
	}
      else 
	{
	  gdk_window_move_resize (priv->main_win,
				  (xthickness + INNER_BORDER),
				  priv->day_name_h
				  + (widget->style->ythickness + INNER_BORDER),
				  allocation->width 
				  - priv->week_width 
				  - (xthickness + INNER_BORDER) * 2,
				  priv->main_h);
	  if (priv->week_win)
	    gdk_window_move_resize (priv->week_win,
				    allocation->width 
				    - priv->week_width 
				    - (xthickness + INNER_BORDER),
				    priv->day_name_h
				    + (widget->style->ythickness + INNER_BORDER),
				    priv->week_width,
				    priv->main_h);
	}
    }
}

/****************************************
 *              Repainting              *
 ****************************************/

static void
calendar_paint_day_names (GtkKimiCalendar *calendar)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  cairo_t *cr;
  char buffer[255];
  int day,i;
  int day_width, cal_width;
  int day_wid_sep;
  PangoLayout *layout;
  PangoRectangle logical_rect;
  gint focus_padding;
  gint focus_width;
  
  cr = gdk_cairo_create (priv->day_name_win);
  
  gtk_widget_style_get (GTK_WIDGET (widget),
			"focus-line-width", &focus_width,
			"focus-padding", &focus_padding,
			NULL);
  
  day_width = priv->day_width;
  cal_width = widget->allocation.width;
  day_wid_sep = day_width + DAY_XSEP;
  
  /*
   * Draw rectangles as inverted background for the labels.
   */

  gdk_cairo_set_source_color (cr, HEADER_BG_COLOR (widget));

  cairo_rectangle (cr,
		   CALENDAR_MARGIN, CALENDAR_MARGIN,
		   cal_width-CALENDAR_MARGIN * 2,
		   priv->day_name_h - CALENDAR_MARGIN);
  cairo_fill (cr);
  
  if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
    {
      cairo_rectangle (cr, 
		       CALENDAR_MARGIN,
		       priv->day_name_h - CALENDAR_YSEP,
		       priv->week_width - CALENDAR_YSEP - CALENDAR_MARGIN,
		       CALENDAR_YSEP);
      cairo_fill (cr);
    }
  
  /*
   * Write the labels
   */

  layout = gtk_widget_create_pango_layout (widget, NULL);

  gdk_cairo_set_source_color (cr, HEADER_FG_COLOR (widget));

  for (i = 0; i < 7; i++)
    {
      if (gtk_widget_get_direction (GTK_WIDGET (calendar)) == GTK_TEXT_DIR_RTL)
	day = 6 - i;
      else
	day = i;
      day = (day + priv->week_start) % 7;
      g_snprintf (buffer, sizeof (buffer), "%s", default_abbreviated_dayname[day]);

      pango_layout_set_text (layout, buffer, -1);
      pango_layout_get_pixel_extents (layout, NULL, &logical_rect);

      cairo_move_to (cr, 
		     (CALENDAR_MARGIN +
		      + (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_LTR ?
			 (priv->week_width + (priv->week_width ? CALENDAR_XSEP : 0))
			 : 0)
		      + day_wid_sep * i
		      + (day_width - logical_rect.width)/2),
		     CALENDAR_MARGIN + focus_width + focus_padding + logical_rect.y);
      pango_cairo_show_layout (cr, layout);
    }
  
  g_object_unref (layout);
  cairo_destroy (cr);
}

static void
calendar_paint_week_numbers (GtkKimiCalendar *calendar)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  cairo_t *cr;

  guint week = 0, year;
  gint row, x_loc, y_loc;
  gint day_height;
  char buffer[32];
  PangoLayout *layout;
  PangoRectangle logical_rect;
  gint focus_padding;
  gint focus_width;
  
  cr = gdk_cairo_create (priv->week_win);
  
  gtk_widget_style_get (GTK_WIDGET (widget),
			"focus-line-width", &focus_width,
			"focus-padding", &focus_padding,
			NULL);
  
  /*
   * Write the labels
   */
  
  layout = gtk_widget_create_pango_layout (widget, NULL);
  
  day_height = calendar_row_height (calendar);
  for (row = 0; row < 6; row++)
    {
      gboolean result;
      
      year = calendar->year;
      if (calendar->day[row][6] < 15 && row > 3 && calendar->month == 11)
	year++;

      result = week_of_year (&week, &year,		
			     ((calendar->day[row][6] < 15 && row > 3 ? 1 : 0)
			      + calendar->month) % 12 + 1, calendar->day[row][6]);
      g_return_if_fail (result);

      /* Translators: this defines whether the week numbers should use
       * localized digits or the ones used in English (0123...).
       *
       * Translate to "%Id" if you want to use localized digits, or
       * translate to "%d" otherwise.
       *
       * Note that translating this doesn't guarantee that you get localized
       * digits. That needs support from your system and locale definition
       * too.
       */
      g_snprintf (buffer, sizeof (buffer), C_("calendar:week:digits", "%d"), week);
      pango_layout_set_text (layout, buffer, -1);
      pango_layout_get_pixel_extents (layout, NULL, &logical_rect);

  /*
   * Draw a rectangle as inverted background for the labels.
   */

/*
cairo_set_source_rgb (cr, 0.15, 0.15, 0.15);
    if (priv->day_name_win)
    cairo_rectangle (cr, 
		     CALENDAR_MARGIN,
		     calendar_top_y_for_row(calendar, row) + STEP,
		     priv->week_width - CALENDAR_MARGIN,
		     day_height - STEP );
  else
    cairo_rectangle (cr,
		     CALENDAR_MARGIN,
		     CALENDAR_MARGIN + calendar_top_y_for_row(calendar, row) + STEP,
		     priv->week_width - CALENDAR_MARGIN,
		     (day_height - STEP) - 2 * CALENDAR_MARGIN);

  cairo_fill (cr);
*/
int radius = priv->week_width/2;

int rect_width  = (priv->week_width - CALENDAR_MARGIN);
int rect_height = (day_height - STEP);

int x0 = CALENDAR_MARGIN; 
int y0 = calendar_top_y_for_row(calendar, row) + STEP;
int x1 = x0 + rect_width;
int y1 = y0 + rect_height;

if (rect_width/2<radius) {
    if (rect_height/2<radius) {
        cairo_move_to  (cr, x0, (y0 + y1)/2);
        cairo_curve_to (cr, x0 ,y0, x0, y0, (x0 + x1)/2, y0);
        cairo_curve_to (cr, x1, y0, x1, y0, x1, (y0 + y1)/2);
        cairo_curve_to (cr, x1, y1, x1, y1, (x1 + x0)/2, y1);
        cairo_curve_to (cr, x0, y1, x0, y1, x0, (y0 + y1)/2);
    } else {
        cairo_move_to  (cr, x0, y0 + radius);
        cairo_curve_to (cr, x0 ,y0, x0, y0, (x0 + x1)/2, y0);
        cairo_curve_to (cr, x1, y0, x1, y0, x1, y0 + radius);
        cairo_curve_to (cr, x1, y1, x1, y1, (x1 + x0)/2, y1);
        cairo_curve_to (cr, x0, y1, x0, y1, x0, y1- radius);
    }
} else {
    if (rect_height/2<radius) {
        cairo_move_to  (cr, x0, (y0 + y1)/2);
        cairo_curve_to (cr, x0 , y0, x0 , y0, x0 + radius, y0);
        cairo_curve_to (cr, x1, y0, x1, y0, x1, (y0 + y1)/2);
        cairo_curve_to (cr, x1, y1, x1, y1, x1 - radius, y1); 
        cairo_curve_to (cr, x0, y1, x0, y1, x0, (y0 + y1)/2);
    } else {
        cairo_move_to  (cr, x0, y0 + radius);
        cairo_curve_to (cr, x0 , y0, x0 , y0, x0 + radius, y0);
        cairo_curve_to (cr, x1, y0, x1, y0, x1, y0 + radius);
        cairo_curve_to (cr, x1, y1, x1, y1, x1 - radius, y1);
        cairo_curve_to (cr, x0, y1, x0, y1, x0, y1- radius);
    }
}

cairo_close_path (cr);

cairo_pattern_t *pat;
pat = cairo_pattern_create_linear (0.0, 0.0,  (double)rect_width, 0.0);
cairo_pattern_add_color_stop_rgba (pat, 0, 0, 0, 0, 1);
cairo_pattern_add_color_stop_rgba (pat, 1, 0.2, 0.2, 0.2, 1);
cairo_set_source (cr, pat);
cairo_fill (cr);
cairo_pattern_destroy (pat);



      gdk_cairo_set_source_color (cr, HEADER_FG_COLOR (widget));

      y_loc = calendar_top_y_for_row (calendar, row) + (day_height - logical_rect.height) / 2;
      x_loc = (priv->week_width
	       - logical_rect.width
	       - CALENDAR_XSEP - focus_padding - focus_width) - 10;

      cairo_move_to (cr, x_loc, y_loc);
      pango_cairo_show_layout (cr, layout);
    }
  
  g_object_unref (layout);
  cairo_destroy (cr);
}

static void
calendar_invalidate_day_num (GtkKimiCalendar *calendar,
			     gint         day)
{
  gint r, c, row, col;
  
  row = -1;
  col = -1;
  for (r = 0; r < 6; r++)
    for (c = 0; c < 7; c++)
      if (calendar->day_month[r][c] == MONTH_CURRENT &&
	  calendar->day[r][c] == day)
	{
	  row = r;
	  col = c;
	}
  
  g_return_if_fail (row != -1);
  g_return_if_fail (col != -1);
  
  calendar_invalidate_day (calendar, row, col);
}

static void
calendar_invalidate_day (GtkKimiCalendar *calendar,
			 gint         row,
			 gint         col)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);

  if (priv->main_win)
    {
      GdkRectangle day_rect;
      
      calendar_day_rectangle (calendar, row, col, &day_rect);
      gdk_window_invalidate_rect (priv->main_win, &day_rect, FALSE);
    }
}

static gboolean
is_color_attribute (PangoAttribute *attribute,
                    gpointer        data)
{
  return (attribute->klass->type == PANGO_ATTR_FOREGROUND ||
          attribute->klass->type == PANGO_ATTR_BACKGROUND);
}

static void
calendar_paint_day (GtkKimiCalendar *calendar,
		    gint	     row,
		    gint	     col)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  cairo_t *cr;
  GdkColor *text_color;
  gchar *detail;
  gchar buffer[32];
  gint day;
  gint x_loc, y_loc;
  GdkRectangle day_rect;

  PangoLayout *layout;
  PangoRectangle logical_rect;
  gboolean overflow = FALSE;
  gboolean show_details;

  g_return_if_fail (row < 6);
  g_return_if_fail (col < 7);

  cr = gdk_cairo_create (priv->main_win);

  day = calendar->day[row][col];
  show_details = (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_DETAILS);

  calendar_day_rectangle (calendar, row, col, &day_rect);

  GdkColor back_text_color;
  back_text_color.red   = 40000;
  back_text_color.green = 40000;
  back_text_color.blue  = 40000;
  
  if (calendar->day_month[row][col] == MONTH_PREV)
    {
      //text_color = PREV_MONTH_COLOR (widget);
      text_color = &back_text_color;
    } 
  else if (calendar->day_month[row][col] == MONTH_NEXT)
    {
      //text_color =  NEXT_MONTH_COLOR (widget);
      text_color = &back_text_color;
    } 
  else 
    {
      day_rect.x += STEP;
	  day_rect.y += STEP;
	  day_rect.width  -= STEP;
	  day_rect.height -= STEP;

      cairo_set_source_rgba (cr, 0.125, 0.125, 0.125, priv->rect_day_transparency);

      if (calendar->selected_day == day)
	  {
        cairo_set_source_rgba (cr, 0.8, 0.8, 0.8, priv->rect_day_transparency);
      }

      gdk_cairo_rectangle (cr, &day_rect);
      cairo_fill(cr);


#if 0      
      if (calendar->highlight_row == row && calendar->highlight_col == col)
	{
	  cairo_set_source_rgb (cr, 0.75, 0.0, 0.0);
	  gdk_cairo_rectangle (cr, &day_rect);
	  cairo_fill (cr);
	}
#endif 


      if (calendar->marked_date[day-1]) {
	  
          text_color = MARKED_COLOR (widget);

      /* + */
//XXX
	  GdkRectangle frame;
	  frame.x = day_rect.x + 6;
	  frame.y = day_rect.y + 12;
	  frame.width = 20;
	  frame.height = 4;
	  
      gdk_cairo_set_source_color (cr, SELECTED_BG_COLOR (widget));
	  gdk_cairo_rectangle (cr, &frame);
      cairo_fill(cr);

	  frame.x = day_rect.x + 4 + frame.width/2;
	  frame.y = day_rect.y + 4;
	  frame.width = 4;
	  frame.height = 20;

      gdk_cairo_set_source_color (cr, SELECTED_BG_COLOR (widget));
	  gdk_cairo_rectangle (cr, &frame);
      cairo_fill(cr);
//end XXX  

      }
      else
          text_color = NORMAL_DAY_COLOR (widget);

      if (calendar->selected_day == day)
	  {
          text_color = SELECTED_FG_COLOR (widget);
      }
    }

  /* Translators: this defines whether the day numbers should use
   * localized digits or the ones used in English (0123...).
   *
   * Translate to "%Id" if you want to use localized digits, or
   * translate to "%d" otherwise.
   *
   * Note that translating this doesn't guarantee that you get localized
   * digits. That needs support from your system and locale definition
   * too.
   */
  g_snprintf (buffer, sizeof (buffer), C_("calendar:day:digits", "%d"), day);

  /* Get extra information to show, if any: */

  if (priv->detail_func)
    detail = gtk_kimi_calendar_get_detail (calendar, row, col);
  else
    detail = NULL;

  layout = gtk_widget_create_pango_layout (widget, buffer);
  pango_layout_set_alignment (layout, PANGO_ALIGN_CENTER);
  pango_layout_get_pixel_extents (layout, NULL, &logical_rect);
  
  //x_loc = day_rect.x + (day_rect.width - logical_rect.width) /2;
  //y_loc = day_rect.y;
  x_loc = day_rect.x + (day_rect.width - logical_rect.width) - 2;
  y_loc = day_rect.y + (day_rect.height - logical_rect.height) + 2;
  
  gdk_cairo_set_source_color (cr, text_color);
  cairo_move_to (cr, x_loc, y_loc);
  pango_cairo_show_layout (cr, layout);

  if (calendar->day_month[row][col] == MONTH_CURRENT &&
     (calendar->marked_date[day-1] || (detail && !show_details)))
    {
      cairo_move_to (cr, x_loc - 1, y_loc);
      pango_cairo_show_layout (cr, layout);
    }

  y_loc += priv->max_day_char_descent;

  if (priv->detail_func && show_details)
    {
      cairo_save (cr);

      if (calendar->selected_day == day)
        gdk_cairo_set_source_color (cr, &widget->style->text[GTK_STATE_ACTIVE]);
      else if (calendar->day_month[row][col] == MONTH_CURRENT)
        gdk_cairo_set_source_color (cr, &widget->style->base[GTK_STATE_ACTIVE]);
      else
        gdk_cairo_set_source_color (cr, &widget->style->base[GTK_STATE_INSENSITIVE]);

      cairo_set_line_width (cr, 1);
      cairo_move_to (cr, day_rect.x + 2, y_loc + 0.5);
      cairo_line_to (cr, day_rect.x + day_rect.width - 2, y_loc + 0.5);
      cairo_stroke (cr);

      cairo_restore (cr);

      y_loc += 2;
    }

  if (detail && show_details)
    {
      gchar *markup = g_strconcat ("<small>", detail, "</small>", NULL);
      pango_layout_set_markup (layout, markup, -1);
      g_free (markup);

      if (day == calendar->selected_day)
        {
          /* Stripping colors as they conflict with selection marking. */

          PangoAttrList *attrs = pango_layout_get_attributes (layout);
          PangoAttrList *colors = NULL;

          if (attrs)
            colors = pango_attr_list_filter (attrs, is_color_attribute, NULL);
          if (colors)
            pango_attr_list_unref (colors);
        }

      pango_layout_set_wrap (layout, PANGO_WRAP_WORD_CHAR);
      pango_layout_set_width (layout, PANGO_SCALE * day_rect.width);

      if (priv->detail_height_rows)
        {
          gint dy = day_rect.height - (y_loc - day_rect.y);
//          pango_layout_set_height (layout, PANGO_SCALE * dy);
          pango_layout_set_ellipsize (layout, PANGO_ELLIPSIZE_END);
        }

      cairo_move_to (cr, day_rect.x, y_loc);
      pango_cairo_show_layout (cr, layout);
    }

  if (GTK_WIDGET_HAS_FOCUS (calendar) 
      && calendar->focus_row == row && calendar->focus_col == col)
    {
      GtkStateType state;

      if (calendar->selected_day == day)
	state = GTK_WIDGET_HAS_FOCUS (widget) ? GTK_STATE_SELECTED : GTK_STATE_ACTIVE;
      else
	state = GTK_STATE_NORMAL;
      
      gtk_paint_focus (widget->style, 
		       priv->main_win,
	               state,
		       NULL, widget, "calendar-day",
		       day_rect.x,     day_rect.y, 
		       day_rect.width, day_rect.height); 
    }

  if (overflow)
    priv->detail_overflow[row] |= (1 << col);
  else
    priv->detail_overflow[row] &= ~(1 << col);

  g_object_unref (layout);
  cairo_destroy (cr);
  g_free (detail);
}

static void
calendar_paint_main (GtkKimiCalendar *calendar)
{
  gint row, col;
  
  for (col = 0; col < 7; col++)
    for (row = 0; row < 6; row++)
      calendar_paint_day (calendar, row, col);
}


static gboolean
gtk_kimi_calendar_expose (GtkWidget	    *widget,
		     GdkEventExpose *event)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  int i;
  
    /* set background image */
    if(priv->bg_pixbuf != NULL) {
        cairo_t *cr = gdk_cairo_create(priv->main_win);

        gdk_draw_pixbuf(priv->main_win, NULL, priv->bg_pixbuf, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NORMAL, 0, 0); 
        cairo_destroy (cr);
    }

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      if (event->window == priv->main_win)
	calendar_paint_main (calendar);
      
      if (event->window == priv->day_name_win)
	calendar_paint_day_names (calendar);
      
      if (event->window == priv->week_win)
	calendar_paint_week_numbers (calendar);
      if (event->window == widget->window)
	{
	  gtk_paint_shadow (widget->style, widget->window, GTK_WIDGET_STATE (widget),
			    GTK_SHADOW_IN, NULL, widget, "calendar",
			    0, 0, widget->allocation.width, widget->allocation.height);
	}
    }
  
  return FALSE;
}

/****************************************
 *           Mouse handling             *
 ****************************************/

static void
calendar_week_button_press ( GtkKimiCalendar *calendar,
                GdkEventButton *event)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  gint x, y;
  gint week = 0, year;
  gint row;

  year = calendar->year;

  x = (gint) (event->x);
  y = (gint) (event->y);

  row = calendar_row_from_y (calendar, y - priv->day_name_h);

  //printf("\nROW - %d\n", row);

  /* If row isn't found, just return. */
  if (row == -1) {
    return;
  }

  if (!(x < (priv->week_width - CALENDAR_MARGIN) && x > 0)) {
    return;
  }

  week_of_year (&week, &year,		
			     ((calendar->day[row][6] < 15 && row > 3 ? 1 : 0)
			      + calendar->month) % 12 + 1, calendar->day[row][6]);

  if (event->type == GDK_BUTTON_PRESS) {
    priv->in_drag = 0;
    g_signal_emit (calendar,
	               gtk_kimi_calendar_signals[WEEK_SELECTED_CLICK_SIGNAL],
	               0);
    //printf("\nWeek - %d\n", week);
  }
}

static void
calendar_main_button_press (GtkKimiCalendar	   *calendar,
			    GdkEventButton *event)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  gint x, y;
  gint row, col;
  gint day_month;
  gint day;
  
  x = (gint) (event->x);
  y = (gint) (event->y);

  /* save click x, y*/
  priv->click_x = x;
  priv->click_y = y;

  row = calendar_row_from_y (calendar, y);
  col = calendar_column_from_x (calendar, x);

  /* If row or column isn't found, just return. */
  if (row == -1 || col == -1)
    return;
  
  day_month = calendar->day_month[row][col];

  if (event->type == GDK_BUTTON_PRESS)
    {
      day = calendar->day[row][col];

     
      if (day_month == MONTH_PREV)
return;
//	calendar_set_month_prev (calendar);
      if (day_month == MONTH_NEXT)
return;
//	calendar_set_month_next (calendar);

      
      if (!GTK_WIDGET_HAS_FOCUS (widget))
	gtk_widget_grab_focus (widget);
	  
      if (event->button == 1) 
	{
	  priv->in_drag = 1;
	  priv->drag_start_x = x;
	  priv->drag_start_y = y;
	}

      calendar_select_and_focus_day (calendar, day);
    }
  else if (event->type == GDK_2BUTTON_PRESS)
    {
      priv->in_drag = 0;
      if (day_month == MONTH_CURRENT)
	g_signal_emit (calendar,
		       gtk_kimi_calendar_signals[DAY_SELECTED_DOUBLE_CLICK_SIGNAL],
		       0);
    }
}

static gboolean
gtk_kimi_calendar_button_press (GtkWidget	  *widget,
			   GdkEventButton *event)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);

  if (event->window == priv->main_win) {
    calendar_main_button_press (calendar, event);
  } else {
      if (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS) {
        calendar_week_button_press (calendar, event);
      }
  }
  
  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);

  return FALSE;
}

static gboolean
gtk_kimi_calendar_button_release (GtkWidget	  *widget,
			     GdkEventButton *event)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  
  gint x, y;
  gint row, col;
  gint day_month;

  x = (gint) (event->x);
  y = (gint) (event->y);

  row = calendar_row_from_y (calendar, y);
  col = calendar_column_from_x (calendar, x);

  /* If row or column isn't found, just return. */
  if (row == -1 || col == -1)
    return;
  
  day_month = calendar->day_month[row][col];

  if (event->button == 1) 
  {
     if (event->window == priv->main_win) {
         if(priv->click_x + 25 < x) {//back
 	         calendar_set_month_prev (calendar);
         } else {
	           if(priv->click_x - 25 > x) { //next
 	               calendar_set_month_next (calendar);
	           } else {
                     if (day_month == MONTH_CURRENT) {
                         g_signal_emit (calendar,
                                        gtk_kimi_calendar_signals[DAY_SELECTED_CLICK_SIGNAL],
                                        0);
                     }
               }
           }
     }
      if (priv->in_drag)
	priv->in_drag = 0;
  }

  return TRUE;
}

/****************************************
 *             Key handling              *
 ****************************************/

static void 
move_focus (GtkKimiCalendar *calendar, 
	    gint         direction)
{
  GtkTextDirection text_dir = gtk_widget_get_direction (GTK_WIDGET (calendar));
 
  if ((text_dir == GTK_TEXT_DIR_LTR && direction == -1) ||
      (text_dir == GTK_TEXT_DIR_RTL && direction == 1)) 
    {
      if (calendar->focus_col > 0)
	  calendar->focus_col--;
      else if (calendar->focus_row > 0)
	{
	  calendar->focus_col = 6;
	  calendar->focus_row--;
	}

      if (calendar->focus_col < 0)
        calendar->focus_col = 6;
      if (calendar->focus_row < 0)
        calendar->focus_row = 5;
    }
  else 
    {
      if (calendar->focus_col < 6)
	calendar->focus_col++;
      else if (calendar->focus_row < 5)
	{
	  calendar->focus_col = 0;
	  calendar->focus_row++;
	}

      if (calendar->focus_col < 0)
        calendar->focus_col = 0;
      if (calendar->focus_row < 0)
        calendar->focus_row = 0;
    }
}

static gboolean
gtk_kimi_calendar_key_press (GtkWidget   *widget,
			GdkEventKey *event)
{
  GtkKimiCalendar *calendar;
  gint return_val;
  gint old_focus_row;
  gint old_focus_col;
  gint row, col, day;
  
  calendar = GTK_KIMI_CALENDAR (widget);
  return_val = FALSE;
  
  old_focus_row = calendar->focus_row;
  old_focus_col = calendar->focus_col;

  switch (event->keyval)
    {
    case GDK_KP_Left:
    case GDK_Left:
      return_val = TRUE;
      if (event->state & GDK_CONTROL_MASK)
	calendar_set_month_prev (calendar);
      else
	{
	  move_focus (calendar, -1);
	  calendar_invalidate_day (calendar, old_focus_row, old_focus_col);
	  calendar_invalidate_day (calendar, calendar->focus_row,
				   calendar->focus_col);
	}
      break;
    case GDK_KP_Right:
    case GDK_Right:
      return_val = TRUE;
      if (event->state & GDK_CONTROL_MASK)
	calendar_set_month_next (calendar);
      else
	{
	  move_focus (calendar, 1);
	  calendar_invalidate_day (calendar, old_focus_row, old_focus_col);
	  calendar_invalidate_day (calendar, calendar->focus_row,
				   calendar->focus_col);
	}
      break;
    case GDK_KP_Up:
    case GDK_Up:
      return_val = TRUE;
      if (event->state & GDK_CONTROL_MASK)
	calendar_set_year_prev (calendar);
      else
	{
	  if (calendar->focus_row > 0)
	    calendar->focus_row--;
          if (calendar->focus_row < 0)
            calendar->focus_row = 5;
          if (calendar->focus_col < 0)
            calendar->focus_col = 6;
	  calendar_invalidate_day (calendar, old_focus_row, old_focus_col);
	  calendar_invalidate_day (calendar, calendar->focus_row,
				   calendar->focus_col);
	}
      break;
    case GDK_KP_Down:
    case GDK_Down:
      return_val = TRUE;
      if (event->state & GDK_CONTROL_MASK)
	calendar_set_year_next (calendar);
      else
	{
	  if (calendar->focus_row < 5)
	    calendar->focus_row++;
          if (calendar->focus_col < 0)
            calendar->focus_col = 0;
	  calendar_invalidate_day (calendar, old_focus_row, old_focus_col);
	  calendar_invalidate_day (calendar, calendar->focus_row,
				   calendar->focus_col);
	}
      break;
    case GDK_KP_Space:
    case GDK_space:
      row = calendar->focus_row;
      col = calendar->focus_col;
      
      if (row > -1 && col > -1)
	{
	  return_val = TRUE;

          day = calendar->day[row][col];
	  if (calendar->day_month[row][col] == MONTH_PREV)
	    calendar_set_month_prev (calendar);
	  else if (calendar->day_month[row][col] == MONTH_NEXT)
	    calendar_set_month_next (calendar);

	  calendar_select_and_focus_day (calendar, day);
	}
    }	
  
  return return_val;
}

/****************************************
 *           Misc widget methods        *
 ****************************************/

static void
calendar_set_background (GtkWidget *widget)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  gint i;
  
  if (GTK_WIDGET_REALIZED (widget))
    {

      if (priv->day_name_win)
	gdk_window_set_background (priv->day_name_win, 
				   BACKGROUND_COLOR (widget));
      if (priv->week_win)
	gdk_window_set_background (priv->week_win,
				   BACKGROUND_COLOR (widget));
      if (priv->main_win)
	gdk_window_set_background (priv->main_win,
				   BACKGROUND_COLOR (widget));
      if (widget->window)
	gdk_window_set_background (widget->window,
				   BACKGROUND_COLOR (widget)); 
    }
}

static void
gtk_kimi_calendar_style_set (GtkWidget *widget,
			GtkStyle  *previous_style)
{
  if (previous_style && GTK_WIDGET_REALIZED (widget))
    calendar_set_background (widget);
}

static void
gtk_kimi_calendar_state_changed (GtkWidget	   *widget,
			    GtkStateType    previous_state)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  int i;
  
  if (!GTK_WIDGET_IS_SENSITIVE (widget))
    {
      priv->in_drag = 0;    
    }

  calendar_set_background (widget);
}

static void
gtk_kimi_calendar_grab_notify (GtkWidget *widget,
			  gboolean   was_grabbed)
{
}

static gboolean
gtk_kimi_calendar_focus_out (GtkWidget     *widget,
			GdkEventFocus *event)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);

  calendar_queue_refresh (calendar);
  
  priv->in_drag = 0; 

  return FALSE;
}

/****************************************
 *          Drag and Drop               *
 ****************************************/

static void
gtk_kimi_calendar_drag_data_get (GtkWidget        *widget,
			    GdkDragContext   *context,
			    GtkSelectionData *selection_data,
			    guint             info,
			    guint             time)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  GDate *date;
  gchar str[128];
  gsize len;

  date = g_date_new_dmy (calendar->selected_day, calendar->month + 1, calendar->year);
  len = g_date_strftime (str, 127, "%x", date);
  gtk_selection_data_set_text (selection_data, str, len);
  
  g_free (date);
}

/* Get/set whether drag_motion requested the drag data and
 * drag_data_received should thus not actually insert the data,
 * since the data doesn't result from a drop.
 */
static void
set_status_pending (GdkDragContext *context,
                    GdkDragAction   suggested_action)
{
  g_object_set_data (G_OBJECT (context),
                     I_("gtk-calendar-status-pending"),
                     GINT_TO_POINTER (suggested_action));
}

static GdkDragAction
get_status_pending (GdkDragContext *context)
{
  return GPOINTER_TO_INT (g_object_get_data (G_OBJECT (context),
                                             "gtk-calendar-status-pending"));
}

static void
gtk_kimi_calendar_drag_leave (GtkWidget      *widget,
			 GdkDragContext *context,
			 guint           time)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);

  priv->drag_highlight = 0;
  gtk_drag_unhighlight (widget);
  
}

static gboolean
gtk_kimi_calendar_drag_motion (GtkWidget      *widget,
			  GdkDragContext *context,
			  gint            x,
			  gint            y,
			  guint           time)
{
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (widget);
  GdkAtom target;
  
  if (!priv->drag_highlight) 
    {
      priv->drag_highlight = 1;
      gtk_drag_highlight (widget);
    }
  
  target = gtk_drag_dest_find_target (widget, context, NULL);
  if (target == GDK_NONE || context->suggested_action == 0)
    gdk_drag_status (context, 0, time);
  else
    {
      set_status_pending (context, context->suggested_action);
      gtk_drag_get_data (widget, context, target, time);
    }
  
  return TRUE;
}

static gboolean
gtk_kimi_calendar_drag_drop (GtkWidget      *widget,
			GdkDragContext *context,
			gint            x,
			gint            y,
			guint           time)
{
  GdkAtom target;

  target = gtk_drag_dest_find_target (widget, context, NULL);  
  if (target != GDK_NONE)
    {
      gtk_drag_get_data (widget, context, 
			 target, 
			 time);
      return TRUE;
    }

  return FALSE;
}

static void
gtk_kimi_calendar_drag_data_received (GtkWidget        *widget,
				 GdkDragContext   *context,
				 gint              x,
				 gint              y,
				 GtkSelectionData *selection_data,
				 guint             info,
				 guint             time)
{
  GtkKimiCalendar *calendar = GTK_KIMI_CALENDAR (widget);
  guint day, month, year;
  gchar *str;
  GDate *date;
  GdkDragAction suggested_action;

  suggested_action = get_status_pending (context);

  if (suggested_action) 
    {
      set_status_pending (context, 0);
     
      /* We are getting this data due to a request in drag_motion,
       * rather than due to a request in drag_drop, so we are just
       * supposed to call drag_status, not actually paste in the
       * data.
       */
      str = (gchar*) gtk_selection_data_get_text (selection_data);

      if (str) 
	{
	  date = g_date_new ();
	  g_date_set_parse (date, str);
	  if (!g_date_valid (date)) 
	      suggested_action = 0;
	  g_date_free (date);
	  g_free (str);
	}
      else
	suggested_action = 0;

      gdk_drag_status (context, suggested_action, time);

      return;
    }

  date = g_date_new ();
  str = (gchar*) gtk_selection_data_get_text (selection_data);
  if (str) 
    {
      g_date_set_parse (date, str);
      g_free (str);
    }
  
  if (!g_date_valid (date)) 
    {
      g_warning ("Received invalid date data\n");
      g_date_free (date);	
      gtk_drag_finish (context, FALSE, FALSE, time);
      return;
    }

  day = g_date_get_day (date);
  month = g_date_get_month (date);
  year = g_date_get_year (date);
  g_date_free (date);	

  gtk_drag_finish (context, TRUE, FALSE, time);

  
  g_object_freeze_notify (G_OBJECT (calendar));
  if (!(calendar->display_flags & GTK_KIMI_CALENDAR_NO_MONTH_CHANGE)
      && (calendar->display_flags & GTK_KIMI_CALENDAR_SHOW_HEADING))
    gtk_kimi_calendar_select_month (calendar, month - 1, year);
  gtk_kimi_calendar_select_day (calendar, day);
  g_object_thaw_notify (G_OBJECT (calendar));  
}

/****************************************
 *              Public API              *
 ****************************************/

/**
 * gtk_kimi_calendar_new:
 * 
 * Creates a new calendar, with the current date being selected. 
 * 
 * Return value: a newly #GtkKimiCalendar widget
 **/
GtkWidget*
gtk_kimi_calendar_new (void)
{
  return g_object_new (GTK_TYPE_KIMI_CALENDAR, NULL);
}

/**
 * gtk_kimi_calendar_display_options:
 * @calendar: a #GtkKimiCalendar.
 * @flags: the display options to set.
 *
 * Sets display options (whether to display the heading and the month headings).
 * 
 * Deprecated: 2.4: Use gtk_kimi_calendar_set_display_options() instead
 **/
void
gtk_kimi_calendar_display_options (GtkKimiCalendar	       *calendar,
			      GtkKimiCalendarDisplayOptions flags)
{
  gtk_kimi_calendar_set_display_options (calendar, flags);
}

/**
 * gtk_kimi_calendar_get_display_options:
 * @calendar: a #GtkKimiCalendar
 * 
 * Returns the current display options of @calendar. 
 * 
 * Return value: the display options.
 *
 * Since: 2.4
 **/
GtkKimiCalendarDisplayOptions 
gtk_kimi_calendar_get_display_options (GtkKimiCalendar         *calendar)
{
  g_return_val_if_fail (GTK_IS_KIMI_CALENDAR (calendar), 0);

  return calendar->display_flags;
}

/**
 * gtk_kimi_calendar_set_rect_day_transparency:
 * @calendar: a #GtkKimiCalendar
 * @transparency: transparency
 **/
void gtk_kimi_calendar_set_rect_day_transparency (GtkKimiCalendar *calendar,
				                           double transparency)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));

  priv->rect_day_transparency = transparency;
}

/**
 * gtk_kimi_calendar_set_background:
 * @calendar: a #GtkKimiCalendar
 * @name: name image
 **/
gboolean
gtk_kimi_calendar_set_background (GtkKimiCalendar	       *calendar,
				  gchar* name)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));

  if(name == NULL) {
    return FALSE;
  }

  /* clear */
  if(priv->bg_pixbuf != NULL) {
    gdk_pixbuf_unref(priv->bg_pixbuf);
  }
 
  /* load image */
  priv->bg_pixbuf = gdk_pixbuf_new_from_file(name, NULL);

  if(priv->bg_pixbuf == NULL) {
    return FALSE;
  }

  return TRUE;
}

/**
 * gtk_kimi_calendar_set_display_options:
 * @calendar: a #GtkKimiCalendar
 * @flags: the display options to set
 * 
 * Sets display options (whether to display the heading and the month  
 * headings).
 *
 * Since: 2.4
 **/
void
gtk_kimi_calendar_set_display_options (GtkKimiCalendar	       *calendar,
				  GtkKimiCalendarDisplayOptions flags)
{
  GtkWidget *widget = GTK_WIDGET (calendar);
  GtkKimiCalendarPrivate *priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);
  gint resize = 0;
  gint i;
  GtkKimiCalendarDisplayOptions old_flags;
  
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));
  
  old_flags = calendar->display_flags;
  
  if (GTK_WIDGET_REALIZED (widget))
    {
      if ((flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_NO_MONTH_CHANGE)
	{
	  resize ++;
	  if (! (flags & GTK_KIMI_CALENDAR_NO_MONTH_CHANGE))
	    {
	      calendar->display_flags &= ~GTK_KIMI_CALENDAR_NO_MONTH_CHANGE;
	    }
	}
      
      if ((flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_SHOW_HEADING)
	{
	  resize++;
	  
	  if (flags & GTK_KIMI_CALENDAR_SHOW_HEADING)
	    {
	      calendar->display_flags |= GTK_KIMI_CALENDAR_SHOW_HEADING;
	    }
	}
      
      
      if ((flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
	{
	  resize++;
	  
	  if (flags & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
	    {
	      calendar->display_flags |= GTK_KIMI_CALENDAR_SHOW_DAY_NAMES;
	      calendar_realize_day_names (calendar);
	    }
	  else
	    {
	      gdk_window_set_user_data (priv->day_name_win, NULL);
	      gdk_window_destroy (priv->day_name_win);
	      priv->day_name_win = NULL;
	    }
	}
      
      if ((flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
	{
	  resize++;
	  
	  if (flags & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
	    {
	      calendar->display_flags |= GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS;
	      calendar_realize_week_numbers (calendar);
	    }
	  else
	    {
	      gdk_window_set_user_data (priv->week_win, NULL);
	      gdk_window_destroy (priv->week_win);
	      priv->week_win = NULL;
	    }
	}

      if ((flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_WEEK_START_MONDAY)
	g_warning ("GTK_KIMI_CALENDAR_WEEK_START_MONDAY is ignored; the first day of the week is determined from the locale");
      
      if ((flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_SHOW_DETAILS)
        resize++;

      calendar->display_flags = flags;
      if (resize)
	gtk_widget_queue_resize (GTK_WIDGET (calendar));
      
    } 
  else
    calendar->display_flags = flags;
  
  g_object_freeze_notify (G_OBJECT (calendar));
  if ((old_flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_SHOW_DAY_NAMES)
    g_object_notify (G_OBJECT (calendar), "show-day-names");
  if ((old_flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_NO_MONTH_CHANGE)
    g_object_notify (G_OBJECT (calendar), "no-month-change");
  if ((old_flags ^ calendar->display_flags) & GTK_KIMI_CALENDAR_SHOW_WEEK_NUMBERS)
    g_object_notify (G_OBJECT (calendar), "show-week-numbers");
  g_object_thaw_notify (G_OBJECT (calendar));
}

/**
 * gtk_kimi_calendar_select_month:
 * @calendar: a #GtkKimiCalendar
 * @month: a month number between 0 and 11.
 * @year: the year the month is in.
 * 
 * Shifts the calendar to a different month.
 * 
 * Return value: %TRUE, always
 **/
gboolean
gtk_kimi_calendar_select_month (GtkKimiCalendar *calendar,
			   guint	month,
			   guint	year)
{
  g_return_val_if_fail (GTK_IS_KIMI_CALENDAR (calendar), FALSE);
  g_return_val_if_fail (month <= 11, FALSE);
  
  calendar->month = month;
  calendar->year  = year;
  
  calendar_compute_days (calendar);
  calendar_queue_refresh (calendar);

  g_object_freeze_notify (G_OBJECT (calendar));
  g_object_notify (G_OBJECT (calendar), "month");
  g_object_notify (G_OBJECT (calendar), "year");
  g_object_thaw_notify (G_OBJECT (calendar));

  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[MONTH_CHANGED_SIGNAL],
		 0);
  return TRUE;
}

/**
 * gtk_kimi_calendar_select_day:
 * @calendar: a #GtkKimiCalendar.
 * @day: the day number between 1 and 31, or 0 to unselect 
 *   the currently selected day.
 * 
 * Selects a day from the current month.
 **/
void
gtk_kimi_calendar_select_day (GtkKimiCalendar *calendar,
			 guint	      day)
{
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));
  g_return_if_fail (day <= 31);
  
  /* Deselect the old day */
  if (calendar->selected_day > 0)
    {
      gint selected_day;
      
      selected_day = calendar->selected_day;
      calendar->selected_day = 0;
      if (GTK_WIDGET_DRAWABLE (GTK_WIDGET (calendar)))
	calendar_invalidate_day_num (calendar, selected_day);
    }
  
  calendar->selected_day = day;
  
  /* Select the new day */
  if (day != 0)
    {
      if (GTK_WIDGET_DRAWABLE (GTK_WIDGET (calendar)))
	calendar_invalidate_day_num (calendar, day);
    }
  
  g_object_notify (G_OBJECT (calendar), "day");

  g_signal_emit (calendar,
		 gtk_kimi_calendar_signals[DAY_SELECTED_SIGNAL],
		 0);
}

/**
 * gtk_kimi_calendar_clear_marks:
 * @calendar: a #GtkKimiCalendar
 * 
 * Remove all visual markers.
 **/
void
gtk_kimi_calendar_clear_marks (GtkKimiCalendar *calendar)
{
  guint day;
  
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));
  
  for (day = 0; day < 31; day++)
    {
      calendar->marked_date[day] = FALSE;
    }

  calendar->num_marked_dates = 0;
  calendar_queue_refresh (calendar);
}

/**
 * gtk_kimi_calendar_mark_day:
 * @calendar: a #GtkKimiCalendar 
 * @day: the day number to mark between 1 and 31.
 * 
 * Places a visual marker on a particular day.
 * 
 * Return value: %TRUE, always
 **/
gboolean
gtk_kimi_calendar_mark_day (GtkKimiCalendar *calendar,
		       guint	    day)
{
  g_return_val_if_fail (GTK_IS_KIMI_CALENDAR (calendar), FALSE);
  
  if (day >= 1 && day <= 31 && calendar->marked_date[day-1] == FALSE)
    {
      calendar->marked_date[day - 1] = TRUE;
      calendar->num_marked_dates++;
      calendar_invalidate_day_num (calendar, day);
    }
  
  return TRUE;
}

/**
 * gtk_kimi_calendar_unmark_day:
 * @calendar: a #GtkKimiCalendar.
 * @day: the day number to unmark between 1 and 31.
 * 
 * Removes the visual marker from a particular day.
 * 
 * Return value: %TRUE, always
 **/
gboolean
gtk_kimi_calendar_unmark_day (GtkKimiCalendar *calendar,
			 guint	      day)
{
  g_return_val_if_fail (GTK_IS_KIMI_CALENDAR (calendar), FALSE);
  
  if (day >= 1 && day <= 31 && calendar->marked_date[day-1] == TRUE)
    {
      calendar->marked_date[day - 1] = FALSE;
      calendar->num_marked_dates--;
      calendar_invalidate_day_num (calendar, day);
    }
  
  return TRUE;
}

/**
 * gtk_kimi_calendar_get_date:
 * @calendar: a #GtkKimiCalendar
 * @year: location to store the year number, or %NULL
 * @month: location to store the month number (between 0 and 11), or %NULL
 * @day: location to store the day number (between 1 and 31), or %NULL
 * 
 * Obtains the selected date from a #GtkKimiCalendar.
 **/
void
gtk_kimi_calendar_get_date (GtkKimiCalendar *calendar,
		       guint	   *year,
		       guint	   *month,
		       guint	   *day)
{
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));
  
  if (year)
    *year = calendar->year;
  
  if (month)
    *month = calendar->month;
  
  if (day)
    *day = calendar->selected_day;
}

/**
 * gtk_kimi_calendar_set_detail_func:
 * @calendar: a #GtkKimiCalendar.
 * @func: a function providing details for each day.
 * @data: data to pass to @func invokations.
 * @destroy: a function for releasing @data.
 *
 * Installs a function which provides Pango markup with detail information
 * for each day. Examples for such details are holidays or appointments. That
 * information is shown below each day when #GtkKimiCalendar:show-details is set.
 * A tooltip containing with full detail information is provided, if the entire
 * text should not fit into the details area, or if #GtkKimiCalendar:show-details
 * is not set.
 *
 * The size of the details area can be restricted by setting the
 * #GtkKimiCalendar:detail-width-chars and #GtkKimiCalendar:detail-height-rows
 * properties.
 *
 * Since: 2.14
 */
void gtk_kimi_calendar_set_detail_func(GtkKimiCalendar *calendar,
	                               gpointer func,
			               gpointer data,
		                       GDestroyNotify destroy)
{
  GtkKimiCalendarPrivate *priv;

  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));

  priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);

  if (priv->detail_func_destroy)
    priv->detail_func_destroy (priv->detail_func_user_data);

  priv->detail_func = func;
  priv->detail_func_user_data = data;
  priv->detail_func_destroy = destroy;

//  gtk_widget_set_has_tooltip (GTK_WIDGET (calendar), NULL != priv->detail_func);

  gtk_widget_queue_resize (GTK_WIDGET (calendar));
}

/**
 * gtk_kimi_calendar_set_detail_width_chars:
 * @calendar: a #GtkKimiCalendar.
 * @chars: detail width in characters.
 *
 * Updates the width of detail cells.
 * See #GtkKimiCalendar:detail-width-chars.
 *
 * Since: 2.14
 */
void
gtk_kimi_calendar_set_detail_width_chars (GtkKimiCalendar *calendar,
                                     gint         chars)
{
  GtkKimiCalendarPrivate *priv;

  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));

  priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);

  if (chars != priv->detail_width_chars)
    {
      priv->detail_width_chars = chars;
      g_object_notify (G_OBJECT (calendar), "detail-width-chars");
      gtk_widget_queue_resize_no_redraw (GTK_WIDGET (calendar));
    }
}

/**
 * gtk_kimi_calendar_set_detail_height_rows:
 * @calendar: a #GtkKimiCalendar.
 * @rows: detail height in rows.
 *
 * Updates the height of detail cells.
 * See #GtkKimiCalendar:detail-height-rows.
 *
 * Since: 2.14
 */
void gtk_kimi_calendar_set_detail_height_rows (GtkKimiCalendar *calendar,
                                     	       gint         rows)
{
  GtkKimiCalendarPrivate *priv;

  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));

  priv = GTK_KIMI_CALENDAR_GET_PRIVATE (calendar);

  if (rows != priv->detail_height_rows)
    {
      priv->detail_height_rows = rows;
      g_object_notify (G_OBJECT (calendar), "detail-height-rows");
      gtk_widget_queue_resize_no_redraw (GTK_WIDGET (calendar));
    }
}

/**
 * gtk_kimi_calendar_get_detail_width_chars:
 * @calendar: a #GtkKimiCalendar.
 *
 * Queries the width of detail cells, in characters.
 * See #GtkKimiCalendar:detail-width-chars.
 *
 * Since: 2.14
 *
 * Return value: The width of detail cells, in characters.
 */
gint gtk_kimi_calendar_get_detail_width_chars (GtkKimiCalendar *calendar)
{
  g_return_val_if_fail (GTK_IS_KIMI_CALENDAR (calendar), 0);
  return GTK_KIMI_CALENDAR_GET_PRIVATE (calendar)->detail_width_chars;
}

/**
 * gtk_kimi_calendar_get_detail_height_rows:
 * @calendar: a #GtkKimiCalendar.
 *
 * Queries the height of detail cells, in rows.
 * See #GtkKimiCalendar:detail-width-chars.
 *
 * Since: 2.14
 *
 * Return value: The height of detail cells, in rows.
 */
gint gtk_kimi_calendar_get_detail_height_rows (GtkKimiCalendar *calendar)
{
  g_return_val_if_fail (GTK_IS_KIMI_CALENDAR (calendar), 0);
  return GTK_KIMI_CALENDAR_GET_PRIVATE(calendar)->detail_height_rows;
}

/**
 * gtk_kimi_calendar_freeze:
 * @calendar: a #GtkKimiCalendar
 * 
 * Does nothing. Previously locked the display of the calendar until
 * it was thawed with gtk_kimi_calendar_thaw().
 *
 * Deprecated: 2.8: 
 **/
void gtk_kimi_calendar_freeze (GtkKimiCalendar *calendar)
{
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));
}

/**
 * gtk_kimi_calendar_thaw:
 * @calendar: a #GtkKimiCalendar
 * 
 * Does nothing. Previously defrosted a calendar; all the changes made
 * since the last gtk_kimi_calendar_freeze() were displayed.
 *
 * Deprecated: 2.8: 
 **/
void gtk_kimi_calendar_thaw (GtkKimiCalendar *calendar)
{
  g_return_if_fail (GTK_IS_KIMI_CALENDAR (calendar));
}

#define __GTK_KIMI_CALENDAR_C__
