#include "process.h"
#include "kimiconsole.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#define BUF 512
#define FORMAT "%512s"

static char* remove_first_word(const char* str)
{
    int i = 0;
    while (str[i] != ' ' && str[i] != '\0')
        i++;

    if (str[i] == '\0') 
        return strdup("");
                          
    return strdup(str + i + 1);
}

static int check_initialized(KimiConsole* kc)
{
    if (kc->state == INITIALIZED) {
        printf("Application already initialized\n");
        return 1;
    } else {
        return 0;
    }
}

static int check_not_initialized(KimiConsole* kc)
{
    if (kc->state == NOT_INITIALIZED) {
        printf("Application must be initialized\n");
        return 1;
    } else {
        return 0;
    }
}

static void output_event(KimiConsole* kc, Event* ev, FILE* out)
{
    int i;
    if (kc->output_mode != OM_TEST && ev->service_event_id)
        fprintf(out, " Module: %s\n", ev->service_event_id);
    if (kc->output_mode != OM_TEST && ev->id)
        fprintf(out, " Id: %s\n", ev->id);
    if (ev->title) fprintf(out, " Title: %s\n", ev->title);
    if (ev->description) fprintf(out, " Description: %s\n", ev->description);
    if (ev->location) fprintf(out, " Location: %s\n", ev->location);

    char buf[BUF];
    strftime(buf, BUF, "%H:%M", localtime(&ev->start_time));
    fprintf(out, " Start time: %s\n", buf);

    strftime(buf, BUF, "%H:%M", localtime(&ev->end_time));
    fprintf(out, " End time:   %s\n", buf);

    if (ev->per) {
        for (i = 0; i < ev->per->len; i++) {
            fprintf(out, " Period:  %s\n", kimi_per_get_ical(g_ptr_array_index(ev->per, i)));
        }
    }
}

static void output_date_event_list(KimiConsole* kc, GArray* del, FILE* out)
{
    if (!del) 
        return;
    char start[BUF];
    int i, j;
    for (i = 0; i < del->len; i++) {
        DateEventList dl = g_array_index(del, DateEventList, i);
        strftime(start, BUF, "%d/%m/%Y\n", gmtime(&dl.date));

        fputs(start, kc->out);  

        GPtrArray* evs = dl.events;     
        for (j = 0; j < evs->len; j++) {        
            Event* ev = g_ptr_array_index(evs, j);              
            output_event(kc, ev, kc->out);                                  
        }                                                                           
    }                                               
}

static void callback_show_banner(const char* title, const char* text)
{
    printf("Kimi message %s: %s\n", title, text);
}

static void callback_show_options(ConfigModule* cm, const char* title, const char* message, GPtrArray* options)
{
    printf("Settings: %s\n%s\n", title, message);

    Option* opt;
    int i;
    int val;
    for (i = 0; i < options->len; i++) {
        opt = g_ptr_array_index(options, i);

        switch (opt->type) {
        case OT_INTEGER:
            printf("Enter integer \"%s\" (%d): ",
                    opt->string,
                    opt->value.state == OS_SET ? opt->value.v_int : 0);
            scanf("%d", &val);
            kimi_conf_option_set_int(opt, val);
            break;
        case OT_STRING:
            printf("Enter string \"%s\" (%s): ",
                    opt->string,
                    opt->value.state == OS_SET ? opt->value.v_text : "no value");
            char buf[BUF_SIZE];
            fgets(buf, BUF_SIZE, stdin);
            kimi_conf_option_set_string(opt, buf);
            break;
    
        }
        opt->value.state = OS_SET;
    }
}

static void process_init(KimiConsole* kc, const char* com)
{
    KimiUICallbacks call;

    call.show_banner = callback_show_banner;
    call.show_options = callback_show_options;

    if (check_initialized(kc))
        return;
    if (kc->module_paths->len != 0) {
        kc->kimi = kimi_init(kc->module_paths, call, NULL);
    } else {
        kc->kimi = kimi_init(NULL, call, NULL);
    }
    kc->state = INITIALIZED;
}

static void process_path(KimiConsole* kc, const char* com)
{
    if (check_initialized(kc)) 
        return;

    GPtrArray* paths = kc->module_paths;
    char* path = remove_first_word(com);
    g_ptr_array_add(paths, path);
    printf("Path %s added to pathlist\n", path);
}

static void process_mlst(KimiConsole* kc, const char* com)
{
    if (check_not_initialized(kc))
        return;

    GPtrArray* modules = kimi_get_modules(kc->kimi);
    int i;
    for (i = 0; i < modules->len; i++) {
        Module* mod = g_ptr_array_index(modules, i);
        fprintf(kc->out, "%s: %s\n", mod->service_string, mod->service_name);
    }

    g_ptr_array_free(modules, TRUE);
}

// adde mod_id ev_id titile decs loc start end 
// path /home/ivashov/repo/modules/empty_module/.libs
// adde emod lol Title Desc Loca 1 2
static void process_adde(KimiConsole* kc, const char* com)
{
    if (check_not_initialized(kc))
        return;

    Event* ev = kimi_event_new(NULL);
    char mod_id[BUF];
    char ev_id[BUF];
    char title[BUF];
    char desc[BUF];
    char loc[BUF];
    char start[BUF];
    char end[BUF];
    char period[BUF*4];
    strcpy(period, "null");
    
    sscanf(com,
            "adde %s %s %s %s %s %s %s %s", 
            mod_id,
            ev_id,
            title,
            desc,
            loc,
            start,
            end,
            period);
            
    if (strcmp(ev_id, "null") != 0) {
        ev->id = strdup(ev_id);
    } else {
        ev->id = NULL;
    }
    
    if (strcmp(period, "null")) {
        ev->per = kimi_per_create_period();
        kimi_per_add_rule(ev->per, kimi_per_create_rule_from_ical(period, 0, 1));
    } else {
        ev->per = NULL;
    }
    
    ev->service_event_id = strdup(mod_id);
    ev->title = strdup(title);
    ev->description = strdup(desc);
    ev->location = strdup(loc);
    ev->start_time = kimi_per_get_time_t(start);
    ev->end_time = kimi_per_get_time_t(end);
    kimi_store(ev, kc->kimi, NULL);
    kimi_event_free(ev, NULL);
}

static void process_help(KimiConsole* kc, const char* com)
{
    kimi_show_banner(kc->kimi, "BANNER", "TEXT OF BANNER!!!");
    printf("Kimi console interface help\n");
    printf("Commands: \n\n");
    printf("path DIRECTORY_WITH_LIBRARIES\n");
    printf(" Add path to modules. Must be used before command \"init\"\n\n");
    printf("init\n Initialize libkimi. This command loads all modules, initializing them\n\n");
    printf("mlst\n Show list of loaded modules\n\n");
    printf("adde MODULE_ID EVENT_ID TITLE DESCRIPTION LOCATION START_TIME END_TIME PERIOD\n Add event to module MODULE_ID\n\n");
    printf("getp START END\n Get events by period [START; END]\n\n");
    printf("rmev MODULE_ID EVENT_ID\n\n");
    printf("intr Switch to interactive mode\n\n");
    printf("file FILE Execute file as script\n\n");
    printf("view Show current DateEventList\n\n");
}

static void process_getp(KimiConsole* kc, const char* com)
{
    if (check_not_initialized(kc))
        return;

    int i, j;
    char start[BUF], end[BUF];

    sscanf(com,
            "getp %s %s",
            start,
            end);

    kimi_get_by_period(kimi_per_get_time_t(start), kimi_per_get_time_t(end), kc->kimi, NULL);

    GArray* del = kc->kimi->date_event_list;
    output_date_event_list(kc, del, kc->out);
}

static void process_rmev(KimiConsole* kc, const char* com)
{
    if (check_not_initialized(kc))
        return;

    char mod_id[BUF];
    char ev_id[BUF];

    sscanf(com,
            "rmev %s %s",
            mod_id,
            ev_id);

    kimi_event_remove(ev_id, mod_id, kc->kimi, NULL);

}

static void process_intr(KimiConsole* kc, const char* com)
{
    kc->isInteractive = 1;
}

static void process_file(KimiConsole* kc, const char* com)
{
    char* path = remove_first_word(com);
    open_and_execute_script(kc, path);
    printf("%s\n", path);
    free(path);
}

static void process_filt(KimiConsole* kc, const char* com)
{
    int i;
    if (check_not_initialized(kc))
        return;
    
    if (!kc->kimi->date_event_list) {
        printf("Date_event_list doesn't exists\n");
    } else {
        char title[BUF];
        char desc[BUF];
        char loc[BUF];
        
        sscanf(com, "filt %s %s %s", title, desc, loc);
        
        Event* mask = kimi_event_new(NULL);
        mask->title = strcmp(title, "null") ? strdup(title) : NULL;
        mask->description = strcmp(desc, "null") ? strdup(desc) : NULL;
        mask->location = strcmp(loc, "null") ? strdup(loc) : NULL;
        
        GPtrArray* res = kimi_filter(mask, kc->kimi);
        
        for (i = 0; i < res->len; i++) {
            Event* ev = g_ptr_array_index(res, i);
            output_event(kc, ev, kc->out);
        }
        
        g_ptr_array_free(res, TRUE);
        kimi_event_free(mask, NULL);
    }
}

static void process_mode(KimiConsole* kc, const char* com)
{
    sscanf(com, "mode %d", &kc->output_mode);
}

static void process_exit(KimiConsole* kc, const char* com)
{
    kc->stop = -1;
}

static void process_view(KimiConsole* kc, const char* com)
{
    output_date_event_list(kc, kc->kimi->date_event_list, kc->out);
}

ProcessFunction get_process_function(const char* com)
{
    char buf[BUF];
    sscanf(com, FORMAT, buf);
    
    if (!strcmp(buf, "init")) {
        return process_init;
    }

    if (!strcmp(buf, "path")) {
        return process_path;
    }

    if (!strcmp(buf, "mlst")) {
        return process_mlst;
    }

    if (!strcmp(buf, "adde")) {
        return process_adde;
    }

    if (!strcmp(buf, "getp")) {
        return process_getp;
    }

    if (!strcmp(buf, "rmev")) {
        return process_rmev;
    }

    if (!strcmp(buf, "help")) {
        return process_help;
    }
    
    if (!strcmp(buf, "intr")) {
        return process_intr;
    }
    
    if (!strcmp(buf, "file")) {
        return process_file;
    }
    
    if (!strcmp(buf, "filt")) {
        return process_filt;
    }

    if (!strcmp(buf, "mode")) {
        return process_mode;
    }

    if (!strcmp(buf, "exit")) {
        return process_exit;
    }

    if (!strcmp(buf, "view")) {
        return process_view;
    }

    printf("%s: command not found\n", buf);
    return 0;
}

void process_command(KimiConsole* kc, const char* com)
{
    ProcessFunction proc = get_process_function(com);
    if (!proc) {
        return;
    }

    proc(kc, com);
}
