/*
 * Copyright (C) 2008-2009 Petrozavodsk State University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#ifndef EVENTS_H
#define EVENTS_H

#include <glib.h>
#include <appdata.h>
#include <libalarm.h>
#define INIT_SIZE (8)
#define MAX_QUERY 1024
struct contact {
    char* firstname;
    char* lastname;
    char* nick;
    char* foto;
    char* address;
//    char* work_address;
    char* phone;
//    char* home_phone;
//    char* work_phone;
//    char* fax;
    char* email;
    char* icq;
//    char* jabber;
    char* vkontakte;
};

/* Guests table fields */
/** Name of guests table */
#define TABLE_GUEST "tblGuests"
/** Event id */
#define G_EVENT_ID "id_event"
/** Guest id */
#define G_GUEST_ID "id_contact"

/** SQL query: create table tblGuests */
#define SQL_CREATE_TABLE_GUEST "CREATE TABLE "TABLE_GUEST"\
        ("G_EVENT_ID"          INTEGER,\
         "G_GUEST_ID"          TEXT)"



/* Event table fields */
/** Name of events table */
#define TABLE_EVENT     "tblEvent"
/** Identifier field */
#define F_ID            "id"
/** Event type field */
#define F_ID_TYPE       "id_type"
/** Title of event */
#define F_TITLE         "title"
/** Time where event start */
#define F_START_TIME    "start_time"
/** Time where event end */
#define F_END_TIME      "end_time"
/** Description of event */
#define F_DESC          "description"
/** Alarm time */
#define F_REMIND        "remind_time"
#define F_PERIOD        "period"
#define F_PER_START     "start_day"
#define F_PER_END       "end_day"
#define F_PER_DAYS      "days"
#define F_PER_REPEAT    "repeat"
#define F_PER_TYPE      "type"
#define F_LOCATION      "location"
#define F_COOKIE        "alarm_cookie"
#define F_SERVICE_ID    "service_event_id"

/** SQL query: create table tblEvent */
#define SQL_CREATE_TABLE_EVENT "CREATE TABLE "TABLE_EVENT"\
    ("F_ID"          INTEGER PRIMARY KEY AUTOINCREMENT,\
     "F_ID_TYPE"     INTEGER,\
     "F_TITLE"       TEXT,\
     "F_START_TIME"  INTEGER,\
     "F_END_TIME"    INTEGER,\
     "F_DESC"        TEXT,\
     "F_REMIND"      INTEGER,\
     "F_PER_START"   INTEGER,\
     "F_PER_END"     INTEGER,\
     "F_PER_DAYS"    TEXT,\
     "F_PER_REPEAT"  INTEGER,\
     "F_PER_TYPE"    INTEGER, \
     "F_LOCATION"    TEXT, \
     "F_COOKIE"      INTEGER, \
     "F_SERVICE_ID"  TEXT)"

/** SQL query: check is table tblEvent exists  */
#define SQL_CHECK_EXISTS "SELECT name from sqlite_master WHERE name='"TABLE_EVENT"'"

/** SQL query: check is table tblGuests exists */
#define SQL_CHECK_EXISTS_GUEST "SELECT name from sqlite_master WHERE name='"TABLE_GUEST"'"


/** SQL query: search event by given field and value of this field */
#define SQL_STMT_SEARCH_BY_FIELD "SELECT "F_ID" FROM "TABLE_EVENT" WHERE %s = ?"


#define DATABASE_ERROR_STRING "Database error"
#define ALLOCATE_ERROR_STRING "Failed allocate %d bytes"

/* Error code for Event Data Access subsystem */
#define EV_DB_ERROR g_quark_from_static_string("ev_db_error_quark")
#define EV_DB_OPEN_ERROR 1
#define EV_DB_OTHER_ERROR 2
#define EV_DB_BUSY_ERROR 3
#define EV_DB_MISUSE_ERROR 4

/* Error code for  Event Manager subsystem */
#define EV_MGR_ERROR g_quark_from_static_string("ev_mgr_error_quark")
#define EV_MGR_ALLOCATE_ERROR 1
#define EV_MGR_DB_ERROR 2
#define EV_MGR_ALARM_ERROR 3
/* Error code for  Event Data Select subsystem */
#define EV_SEL_ERROR g_quark_from_static_string("ev_sel_error_quark")
#define EV_SEL_DB_ERROR 1
#define EV_SEL_ALLOCATE_ERROR 2

/*typedef struct 

{
    time_t start_time;
    time_t end_time;
    int* days;
    int n_days;
    int repeat;
    int type;
} period;

typedef struct {
    int id;
    int id_type;
    char* title;
    time_t start_time;
    time_t end_time;
    char* description;
    time_t remind;
    cookie_t alarm_cookie;
    //contacts* guests;
    period* per;
    char *location;
} event;
*/

/* ========================================================================== */
/* Event Data Select subsystem                                                */
/* ========================================================================== */

/**
 * Select event by its ID
 *
 * Function returns errors: EV_SEL_ALLOCATE_ERROR
 * @param id event's identifier
 * @param ev event 
 * @param app_data global application information
 * @param error structure that holds info about an error (if any).
 * @return 0 for success, -1 otherwise.
 * */
int ev_sel_get_by_id(const int id, event* ev, AppData* app_data, GError** error);

/**
 * Select events for the period
 *
 * @param start    period start date
 * @param end      period end date 
 * @param app_data global application information
 * @param error    structure that holds info about an error (if any)
 * @return 0 for success, -1 otherwise.
 * */
int ev_sel_get_by_period(const time_t start, const time_t end, 
        AppData* app_data, GError** error);

/**
 * Free events array
 *
 * @param arr
 *
 */
void ev_sel_event_array_free(GPtrArray* arr);

/**
 * Search if field for value
 *
 * @param field
 * @param value value to search
 * @param err GError
 * @return event
 */
event* ev_sel_search_by_field(const char* field, const char* value, AppData* data, GError* err);

/**
 * Filter events 
 *
 * @param data AppData pointer
 * @param dil data id list 
 * @param filter   filter
 * @param error    structure that holds info about an error (if any)
 * @return searched events.
 * */
GPtrArray* ev_sel_filter(AppData *app, date_id_list *dil, const char *filter, GError **err);

/**
 * Analyze query results to check for periodic events 
 *
 * @param results  SQL query results
 * @param entries  number of entries int the result
 * @param cols     number of columns in an entry
 * @param app_data global application information
 * */
void ev_db_pereodic_analyze(GPtrArray* results, time_t start, time_t end, int* entries, int* cols, AppData* app);

/**
 * Allocate memory for array of the date_id_list structures 
 *
 * @param list date_id_list structures array
 * @param list_size size of the array
 * @param ids_size  size of the ids array for each date_id_list structure
 * */
void ev_db_date_id_list_alloc(struct date_id_list *list[], const size_t list_size, 
        const size_t ids_size);

/**
 * Free memory used by array of the date_id_list structures 
 *
 * @param list date_id_list structures array
 * @param list_size size of the array
 * */
void ev_db_date_id_list_free(AppData *app);

/**
 * @brief Convert days string to days array
 * @param days days, with comma delimiter
 * @return GArray of day numbers
 */
GArray* ev_sel_days_string_to_array(const char* days);

/* ========================================================================== */
/* Event Data Access                                                          */
/* ========================================================================== */

/**
 * @brief Init database
 *
 * Check if database exists. Create it if not
 * Pointer to database stores to AppData
 * Function returns errors: EV_DB_OPEN_ERROR (cannot open database), EV_DB_SQLITE_BUSY (database opened by other application)

 * @param app pointer to AppData
 * @param file database file path
 * @param err GError pointer
 * @return error code
 * @retval 0 no error
 * @retval !=0 error
 */
int ev_db_init(AppData *app, const char *file, GError **err);

/**
 * @brief Close databas
 * 
 * @param appdata AppData
 * @param GError pointer
 * @return error code
 * @retval 0  no error
 * @retval !=0 error
 */
int ev_db_close(AppData *app, GError *err);

/**
 * @brief Do request to database
 *
 * Function returns errors: EV_DB_SQLITE_BUSY (database opened by other application)
 * @param stmt precompiled query
 * @param n pointer to number of results
 * @param m number of fields
 * @param data AppData
 * @param error
 * @param query parameters. Must be pairs (DBType, value) ends with DB_END. Example:
 */
GPtrArray* ev_db_sql_query_prep_str(sqlite3_stmt* stmt, int* n, int* m, AppData* data, GError** error, ...);

/**
 * @brief Clear result array
 *
 * @param arr array
 * @param n Records count
 * @param m Fields count
 */
void ev_db_sql_query_clear_2(GPtrArray* arr, int m);

/* ========================================================================== */
/* Event manager                                                              */
/* ========================================================================== */

/**
 * @brief Creates new event struct.
 *
 * Function creates new event structure and initializes it.
 *
 * @param error Double pointer to GError
 * @return struct event
 * @retval 0 success
 * @retval 1 error
 */
event *ev_mgr_new(GError **error);

/**
 * @brief Store event structure into database
 *
 * Function stores event into database.
 * If ev->id = 0, then new database record creates.
 * Function returns errors: EV_MGR_DB_ERROR (database error)
 *
 * @param ev event, that will be overwrited
 * @param app AppData
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval 1 error
 */
int ev_mgr_store(event *ev, AppData *app, GError **error);

/**
 * @brief Remove structure from database
 *
 * Function removes event from database
 * Function returns errors: EV_MGR_DB_ERROR (database error)
 *
 * @param id event id
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval !=0 error
 */
int ev_mgr_remove(int id, AppData* app, GError** error);

/**
 * @brief free event structure
 *
 * Function frees memory, that has allocated for event structure
 *
 * @param event event structure
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval !=0 error
 */
int ev_mgr_clear(event *ev, GError **error);

/**
 * @brief Set event reminder
 *
 * Function set event reminder
 * And set alarm_cookie field
 * Function returns errors: EV_MGR_ALARM_ERROR (alarm error(not runned daemon))
 * @param ev event
 * @param current_time Current time
 * @param data
 * @param error GError
 */
int ev_mgr_reminder_add(event* ev, time_t current_time, AppData* data, GError** error);

/**
 * @brief Remove event reminder
 *
 * @param ev event
 * @param data
 * @param error GError
 */
int ev_mgr_reminder_remove(event* ev, AppData* data, GError** error);

/**
 * @brief Get repeat type of event
 *
 * @param ev struct event
 * @return RepeatType
 */
RepeatType ev_mgr_period_to_rt(event *ev);

/**
 * @brief Creates struct period from RepeatType and stores it to struct event
 *
 * @param ev struct event
 * @param ev RepeatType
 */
void ev_mgr_rt_to_period(RepeatType rt, event* ev);


// Contacts

void ev_add_contact(int event_id, const char* contact_id, AppData* data);
GList* ev_get_contacts(int event_id, AppData* data);
int ev_check_remove_contacts(const char* contact_id, AppData* data);
GtkWidget* ev_list_to_widget(GList* list, AppData* data);


/* For Maemo 5 */
/* Returns List of OssoABookContact */
GList* ev_mgr_get_select_contacts();
char* ev_mgr_get_contact_id(OssoABookContact* cont);
GList* ev_mgr_select_contact_from_event(int event_id, AppData* data);


/* Dialogs */
void ev_mgr_contact_dialog_add(int event_id, AppData* data);
void ev_mgr_contact_dialog_delete(int event_id, AppData* data);
void ev_mgr_contact_dialog_view(int event_id, AppData* data);
#endif
