/*
 * Copyright (C) 2008-2009 Petrozavodsk State University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
/* ========================================================================== */
/* Event Manager                                                              */
/*                                                                            */
/* Author: Ivashov K., Petrozavodsk State University                          */
/* Date  : 2/09/2009                                                         */
/* ========================================================================== */

#include <events/events.h>
#include <string.h>
#define BUF 512
#define SECONDS_IN_DAY (24 * 60 * 60)
#include <stdarg.h>
/**
 * Struct, that pass into g_hash_table_foreach()
 */
typedef struct
{
    /**
     * AppData, with initialized id_list
     */
    AppData* app;

    /**
     * Current id_list element
     */
    int current;
} ev_sel_date_app_n;

/**
 * Contains event id and it's time
 * This is value for hashtable
 */
typedef struct
{
    int id;
    int hour;
    int min;
    int sec;
} ev_sel_array_element;

/**
 * @brief Creates new event struct.
 *
 * Function creates new event structure and initializes it.
 * 
 * @param error Double pointer to GError
 * @return struct event
 * @retval 0 success
 * @retval 1 error
 */
event* ev_mgr_new(GError** error)
{
    event* ev = (event*) calloc(1, sizeof(event)); /* Allocate memory */
    if (!ev) {
        g_set_error(error, EV_MGR_ERROR, EV_MGR_ALLOCATE_ERROR, ALLOCATE_ERROR_STRING, sizeof(event));
    }
    return ev;
}

/**
 * @brief Store event structure into database
 *
 * Function stores event into database.
 * If ev->id = 0, then new database record will be created.
 * Function returns errors: EV_MGR_DB_ERROR (database error)
 *
 * @param ev event, that will be overwrited
 * @param app AppData
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval 1 error
 */
int ev_mgr_store(event* ev, AppData* app, GError** error)
{
    char days[128];
    int i;

    /* If event has reminder then remove it */
    GError* err = NULL;
    ev_mgr_reminder_remove(ev, app, &err);
    if(err)
        g_error_free(err);
    /* Convert days array to string */
    if (ev->per && ev->per->days) {
        strcpy(days, "");

        for (i = 0; i < ev->per->n_days; i++) {
            char day[3];
            sprintf(day, "%d,", ev->per->days[i]);
            strcat(days, day);
        }

        days[strlen(days) - 1] = '\0';
    }

    sqlite3_stmt* stmt;

    int n; /* Records number */
    int m; /* Fields number */
    GPtrArray* arr = NULL;
    err = NULL;

    /* Overwrite existing record */

    if (ev->id != 0) {
        stmt = app->db_data.insert;

        arr = ev_db_sql_query_prep_str(stmt, &n, &m, app, &err, 
                DB_INT, ev->id,
                DB_INT, ev->id_type,
                DB_TEXT, ev->title ? ev->title : "",
                DB_INT, ev->start_time,
                DB_INT, ev->end_time,
                DB_TEXT, ev->description ? ev->description : "",
                DB_INT, ev->remind, 
                DB_INT, ev->per ? ev->per->start_time : 0, 
                DB_INT, ev->per ? ev->per->end_time : 0,
                DB_TEXT, ev->per && ev->per->days ? days : NULL,
                DB_INT, ev->per ? ev->per->repeat : 0, 
                DB_INT, ev->per ? ev->per->type : 0,
                DB_TEXT, ev->location ? ev->location : "",
                DB_INT64, ev->alarm_cookie,
                DB_TEXT, ev->service_event_id,
                DB_END);

    } else { /* Create new record */
        stmt = app->db_data.insert_new;

        arr = ev_db_sql_query_prep_str(stmt, &n, &m, app, &err,
                DB_INT, ev->id_type,
                DB_TEXT, ev->title ? ev->title : "",
                DB_INT, ev->start_time,
                DB_INT, ev->end_time,
                DB_TEXT, ev->description ? ev->description : "",
                DB_INT, ev->remind,
                DB_INT, ev->per ? ev->per->start_time : 0,
                DB_INT, ev->per ? ev->per->end_time : 0,
                DB_TEXT, ev->per && ev->per->days ? days : NULL,
                DB_INT, ev->per ? ev->per->repeat : 0,
                DB_INT, ev->per ? ev->per->type : 0,
                DB_TEXT, ev->location ? ev->location : "",
                DB_INT64, ev->alarm_cookie,
                DB_TEXT, ev->service_event_id,
                DB_END);
    }
    if (err) {
        g_error_free(err);
        g_set_error(error, EV_MGR_ERROR, EV_MGR_DB_ERROR, DATABASE_ERROR_STRING);
        return -1;
    }

    ev_db_sql_query_clear_2(arr, m);
    
    /* Get new record identifier */
    if (ev->id == 0) {
        int rowid = sqlite3_last_insert_rowid(app->sqlite);

        err = NULL;
        GPtrArray* result = ev_db_sql_query_prep_str(app->db_data.get_last_id, &n, &m, app, &err, DB_INT, rowid, DB_END);
        if(err) {
            g_error_free(err);
            g_set_error(error, EV_MGR_ERROR, EV_MGR_DB_ERROR, DATABASE_ERROR_STRING);
            return -1;
        }
        if (result) {
            char** string = g_ptr_array_index(result, 0); 
            ev->id = atoi(string[0]);
            g_debug("Last inserted id = %d\n", ev->id);
        }
        else {
            g_debug("Cannot get last inserted id");
        }
        ev_db_sql_query_clear_2(result, 1);
    }
    /* Set reminder for event */
    ev_mgr_reminder_add(ev, time(NULL), app, &err);
    return 0;
}


/**
 * @brief Remove structure from database
 *
 * Function removes event from database
 *
 * @param id event id
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval !=0 error
 */
int ev_mgr_remove(int id, AppData* app, GError** error)
{
    GError* err = NULL;

    int n;
    int m;

    /* Check is reminder is set */
    GPtrArray* result = ev_db_sql_query_prep_str(app->db_data.get_alarm, &n, &m, app, &err, DB_INT, id, DB_END);

    if (err) {
        g_error_free(err);
        g_set_error(error, EV_MGR_ERROR, EV_MGR_DB_ERROR, DATABASE_ERROR_STRING);
        return -1;
    }

    char** string = g_ptr_array_index(result, 0);
    cookie_t alarm_cookie = string[0] ? atoi(string[0]) : 0;
    ev_db_sql_query_clear_2(result, m);

    if (alarm_cookie) {
        g_debug("Deleting alarm with cookie %d\n", alarm_cookie);
        alarmd_event_del(alarm_cookie);
    }

    err = NULL;
    g_debug("Deleting event with id %d\n", id);
    result = ev_db_sql_query_prep_str(app->db_data.del, &n, &m, app, &err, DB_INT, id, DB_END);
    if (err) {
        g_error_free(err);
        g_set_error(error, EV_MGR_ERROR, EV_MGR_DB_ERROR, DATABASE_ERROR_STRING);
        return -1;
    }
    ev_db_sql_query_prep_str(app->db_data.remove_by_event_id, &n, &m, app, NULL, DB_INT, id, DB_END);

    ev_db_sql_query_clear_2(result, m);
    return 0;
}

/**
 * @brief Аree event structure
 *
 * Function frees memory, that has allocated for event structure
 *
 * @param event event structure
 * @param error Double pointer to GError
 * @return return code
 * @retval 0 success
 * @retval !=0 error
 */
int ev_mgr_clear(event* ev, GError** error)
{
    if (!ev)
        return 0;
    if (ev->title) free(ev->title);
    if (ev->description) free(ev->description);
    if (ev->location) free(ev->location);
    if (ev->service_event_id) free(ev->service_event_id);

    if (ev->per) {
        if (ev->per->days)
            free(ev->per->days);

        free(ev->per);
    }

    free(ev);
    return 0;
}

/**
 * @brief Hash function for GDate
 *
 * @param p GDate pointer
 * @return data hash
 */
static guint ev_sel_date_hash(gconstpointer p)
{
    GDate* date = (GDate*) p;
    /* hash = year * 366 + day */
    return g_date_get_day_of_year(date) + g_date_get_year(date) * 366;
}

/**
 * @brief Compare two dates
 *
 * @param a first date
 * @param b second date
 * @return result
 * @retval TRUE if dates equals
 * @retval FALSE otherwise
 */
static gboolean ev_sel_date_compare(gconstpointer a, gconstpointer b)
{
    if (g_date_compare((GDate*) a, (GDate*) b) == 0) {
        return TRUE;
    } else {
        return FALSE;
    }
}
/**
 * @brief Destroy key of hashtable
 *
 * @param a date
 */

static void ev_sel_date_destroy_key(gpointer a)
{
    g_date_free((GDate*) a);
}

/**
 * @brief Destroy value of hashtable
 * 
 * @param a array
 */
static void ev_sel_date_destroy_value(gpointer a)
{
    g_array_free((GArray*) a, TRUE);
}

/**
 * @brief Add element into hashtable
 *
 * If date int hashtable then elem will be added to
 * end of value array,
 * else new record will be created
 *
 * @param hash GHashtable
 * @param date data
 * @param elem value
 */
static void ev_sel_date_add_event_to_hash(GHashTable* hash, GDate* date, ev_sel_array_element elem)
{
    gpointer val = g_hash_table_lookup(hash, date);

    /* Key already in hashtable */
    if (val) {
        GArray *arr = (GArray*) val;
        g_array_append_val(arr, elem);
    } else {
        GArray *arr = g_array_new(FALSE, FALSE, sizeof(ev_sel_array_element));
        g_array_append_val(arr, elem);
        GDate *date_to_store =  g_date_new_dmy(g_date_get_day(date), g_date_get_month(date), g_date_get_year(date));
        g_hash_table_insert(hash, date_to_store, arr);
    }
}

/**
 * @brief Convert GDate to time_t
 * 
 * @param d GDate
 * @return time_t
 */
static time_t gdate2time_t(GDate* d)
{
    struct tm t;
    g_date_to_struct_tm(d, &t);
    /* Get current daylight saving */
    /*struct tm tc;
    time_t c = time(NULL);
    tc = *localtime(&c);
    int dst = tc.tm_isdst;*/
    t.tm_isdst = -1;
    return mktime(&t);
}

/**
 * @brief Compare two dates
 *
 * @param a first date
 * @param b second date
 * @return compare esult
 * @retval <0, if a < b
 * @retval >0, if a > b
 * @retval  0, if a = b 
 */
static gint ev_sel_date_compare2(gconstpointer a, gconstpointer b)
{
    ev_sel_array_element* e1 = (ev_sel_array_element*)a;
    ev_sel_array_element* e2 = (ev_sel_array_element*)b;

    int e1abs = e1->hour * 3600 + e1->min * 60 + e1->sec;
    int e2abs = e2->hour * 3600 + e2->min * 60 + e2->sec;

    return e1abs - e2abs; 
}

/**
 * @brief Function add element from hashtable to date_id_list
 * 
 * @param key date
 * @param value event identifiers
 * @param user_data ev_sel_date_app_n, contains counter of elements and date_id_list
 */
static void ev_sel_date_callback(gpointer key, gpointer value, gpointer user_data)
{
    GDate* date = (GDate*) key;
    GArray* arr = (GArray*) value;

    g_array_sort(arr, ev_sel_date_compare2);

    ev_sel_date_app_n* dn = (ev_sel_date_app_n*) user_data;
    date_id_list* dil = dn->app->id_list;

    int i;

    dil[dn->current].date = gdate2time_t(date);
    dil[dn->current].ids = malloc(arr->len * sizeof(int));
    dil[dn->current].ids_size = arr->len;

    for (i = 0; i < arr->len; i++) {
        dil[dn->current].ids[i] = g_array_index(arr, ev_sel_array_element, i).id;
    }

    dn->current++;
}


static void ev_sel_date_to_period_begin(GDate* date, int per_type) 
{
    switch (per_type) {
    case 2:
        g_date_subtract_days(date, g_date_get_weekday(date) - 1);
        break;

    case 3:
        g_date_set_day(date, 1);
        break;
    }

}

/**
 * @brief Copy second date to first
 * @param d1 destination
 * @param d2 source
 */
static void ev_sel_copy_date(GDate* d1, GDate* d2) {
    g_date_clear(d1, 1);
    g_date_set_day(d1, g_date_get_day(d2));
    g_date_set_month(d1, g_date_get_month(d2));
    g_date_set_year(d1, g_date_get_year(d2));
}

/**
 * @brief Find next day in period 
 *
 * @param event_start 
 * @param start
 * @param per_type
 * @param per_repeat
 */
void ev_sel_get_next_in_period(const GDate* event_start, GDate* start, int per_type, int per_repeat) {
    GDate ev_tmp;
    g_date_clear(&ev_tmp, 1);

    g_date_set_day(&ev_tmp, g_date_get_day(event_start));
    g_date_set_month(&ev_tmp, g_date_get_month(event_start));
    g_date_set_year(&ev_tmp, g_date_get_year(event_start));


    switch (per_type) {
    case 2:
        g_date_subtract_days(start, g_date_get_weekday(start) - 1);
        g_date_subtract_days(&ev_tmp, g_date_get_weekday(&ev_tmp) - 1);

        break;

    case 3:
        g_date_set_day(start, 1);
        g_date_set_day(&ev_tmp, 1);

        break;
    }

    /* Search day, that match period */
    int j;
    for (j = 0; j < per_repeat; j++) {
        int diff = 0;
        switch (per_type) {
        case 1:
            diff = g_date_days_between(&ev_tmp, start); /* Days between event day and current day */
            break;

        case 2:
            diff = g_date_days_between(&ev_tmp, start) / 7; /* Weeks between event day and current day */
            break;

        case 3:
            diff = (g_date_get_year(start) * 12 + (g_date_get_month(start) - 1)) /* Months between event day and current day  */
                 - (g_date_get_year(&ev_tmp) * 12 + (g_date_get_month(&ev_tmp) - 1)) ;
            break;
        }


        if (diff % per_repeat == 0) {
            break;
        }

        switch (per_type) {
        case 1:
            g_date_add_days(start, 1);
            break;

        case 2:
            g_date_add_days(start, 7);
            break;

        case 3:
            g_date_add_months(start, 1);
            break;
        }
    }
}

GArray* ev_sel_days_string_to_array(const char* days)
{
    GArray* result = NULL; //Days array
    gchar** daysc = g_strsplit(days, ",", 128); /* Tokenize string */

    /* Convert days string to days array */
    if (daysc) {
        result = g_array_new(FALSE, FALSE, sizeof(int));

        int k;
        for (k = 0; daysc[k]; k++) {
            int day = atoi(daysc[k]);
            g_array_append_val(result, day);
        }
        g_strfreev(daysc);
    }
    return result;
}

static void pr_dt(GDate* date)
{
    g_debug("%d %d %d\n", g_date_get_day(date), g_date_get_month(date), g_date_get_year(date));
}

/**
 * Analyze query results to check for periodic events
 *
 * @param results  SQL query results
 * @param entries  number of entries int the result
 * @param cols     number of columns in an entry
 * @param app_data global application information
 */
void ev_db_pereodic_analyze(GPtrArray* results, time_t start, time_t end, int *entries, int *cols, AppData *app)
{
    /* Hashtable, where stores dates and identifiers */
    GHashTable* hash = g_hash_table_new_full(ev_sel_date_hash, ev_sel_date_compare, ev_sel_date_destroy_key, ev_sel_date_destroy_value); 

    GDate d[6];

    GDate* start_d = &d[0]; 
    GDate* ev_d = &d[1]; 
    GDate* end_d = &d[2]; 
    GDate* per_start_d = &d[3];
    GDate* per_end_d = &d[4];
    GDate* test = &d[5];
    int i;
    /* Check is event in period */
    for (i = 0; i < *entries; i++) {
        g_date_clear(d + 1, 6);
        char** string = g_ptr_array_index(results, i);
        int id = atoi(string[6]);
        time_t ev = atoi(string[0]); /* event start time */

        struct tm* ev_time = localtime(&ev); /* Convert from time_t to struct tm */

        int hour = ev_time->tm_hour;
        int min = ev_time->tm_min;
        int sec = ev_time->tm_sec;

        time_t per_start = string[1] ? atoi(string[1]) : 0; /* Period start */
        time_t per_end = string[2] ? atoi(string[2]) : 0; /* Period end */

        GArray* days = string[3] ? ev_sel_days_string_to_array(string[3]) : NULL; //Days array

        time_t per_repeat = string[4] ? atoi(string[4]) : 0; 
        time_t per_type = string[5] ? atoi(string[5]) : 0; 

        g_date_set_time_t(ev_d, ev); /* Convert from time_t to GDate */
        g_date_set_time_t(start_d, start);
        g_date_set_time_t(end_d, end);
        g_date_set_time_t(per_start_d, per_start);
        g_date_set_time_t(per_end_d, per_end);

        int real_date_added = 0; /* This will be set if event start day in period. */

        GDate* max = g_date_compare(start_d, per_start_d) > 0 ? start_d : per_start_d; /* Start day for period calculating.
                                                                                        * max from (interval start; period start) */
        g_date_set_dmy(test, /* Copy max to temporary variable */
                g_date_get_day(max),
                g_date_get_month(max),
                g_date_get_year(max));

        ev_sel_get_next_in_period(ev_d, test, per_type, per_repeat);
        switch (per_type) {
            case 1: /* Type: day */

            /* Store event to hash */
            while (g_date_compare(test, end_d) <= 0 && g_date_compare(test, per_end_d) <= 0) {
                if (g_date_compare(test, ev_d) == 0)  /* Set flag if day in period */
                    real_date_added = 1;

                ev_sel_array_element elem;
                elem.id = id;
                elem.hour = hour;
                elem.min = min;
                elem.sec = sec;
                ev_sel_date_add_event_to_hash(hash, test, elem); /* Store event to hash */
                g_date_add_days(test, per_repeat); /* Next day in period */
            }
            break;

        case 2: /* Type: week */
            while (g_date_compare(test, end_d) <= 0 && g_date_compare(test, per_end_d) <= 0) {
                int k;
                /* Add elements from days array to current date and append result to hash */
                for (k = 0; k < days->len; k++) {
                    GDate tmp;
                    ev_sel_copy_date(&tmp, test);

                    g_date_add_days(&tmp, g_array_index(days, int, k) - 1);

                    if (g_date_compare(max, &tmp) <= 0 && g_date_compare(&tmp, end_d) <= 0 && g_date_compare(&tmp, per_end_d) <= 0) {
                        if (g_date_compare(&tmp, ev_d) == 0)
                            real_date_added = 1;
                        ev_sel_array_element elem;

                        elem.id = id;
                        elem.hour = hour;
                        elem.min = min;
                        elem.sec = sec;

                        ev_sel_date_add_event_to_hash(hash, &tmp, elem);
                    }
                }
                g_date_add_days(test, per_repeat * 7);
            }               
            break;

        case 3: /* Type: Month */
            while (g_date_compare(test, end_d) <= 0 && g_date_compare(test, per_end_d) <= 0) {
                int k;

                for (k = 0; k < days->len; k++) {
                    GDate tmp;

                    ev_sel_copy_date(&tmp, test);
                    g_date_add_days(&tmp, g_array_index(days, int, k) - 1);

                    if (g_date_compare(max, &tmp) <= 0 && g_date_compare(&tmp, end_d) <= 0 && g_date_compare(&tmp, per_end_d) <= 0) {
                        ev_sel_array_element elem;

                        if (g_date_compare(&tmp, ev_d) == 0)
                            real_date_added = 1;

                        elem.id = id;
                        elem.hour = hour;
                        elem.min = min;
                        elem.sec = sec;

                        ev_sel_date_add_event_to_hash(hash, &tmp, elem);
                    }
                }
                g_date_add_months(test, per_repeat);
            }
            break;
        }

        /* If event start_time not in hash then
         * Append start_time to hash */
        if (!real_date_added && g_date_compare(ev_d, end_d) <= 0 && g_date_compare(ev_d, start_d) >= 0) {
            ev_sel_array_element elem;

            elem.id = id;
            elem.hour = hour;
            elem.min = min;
            elem.sec = sec;

            ev_sel_date_add_event_to_hash(hash, ev_d, elem);
        }
        if (days) g_array_free(days, TRUE);
    }

    ev_sel_date_app_n dn;

    dn.app = app;
    dn.current = 0;
    int size_to_alloc = (g_hash_table_size(hash) + 1) * sizeof(struct date_id_list);
    date_id_list *dil = malloc(size_to_alloc);

    app->id_list = dil;

    g_hash_table_foreach (hash, ev_sel_date_callback, (gpointer)&dn); /* Convert hashtable to date_id_list */
    g_hash_table_destroy(hash);

    dil[dn.current].date = -1;
}

/**
 * @brief Set event reminder
 *
 * Function set event reminder
 * And set alarm_cookie field
 * Function returns errors: EV_MGR_ALARM_ERROR (alarm error(not runned daemon))
 * @param ev event
 * @param current_time Current time
 * @param data
 * @param error GError
 */
int ev_mgr_reminder_add(event* ev, time_t current_time, AppData* data, GError** error)
{
    if (!ev->remind) 
        return 0;

    int i;
    int j;

    GDate current_date;
    GDate event_date;
    /* Time, where begins iterations */
    time_t start_time;

    /* Remove previous alarm  */
    if (ev->alarm_cookie) {
        alarmd_event_del(ev->alarm_cookie);
    }

    time_t next_event_time = -1;

    time_t event_time = ev->start_time;

    g_date_clear(&current_date, 1);
    g_date_set_time_t(&current_date, current_time);

    g_date_clear(&event_date, 1);
    g_date_set_time_t(&event_date, event_time);

    /* Time of begin week/month */
    GDate event_time_b;

    if (ev->per && ev->per->type) {
        g_date_set_time_t(&event_time_b, event_time);
        ev_sel_date_to_period_begin(&event_time_b, ev->per->type);
    }

    struct tm* t = localtime(&event_time); /* Convert from time_t to struct tm */

    int hour = t->tm_hour;
    int min = t->tm_min;
    int sec = t->tm_sec;

    t = localtime(&current_time);
    t->tm_hour = hour;
    t->tm_min = min;
    t->tm_sec = sec;

    start_time = mktime(t);

    int cont = 1;
    for (i = 0; cont; i++) {
        time_t time_to_check = SECONDS_IN_DAY * i + start_time;

        /* check if time_to_check in period */
        if (time_to_check - ev->remind <= current_time) {
            continue;
        }

        if (ev->per && time_to_check <= ev->per->start_time) {
            continue;
        }

        GDate time_to_check_b;
        g_date_set_time_t(&time_to_check_b, time_to_check);

       /* If time_to_check equals event_start time then break loop */
        if (!g_date_compare(&time_to_check_b, &event_date)) {
            next_event_time = time_to_check;
            break;
        }
        /* If event has period then check, is time_to_check in period */
        if (ev->per && ev->per->type) {
            /* Time_to_check  -> brgin of week/month */
            ev_sel_date_to_period_begin(&time_to_check_b, ev->per->type);

            int diff = 0;
            switch (ev->per->type) {
            case 1:
                diff = g_date_days_between(&event_time_b, &time_to_check_b); /* Days between event day and current day */
                break;

            case 2:
                diff = g_date_days_between(&event_time_b, &time_to_check_b) / 7; /* Weeks between event day and current day */
                break;

            case 3:
                diff = (g_date_get_year(&time_to_check_b) * 12 + (g_date_get_month(&time_to_check_b) - 1)) /* Months between event day and current day  */
                     - (g_date_get_year(&event_time_b) * 12 + (g_date_get_month(&event_time_b) - 1)) ;
                break;
            }
            /* time_to_check is not in period */
            if (diff % ev->per->repeat != 0) {
                continue;
            } else if (ev->per->type == 1) {/* Time to check in period and type = day => we needn't look days array */
                next_event_time = time_to_check;
                break;
            }

            GDate time_to_check_day;
            g_date_set_time_t(&time_to_check_day, time_to_check);

            for(j = 0; j < ev->per->n_days; j++) {
                if (g_date_days_between(&time_to_check_b, &time_to_check_day) == ev->per->days[j] - 1) {
                    next_event_time = time_to_check;
                    cont = 0;
                    break;
                }
            }
        } else if (time_to_check > event_time) {
            break;
        }
    } 

    if (next_event_time > 0) {
        time_t remind_time = next_event_time - ev->remind; /* Calculating remind time */ 

        alarm_event_t* eve = NULL;
        alarm_action_t *act = NULL;
        
        eve = alarm_event_create();
        alarm_event_set_alarm_appid(eve, "Kimi");

        eve->alarm_time = remind_time;

        char exec_name[MAX_QUERY];

        /* Set path to database */
        char* database_path = build_full_path("");
        sprintf(exec_name, "kimialarm %d %s", ev->id, database_path);

        g_debug("Path: %s\n", exec_name);
        
        act = alarm_event_add_actions(eve, 1);
        alarm_action_set_exec_command(act, exec_name);

        act->flags = ALARM_ACTION_WHEN_DELAYED
            | ALARM_ACTION_TYPE_EXEC
            | ALARM_ACTION_WHEN_TRIGGERED; 
        /* Add the alarm to alarmd and
        * verify that it was correctly created */
        cookie_t alarm_cookie = alarmd_event_add(eve);

        ev->alarm_cookie = alarm_cookie;
        alarm_event_delete(eve);

        /* Update record in database */
        int n, m;
        GPtrArray* result = ev_db_sql_query_prep_str(data->db_data.update_cookie, &n, &m, data, NULL, DB_INT64, alarm_cookie, DB_INT, ev->id, DB_END);
        ev_db_sql_query_clear_2(result, m);
        free(database_path);
    }
    return 0;
}

/**
 * @brief Remove event reminder
 *
 * @param ev event
 * @param data
 * @error GError
 */
int ev_mgr_reminder_remove(event* ev, AppData* data, GError** error)
{

    if (ev->alarm_cookie) {
        alarmd_event_del(ev->alarm_cookie);
    } else {
        event* ev2 = ev_mgr_new(NULL);
        ev_sel_get_by_id(ev->id, ev2, data, NULL);
        alarmd_event_del(ev2->alarm_cookie);
        ev_mgr_clear(ev2, NULL);
    }

    ev->alarm_cookie = 0;

    return 0;
}

RepeatType ev_mgr_period_to_rt(event *ev)
{
    if (!ev->per || !ev->per->type) /* No repeat */
        return REPEAT_UNSET;

    switch (ev->per->type) {
    /* Day */
    case 1:
        return REPEAT_EVERY_DAY;

    /* Week */
    case 2:
        if (ev->per->repeat == 2)
            return REPEAT_EVERY_2WEEK;

        if (ev->per->n_days == 5)
            return REPEAT_WORK_DAYS;

        if (ev->per->n_days == 3)
            return REPEAT_MON_WED_FRI;

        if (ev->per->n_days == 2)
            return REPEAT_TUE_THU;

        return REPEAT_EVERY_WEEK;

     /* Month */
     case 3:
         if (ev->per->repeat == 1)
             return REPEAT_EVERY_MONTH;

         if (ev->per->repeat == 12) /* Every year */
             return REPEAT_EVERY_YEAR;

     default:
         return REPEAT_UNSET;
    }
}

/**
 * @global Set period to given event
 *
 * @param ev struct event
 * @param type period type (1 - day, 2 - week, 3 - month, 0 - no period)
 * @param repeat how often event repeats
 * @param ... days list (must end with 0)
 *
 */
static void ev_mgr_set_period(event* ev, int type, int repeat, ...)
{
    int i;
    int arg;
    period* per = calloc(1, sizeof(period));
    per->type = type;
    per->repeat = repeat;

    if (type != 1) {
       int days[31];
       va_list arg_list;
       va_start(arg_list, repeat);

       /* Iterate day numbers */
       for (i = 0; (arg = va_arg(arg_list, int)) != 0; i++) {
           days[i] =  arg;
       }

       per->days = malloc(i * sizeof(int));
       memcpy(per->days, days, i * sizeof(int));
       per->n_days = i;

       va_end(arg_list);
    }
    per->start_time = ev->start_time; /* Start of period - start_time# - 1 day */
    per->end_time = 2147483647; /* End of period - End of Unix time_t */
    ev->per = per;
}

void ev_mgr_rt_to_period(RepeatType rt, event* ev)
{
    GDate date;
    g_date_clear(&date, 1);
    g_date_set_time_t(&date, ev->start_time);
    int weekday = g_date_get_weekday(&date);
    int monthday = g_date_get_day(&date);

    switch (rt) {
        /* Reminder doesn't set */
    case REPEAT_UNSET:
        ev->per = NULL;
        break;

        /* Repeat event every day */
    case REPEAT_EVERY_DAY:
        ev_mgr_set_period(ev, 1, 1);
        break;

        /* Repeat event from monday to friday */
    case REPEAT_WORK_DAYS:
        ev_mgr_set_period(ev, 2, 1, 1, 2, 3, 4, 5, 0);
        break;

        /* Repeat event monday, wednesday, friday */
    case REPEAT_MON_WED_FRI:
        ev_mgr_set_period(ev, 2, 1, 1, 3, 5, 0);
        break;

        /* Repead event tuesday, thursday */
    case REPEAT_TUE_THU:
        ev_mgr_set_period(ev, 2, 1, 2, 4, 0);
        break;

        /* Repeat event every weeks */
    case REPEAT_EVERY_WEEK:
        ev_mgr_set_period(ev, 2, 1, weekday, 0);
        break;

        /* Repeat event every two weeks */
    case REPEAT_EVERY_2WEEK:
        ev_mgr_set_period(ev, 2, 2, weekday, 0);
        break;

        /* Repeat event every month */
    case REPEAT_EVERY_MONTH:
        ev_mgr_set_period(ev, 3, 1, monthday, 0);
        break;

        /* Repeat event every twelve months */
    case REPEAT_EVERY_YEAR:
        ev_mgr_set_period(ev, 3, 12, monthday, 0);
        break;

    case REPEAT_OTHER:
        ev->per = NULL;
    }

    if (rt == REPEAT_MON_WED_FRI) {
        g_debug("%d\n", ev->per->n_days);
    }
}


void ev_add_contact(int event_id, const char* contact_id, AppData* data)
{
    g_debug("Adding contact %s to event %d\n", contact_id, event_id);
    GError* err = NULL;
    int n, m;
    GPtrArray* arr;
    arr = ev_db_sql_query_prep_str(data->db_data.add_contact, &n, &m, data, &err,
            DB_INT, event_id,
            DB_TEXT, contact_id,
            DB_END);
                                                                  
}

int ev_check_remove_contacts(const char* id, AppData* data)
{
    int n, m;
    //GList* list = osso_abook_aggregator_lookup(data->aggr, contact_id);
    
    EBookQuery* query = e_book_query_field_test(E_CONTACT_UID, E_BOOK_QUERY_IS, id);
    GList* list = NULL;
    gboolean ret = e_book_get_contacts(data->book, query, &list, NULL);

    e_book_query_unref(query);
    /* Contact doesn't exist */
    if (!list) {
        ev_db_sql_query_prep_str(data->db_data.remove_by_contact_id, &n, &m, data, NULL,
                DB_TEXT, id,
                DB_END);
        
        g_debug("Removing nonexistent contact: %s", id);
    return 1;
    }
    return 0;
}

GList* ev_mgr_get_select_contacts()
{
    GtkWidget *chooser;
    GList *selection;

    chooser = osso_abook_contact_chooser_new (NULL, "Choose a contact");

    gtk_dialog_run (GTK_DIALOG (chooser));
    gtk_widget_hide (chooser);

    selection = osso_abook_contact_chooser_get_selection
        (OSSO_ABOOK_CONTACT_CHOOSER (chooser));

    gtk_widget_destroy (chooser);
    return selection;
}

char* ev_mgr_get_contact_id(OssoABookContact* cont)
{
    return strdup(osso_abook_contact_get_persistent_uid(cont));
}

GList* ev_mgr_select_contact_from_event(int id, AppData* data)
{
    int i = 0;
    int m = 0;
    int n = 0;

    GList* list = NULL;

    GPtrArray* arr = NULL;
    arr = ev_db_sql_query_prep_str(data->db_data.get_contacts_by_id, &n, &m, data, NULL,
            DB_INT, id,
            DB_END);
    
    if (!arr)
        return NULL;
    
    /* Iterate result and store contacts to list */
    for (i = 0; i < n; i++) {
        char** string = g_ptr_array_index(arr, i);
        char* contact_id = string[0];
        if (!ev_check_remove_contacts(contact_id, data)) {
            list = g_list_append(list, contact_id);
        }
    }

    /* Show dialog */
    GtkWidget* chooser;
    GList* selection;

    chooser = osso_abook_contact_chooser_new (NULL, "Choose a contact");
    
    /* Get EBookView */
    size_t size = g_list_length(list);

    EBookQuery** queries = malloc(size * sizeof(EBookQuery*));
    for (i = 0; i < size; i++)
    {
        const char* id = g_list_nth_data(list, i);
        g_debug("Add contact to list: %s\n", id);
        queries[i] = e_book_query_field_test(E_CONTACT_UID, E_BOOK_QUERY_IS, id);
        
    }

    EBookQuery* query = e_book_query_or(size, queries, TRUE);

    EBookView* book_view;
    GtkWidget* contact_view;

    GError* error = NULL;
    if (!e_book_get_book_view(data->book, query, NULL, 0, &book_view, &error))
    {
        g_critical("Couldn't create book view: %s", error->message);
        g_error_free(error);
        return NULL;
    }
    e_book_view_start(book_view);
    /* Create OssoABookContactModel */
    OssoABookContactModel* contact_model = osso_abook_contact_model_new();
    osso_abook_list_store_set_book_view(OSSO_ABOOK_LIST_STORE(contact_model), book_view);

    osso_abook_contact_chooser_set_model(OSSO_ABOOK_CONTACT_CHOOSER(chooser), contact_model);
    
    gtk_dialog_run(GTK_DIALOG (chooser));
    gtk_widget_hide(chooser);

    selection = osso_abook_contact_chooser_get_selection
        (OSSO_ABOOK_CONTACT_CHOOSER(chooser));

    gtk_widget_destroy(chooser);

    e_book_query_unref(query);
    ev_db_sql_query_clear_2(arr, 1);
    free(queries);
    return selection;
}


void ev_mgr_contact_dialog_add(int event_id, AppData* data)
{
    GList* l = NULL;
    GList* list = ev_mgr_get_select_contacts();
    for (l = list; l; l = l->next) {
        const char* uid = ev_mgr_get_contact_id(l->data);
        ev_add_contact(event_id, uid, data);
    }

}

void ev_mgr_contact_dialog_delete(int event_id, AppData* data)
{
    GList* list = ev_mgr_select_contact_from_event(event_id, data);
    if (!list)
        return;
    
    int n, m;

    ev_db_sql_query_prep_str(data->db_data.remove_contact, &n, &m, data, NULL,
            DB_TEXT, ev_mgr_get_contact_id(list->data),
            DB_INT, event_id,
            DB_END);

}

void ev_mgr_contact_dialog_view(int event_id, AppData* data)
{
    GList* list = ev_mgr_select_contact_from_event(event_id, data);
    if (list) {
        GtkWidget* starter;
        GtkWidget* dialog;
        GtkWidget* content;

        /* Init starter widget */
        starter = osso_abook_touch_contact_starter_new_with_contact
            (NULL, list->data);
        /* Create starter dialog */
        dialog = gtk_dialog_new();

        /* Get content area from dialog */
        content = GTK_DIALOG(dialog)->vbox;

        gtk_container_add (GTK_CONTAINER(content), starter);
    
        gtk_widget_show_all (dialog);
    }
}
