/*
 * Copyright (C) 2008-2009 Petrozavodsk State University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
/*
 * Google Calendar Module
 * Author: Epifanov S.A.
 * Last Modification Date: 03.09.2009
*/

#include <remote_storage/protocols/google.h>

#ifdef TEST
#include "../test/CTF/src/ctf.h"
#endif
/**
 * Authentification with Google
 * @param resp_code Pointer on response code storage
 * @param Pointer on Application Data structure
 * @return Authentification token
 * */


char* protocols_google_authentication(int* resp_code, AppData* data, GError** error)
{

    char url[ARRAY_MAX_SIZE];
    char* tmp;
    char* login;
    char* passwd;
    char auth[ARRAY_MAX_SIZE];
    char sid[ARRAY_MAX_SIZE];
    char lsid[ARRAY_MAX_SIZE];

    login = NULL;
    passwd = NULL;

/* Get login and password from config */
    login = g_key_file_get_string(data->KeyFile, "Service", "GoogleLogin", NULL);
    passwd = g_key_file_get_string(data->KeyFile, "Service", "GooglePass", NULL);

/* Set Authorization Request */
    if (strlen(login) == 0){
	g_set_error(error, REM_STRG_ERR, ACCOUNT_ERROR_CODE, ACCOUNT_ERROR, NULL);
	return NULL;
    }

    sprintf(url, AUTH_URL, login, passwd, SOURCE, SERVICE);

    free(login);
    free(passwd);

/* Try to authenticate */    
#ifndef TEST
    if ((tmp = protocols_curl_get(url, data, NULL, 1, resp_code)) == NULL){
	g_set_error(error, REM_STRG_ERR, AUTH_ERROR_CODE, AUTH_ERROR, NULL);
	return NULL;
    }
#else
    tmp = strdup("lalalalla ulaulaula Auth=lololo");
    static int st = 0;
    if(st == 0)
        *resp_code = 200;
    else 
        *resp_code = 666;
    
    st++;
#endif

    if ((*resp_code) == SERV_ANSWER_OK) {
	sscanf(tmp, "%s %s %s", &sid, &lsid, &auth);
	if ((strstr(auth, "Auth=") == NULL) && strstr(auth, "auth=") == NULL) {
	    g_set_error(error, REM_STRG_ERR, AUTH_ERROR_CODE, AUTH_ERROR, NULL);
	    return NULL;
	}
	free(tmp);
	return strdup(auth);
    } else {
	g_set_error(error, REM_STRG_ERR, AUTH_ERROR_CODE, AUTH_ERROR, NULL);
	free(tmp);
	return NULL;
    }
}


/**
 * Json Format Parser
 * @param tmp Pointer on events string
 * @return Pointer on events
 * */

event* protocols_json_request_parser(char* tmp)
{
    struct json_object* entry_root;
    struct json_object* event_root;
    struct json_object* buffer;
    struct tm time;
    int remind;
    char* id_string;
    char* recurrence_string;
/*Time String Buffer for periodic event case */
    char per_start_time_string[32];
    char per_end_time_string[32];
/*Same Buffers for other cases */
    char* start_time_string;
    char* end_time_string;
    int events_num;
    int counter;
    event** events;

    entry_root = NULL;
    event_root = NULL;
    buffer = NULL;
/* Getting root of Google calendar event*/

    if ((entry_root = json_tokener_parse(tmp)) == NULL){
	return NULL;
    }
    
    if((entry_root = json_object_object_get(entry_root, JSON_FEED)) == NULL){
	return NULL;
    }
   
    if ((entry_root = json_object_object_get(entry_root, JSON_ENTRY)) == NULL){
	events_num = 0;
    } else {
    events_num = json_object_array_length(entry_root);
    }

/* Events array initialization */

    if ((events = malloc((events_num + 1) * sizeof(event*))) == NULL){
	return NULL;
    }

    for (counter = 0; counter < events_num; counter++) {
	events[counter] = ev_mgr_new(NULL);
    }

    events[events_num] = NULL;
    

/* Getting event parameters */
    for (counter = 0; counter < events_num; counter++) {

	if ((event_root = json_object_array_get_idx(entry_root, counter)) == NULL){
	    return NULL;
	} else {

/*XXX: Get Event ID - it's a Bicycle with square wheels!*/
	    if ((buffer = json_object_object_get(event_root, JSON_ID)) == NULL){
		return NULL;
	    } else {
	    buffer = json_object_object_get(buffer, JSON_T);

	    id_string = strdup(json_object_get_string(buffer));

	    if ((events[counter]->service_event_id = malloc(strlen(id_string) + strlen(JSON_GOOGLE) + 1)) == NULL){
		return NULL;
	    }

	    strcpy(events[counter]->service_event_id, JSON_GOOGLE);
	    strcat(events[counter]->service_event_id, id_string);
	    free(id_string);
	    }
/* Get Title */
	    if ((buffer = json_object_object_get(event_root, JSON_TITLE)) == NULL){
		return NULL;
	    } else {
	    buffer = json_object_object_get(buffer, JSON_T);
	    events[counter]->title = strdup(json_object_get_string(buffer));
	    }
/* Get Content */
	    if ((buffer = json_object_object_get(event_root, JSON_CONTENT)) == NULL){
		return NULL;
	    } else {
	    buffer = json_object_object_get(buffer, JSON_T);
	    events[counter]->description = strdup(json_object_get_string(buffer));
	    }
/* Get Start Time */
	    if ((buffer = json_object_object_get(event_root, JSON_GD_WHEN)) == NULL){
		/*Periodic event?*/
		if ((buffer = json_object_object_get(event_root, JSON_GD_RECURRENCE)) == NULL){

		    return NULL;

		} else {

		    buffer = json_object_object_get(buffer, JSON_T);
		    recurrence_string = strdup(json_object_get_string(buffer));
		    sscanf(recurrence_string, JSON_ICAL_RECURR_FORMAT,
				per_start_time_string, per_end_time_string);
		    strptime (per_start_time_string, JSON_GOOGLE_RECURR_TIME_FORMAT, &time);
		    events[counter]->start_time = mktime(&time);
		    strptime(per_end_time_string, JSON_GOOGLE_RECURR_TIME_FORMAT, &time);
		    events[counter]->end_time = mktime(&time);
		}

	    } else {

		buffer = json_object_array_get_idx(buffer,0);
		buffer = json_object_object_get(buffer, JSON_START_TIME);
		start_time_string = strdup(json_object_get_string(buffer));

		strptime (start_time_string, JSON_GOOGLE_TIME_FORMAT, &time);
		events[counter]->start_time = mktime(&time);
		free(start_time_string);
/* Get End Time */
		buffer = json_object_object_get(event_root, JSON_GD_WHEN);
		buffer = json_object_array_get_idx(buffer, 0);
		buffer = json_object_object_get(buffer, JSON_END_TIME);
		end_time_string = strdup(json_object_get_string(buffer));

		strptime(end_time_string, JSON_GOOGLE_TIME_FORMAT, &time);
		events[counter]->end_time = mktime(&time);
		free(end_time_string);

	    }
/* Get Event Location */
	    buffer = json_object_object_get(event_root, JSON_GD_WHERE);
	    buffer = json_object_array_get_idx(buffer, 0);
	    buffer = json_object_object_get(buffer, JSON_VALUE_STRING);
	    events[counter]->location = strdup(json_object_get_string(buffer));

/* Get Reminder settings */
	    if ((buffer = json_object_object_get(event_root, JSON_GD_WHEN)) == NULL){
		buffer = json_object_object_get(event_root, JSON_GD_REMINDER);

		    if (buffer != NULL){
			buffer = json_object_array_get_idx(buffer, 0);
			buffer = json_object_object_get(buffer, JSON_MINUTES);
			remind = json_object_get_int(buffer) * 60;
			events[counter]->remind = (time_t)remind;
		    }

	    } else {

		buffer = json_object_array_get_idx(buffer, 0);
		buffer = json_object_object_get(buffer, JSON_GD_REMINDER);

		if (buffer != NULL){
		    buffer = json_object_array_get_idx(buffer, 0);
		    buffer = json_object_object_get(buffer, JSON_MINUTES);
		    remind = json_object_get_int(buffer) * 60;
		    events[counter]->remind = (time_t)remind;
		}
	    }
/* Memo or Meeting check */
	    if (events[counter]->start_time == events[counter]->end_time){
		events[counter]->id_type = MEMO;
	    } else {
		events[counter]->id_type = MEETING;
	    }
	}
    }
    return events;
}


/**
 * Download events from google
 * @param data Pointer on application data structure
 * @param error Pointer on error handler
 * @return Pointer on events
 * */

event* protocols_google_events_download(AppData* data, GError** error)
{
 
    char* tmp = NULL;
    char* auth = NULL;
    char url[ARRAY_MAX_SIZE];
    char auth_header[ARRAY_MAX_SIZE];
    struct curl_slist* headers = NULL;
    int resp_code = NULL;
    event** events;

/* Send Authorization request */
#ifndef TEST

    if ((auth = protocols_google_authentication(&resp_code, data, error)) == NULL){
	return NULL;
    }

/* Send request to server */
    strcpy(url, EVENTS_URL);
    strcpy(auth_header, AUTH_HEADER);
    strcat(auth_header, auth);

    headers = protocols_curl_set_header(headers, auth_header);

    if ((tmp = protocols_curl_get(url, data, headers, 0, &resp_code)) == NULL){
     	g_set_error(error, REM_STRG_ERR, DOWNLOAD_ERROR_CODE, DOWNLOAD_ERROR, NULL);
	free(auth);
	return NULL;
    }

#else
    static int state = 0;
    
    switch (state) {
    case 0:
	tmp = strdup(GOOGLE_ANSW_STR_CORRECT);
	resp_code = 200;
	break;
    case 1:
	tmp = strdup(GOOGLE_ANSW_STR_INCORRECT);
	resp_code = 200;
	break;
    default:
	resp_code = 666;
    }
    state++;
#endif

/* Server Answer Analyze */
    if (resp_code != SERV_ANSWER_OK) {
	g_set_error(error, REM_STRG_ERR, DOWNLOAD_ERROR_CODE, DOWNLOAD_ERROR, NULL);
	free(auth);
	free(tmp);
	return NULL;
    }

    if ((events = protocols_json_request_parser(tmp)) == NULL){
	g_set_error(error, REM_STRG_ERR, DOWNLOAD_ERROR_CODE, DOWNLOAD_ERROR, NULL);
	free(tmp);
	free(auth);
	return NULL;
    }

#ifdef TEST

    if ((!strcmp(events[0]->title,"test_title"))
	&& (!strcmp(events[0]->description,"test_content"))
	&& (!strcmp(events[0]->location,"test_where"))){
	free(tmp);
	free(auth);
	return events;
    } else {
	free(tmp);
	free(auth);
	return NULL;

    }

#else
    free(auth);
    free(tmp);
    return events;
#endif
}


/**
 * Upload event to google
 * @param data Pointer on application data structure
 * @param error Pointer on error handler
 * @return Pointer on string, what contain a event ID at Google
 * */

int protocols_google_event_upload(AppData* data, GError** error)
{
 
    char* tmp;
    char* auth;
    char url[ARRAY_MAX_SIZE];
    char auth_header[ARRAY_MAX_SIZE];
    char content_header[ARRAY_MAX_SIZE];
    char send_data[ARRAY_MAX_SIZE];
    char* send_string;
    struct curl_slist* headers = NULL;
    int resp_code;
    struct tm* time;
    time_t time_buffer;
    char time_string_start[TIME_ARRAY_SIZE];
    char time_string_end[TIME_ARRAY_SIZE];
    struct json_object* entry_root;
    struct json_object* event_root;
    struct json_object* buffer;
    char* id_string;

    resp_code = 0;

/* Send Authorization request */
    if ((auth = protocols_google_authentication(&resp_code, data, error)) == NULL){
	return 1;
    }


/* Set URL and headers */
    strcpy(url, EVENTS_URL);
    strcpy(auth_header, AUTH_HEADER);
    strcat(auth_header, auth);

    strcpy(content_header, "Content-Type: application/atom+xml");

    headers = protocols_curl_set_header(headers, content_header);
    headers = protocols_curl_set_header(headers, auth_header);

/* Get Event Start and End time */
    
    time_buffer = data->ev_upload->start_time;
    if ((time = localtime(&time_buffer)) == NULL) {
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
        free(auth);
	return 1;
    }
    
    if (strftime(time_string_start, sizeof(time_string_start), JSON_GOOGLE_TIME_FORMAT, time) == NULL) {
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
        free(auth);
	return 1;
    }

    if (data->ev_upload->id_type == MEETING) {
	time_buffer = data->ev_upload->end_time;
	if ((time = localtime(&time_buffer)) == NULL) {
	    g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
	    free(auth);
	    return 1;
	}
    } else {
	time_buffer = data->ev_upload->start_time;
        if ((time = localtime(&time_buffer)) == NULL) {
	    g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
	    free(auth);
            return 1;
	}
	time->tm_min = time->tm_min + 1;
    }

    if (strftime(time_string_end, sizeof(time_string_end), JSON_GOOGLE_TIME_FORMAT, time) == NULL) {
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
        free(auth);
        return 1;
    }


/* Generate send data string */
    send_string = malloc(strlen(POST_MESSAGE_TEMPLATE)
			 + strlen(data->ev_upload->title)
			 + strlen(data->ev_upload->description)
			 + strlen(data->ev_upload->location)
			 + strlen(time_string_start)
			 + strlen(time_string_end) + 1);

    if (send_string == NULL){
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
        free(auth);
	return 1;
    }

    sprintf(send_string, POST_MESSAGE_TEMPLATE,
	    data->ev_upload->title, data->ev_upload->description,
	    data->ev_upload->location, time_string_start, time_string_end);

    if ((tmp = protocols_curl_post(url, send_string, data, headers, 0, &resp_code)) == NULL){
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
        free(auth);
	return 1;
    }

    free(send_string);

/* Server Answer Analyze */
    if (resp_code != SERV_ANSWER_CREATED) {
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
	free(tmp);
        free(auth);
	return 1;
    }

/* Get event Google ID from answer */
    if ((entry_root = json_tokener_parse(tmp)) == NULL){
	g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
	free(tmp);
        free(auth);
        return 1;
    } else {

	entry_root = json_object_object_get(entry_root, JSON_ENTRY);
	    
/*XXX: Get Event ID - it's a Bicycle with square wheels!*/
	if ((buffer = json_object_object_get(entry_root, JSON_ID)) == NULL){
	    g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
	    free(tmp);
	    free(auth);
	    return 1;
	} else {
	    buffer = json_object_object_get(buffer, JSON_T);
	    id_string = strdup(json_object_get_string(buffer));

	    if ((data->ev_upload->service_event_id = malloc(strlen(id_string) + strlen(JSON_GOOGLE) + 1)) == NULL){
		g_set_error(error, REM_STRG_ERR, UPLOAD_ERROR_CODE, UPLOAD_ERROR, NULL);
		free(tmp);
		free(auth);
		return 1;
	    }
	    
	    strcpy(data->ev_upload->service_event_id, JSON_GOOGLE);
	    strcat(data->ev_upload->service_event_id, id_string);
	    free(id_string);
	}
    
    }
    free(tmp);
    free(auth);
    return 0;
}
