/*
 * Copyright (C) 2008-2009 Petrozavodsk State University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
/* ========================================================================== */
/* Events Manager                                                             */
/*                                                                            */
/* Author: Ivashov K., Petrozavodsk State University                          */
/* Date  : 2/09/2009                                                         */
/* ========================================================================== */

#include <events/events.h>
#include <stdarg.h>
#define SQL_STMT_INSERT "INSERT OR REPLACE INTO "TABLE_EVENT" \
("F_ID"         ,\
 "F_ID_TYPE"    ,\
 "F_TITLE"      ,\
 "F_START_TIME" ,\
 "F_END_TIME"   ,\
 "F_DESC"       ,\
 "F_REMIND"     ,\
 "F_PER_START"  ,\
 "F_PER_END"    ,\
 "F_PER_DAYS"   ,\
 "F_PER_REPEAT" ,\
 "F_PER_TYPE"   ,\
 "F_LOCATION"   ,\
 "F_COOKIE"     ,\
 "F_SERVICE_ID") VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"

#define SQL_STMT_INSERT_NEW "INSERT OR REPLACE INTO "TABLE_EVENT" \
("F_ID_TYPE"    ,\
 "F_TITLE"      ,\
 "F_START_TIME" ,\
 "F_END_TIME"   ,\
 "F_DESC"       ,\
 "F_REMIND"     ,\
 "F_PER_START"  ,\
 "F_PER_END"    ,\
 "F_PER_DAYS"   ,\
 "F_PER_REPEAT" ,\
 "F_PER_TYPE"   ,\
 "F_LOCATION"   ,\
 "F_COOKIE"     ,\
 "F_SERVICE_ID") VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"

#define SQL_STMT_GET_LAST_ID "SELECT "F_ID" FROM "TABLE_EVENT" WHERE rowid = ?"
#define SQL_STMT_GET_ALARM "SELECT "F_COOKIE" FROM "TABLE_EVENT" WHERE "F_ID" = ?"
#define SQL_STMT_DELETE "DELETE FROM "TABLE_EVENT" WHERE "F_ID" = ?"
#define SQL_STMT_GET_PERIOD "SELECT "F_START_TIME", "F_PER_START", "F_PER_END","F_PER_DAYS", "F_PER_REPEAT", "F_PER_TYPE", "F_ID" FROM "TABLE_EVENT" "
#define SQL_STMT_GET_BY_ID "SELECT "F_ID", "F_ID_TYPE", "F_TITLE","F_START_TIME", "F_END_TIME", "F_DESC", "F_REMIND", "F_PER_START", "F_PER_END", "F_PER_DAYS", "F_PER_REPEAT", "F_PER_TYPE", "F_LOCATION", "F_COOKIE", "F_SERVICE_ID" FROM "TABLE_EVENT" WHERE "F_ID" = ?"
#define SQL_STMT_UPDATE_COOKIE "UPDATE "TABLE_EVENT" SET "F_COOKIE" = ? WHERE "F_ID" = ?"

#define SQL_STMT_ADD_CONTACT "INSERT OR REPLACE INTO "TABLE_GUEST" ("G_EVENT_ID", "G_GUEST_ID") VALUES (?, ?)"
#define SQL_STMT_GET_CONTACTS_BY_ID "SELECT "G_GUEST_ID" FROM "TABLE_GUEST" WHERE "G_EVENT_ID" = ?"
#define SQL_STMT_REMOVE_BY_CONTACT_ID "DELETE FROM "TABLE_GUEST" WHERE "G_GUEST_ID" = ?" 
#define SQL_STMT_REMOVE_BY_EVENT_ID "DELETE FROM "TABLE_GUEST" WHERE "G_EVENT_ID" = ?"
#define SQL_STMT_REMOVE_CONTACT "DELETE FROM "TABLE_GUEST" WHERE "G_GUEST_ID" = ? AND "G_EVENT_ID" = ?"
/**
 * @brief check error code after sqlite3_step
 *
 * @param stmt sqlite3_stmt
 * @param err GError, 
 * @param code return code of sqlite3_step
 * @return error code
 * @retval 0 no error
 * @retval !=0 error
 *
 */
static int ev_db_check_step_errors(sqlite3* sqlite, sqlite3_stmt* stmt, GError** err, int code)
{
    /* Query error */
    g_assert(code != SQLITE_MISUSE);
    switch (code) {
    case SQLITE_MISUSE:
        g_set_error(err, EV_DB_ERROR, EV_DB_MISUSE_ERROR, "%s", sqlite3_errmsg(sqlite));
        sqlite3_finalize(stmt);
        return -1;

    /* Database busy */
    case SQLITE_BUSY:
        g_set_error(err, EV_DB_ERROR, EV_DB_BUSY_ERROR, "%s", sqlite3_errmsg(sqlite));
        return -1;

    /* Runtime error */
    case SQLITE_ERROR:
    case SQLITE_IOERR:
    case SQLITE_READONLY:
        g_set_error(err, EV_DB_ERROR, EV_DB_OTHER_ERROR, "%s", sqlite3_errmsg(sqlite));
        return -1;
    } 
    return 0;
}

/**
 * @brief Initialize prepared statements
 * @param data AppData
 */
static void ev_db_init_create_stmt(AppData* data)
{
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_INSERT, -1, &data->db_data.insert, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_INSERT_NEW, -1, &data->db_data.insert_new, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_GET_LAST_ID, -1, &data->db_data.get_last_id, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_DELETE, -1, &data->db_data.del, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_GET_ALARM, -1, &data->db_data.get_alarm, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_GET_PERIOD, -1, &data->db_data.get_period, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_GET_BY_ID, -1, &data->db_data.get_by_id, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_UPDATE_COOKIE, -1, &data->db_data.update_cookie, NULL);


    sqlite3_prepare_v2(data->sqlite, SQL_STMT_ADD_CONTACT, -1, &data->db_data.add_contact, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_GET_CONTACTS_BY_ID, -1, &data->db_data.get_contacts_by_id, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_REMOVE_BY_CONTACT_ID, -1, &data->db_data.remove_by_contact_id, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_REMOVE_CONTACT, -1, &data->db_data.remove_contact, NULL);
    sqlite3_prepare_v2(data->sqlite, SQL_STMT_REMOVE_BY_EVENT_ID, -1, &data->db_data.remove_by_event_id, NULL);

}

/**
 * @brief Close database
 * 
 * @param appdata AppData
 * @param GError pointer
 * @return error code
 * @retval 0  no error
 * @retval !=0 error
 */
int ev_db_close(AppData* app, GError* err)
{
    g_debug("DataBase closed\n");

    sqlite3_finalize(app->db_data.insert_new);
    sqlite3_finalize(app->db_data.insert);
    sqlite3_finalize(app->db_data.get_last_id);
    sqlite3_finalize(app->db_data.get_alarm);
    sqlite3_finalize(app->db_data.del);
    sqlite3_finalize(app->db_data.get_period);
    sqlite3_finalize(app->db_data.get_by_id);
    sqlite3_finalize(app->db_data.update_cookie);
    sqlite3_finalize(app->db_data.add_contact);
    sqlite3_finalize(app->db_data.get_contacts_by_id);
    sqlite3_finalize(app->db_data.remove_by_contact_id);
    sqlite3_finalize(app->db_data.remove_contact);
    sqlite3_finalize(app->db_data.remove_by_event_id);
    //sqlite3_finalize(app->db_data.insert_new);

    sqlite3_close(app->sqlite);
}

/**  
 * @brief Init database
 * 
 * Check if database exists. Create it if not
 * Pointer to database stores to AppData
 * @param app pointer to AppData
 * @param file database file path
 * @param err GError pointer
 * @return error code
 * @retval 0 no error
 * @retval !=0 error
 */
int ev_db_init(AppData* app, const char* file, GError** err)
{
    int code = sqlite3_open(file, &app->sqlite);
    g_debug("Database opened with code %d\n", code);
    if (code != SQLITE_OK) {
        g_set_error(err, EV_DB_ERROR, EV_DB_OPEN_ERROR, "Failed to open file: %s", file);
        return -1;
    }

    /* Table events */
    sqlite3_stmt* stmt = NULL;
    code = sqlite3_prepare(app->sqlite, SQL_CHECK_EXISTS, MAX_QUERY, &stmt, NULL);
    if (!stmt) {
        sqlite3_close(app->sqlite); 
        return -1;
    }
    code = sqlite3_step(stmt); /* Check if tblEvent exists*/

    if (ev_db_check_step_errors(app->sqlite, stmt, err, code))
        return -1;

    /* tblEvent exists */
    if (code == SQLITE_ROW && !strcmp("tblEvent", sqlite3_column_text(stmt, 0))) {
        sqlite3_finalize(stmt);
    } else {
        code = sqlite3_prepare(app->sqlite, SQL_CREATE_TABLE_EVENT, MAX_QUERY, &stmt, NULL);
        g_assert(code == SQLITE_OK);
        code = sqlite3_step(stmt); /* Store table to database */

        if (ev_db_check_step_errors(app->sqlite, stmt, err, code))
            return -1;

        sqlite3_finalize(stmt);
    }

    /* Table guests */
    stmt = NULL;
    code = sqlite3_prepare(app->sqlite, SQL_CHECK_EXISTS_GUEST, MAX_QUERY, &stmt, NULL);
    if (!stmt) {
        sqlite3_close(app->sqlite);
        return -1;
    }
    code = sqlite3_step(stmt);

    if (ev_db_check_step_errors(app->sqlite, stmt, err, code))
        return -1;

        /* tblGuests exists */
    if (code == SQLITE_ROW && !strcmp(TABLE_GUEST, sqlite3_column_text(stmt, 0))) {
        sqlite3_finalize(stmt);
    } else {
        code = sqlite3_prepare(app->sqlite, SQL_CREATE_TABLE_GUEST, MAX_QUERY, &stmt, NULL);
        g_assert(code == SQLITE_OK);
        code = sqlite3_step(stmt); /* Store table to database */

        if (ev_db_check_step_errors(app->sqlite, stmt, err, code))
            return -1;

        sqlite3_finalize(stmt);
    }



    /* Prepare sqlite3_stmt`s */
    ev_db_init_create_stmt(app);
    return 0;
}

GPtrArray* ev_db_sql_query_prep_str(sqlite3_stmt* stmt, int* n, int* mm, AppData* data, GError** error, ...)
{
    /* Function cannot work without this parameters */
    if (!stmt || !data) {
        g_warning("Event data access: ev_db_sql_query_prep_str: wrong args: stmt = %p, data = %p\n", stmt, data);
        return NULL;
    }

    int cont;/* continue loop */
    int m;/* number of fields in result */
    int i;
    int j; /* field counter */

    /* Reset SQL statement */
    sqlite3_reset(stmt);
    sqlite3_clear_bindings(stmt);

    va_list arg_list;
    va_start(arg_list, error);
    int int64_val;
    int int_val;
    char* text_val;
    i = 1;
    cont = 1;
    /* For each pair (type, value) bind parameters to statement */
    while (cont) {
        DBType type = va_arg(arg_list, DBType);

        switch (type) {
        case DB_END:
            cont = 0;
            break;

        case DB_INT:
            int_val = va_arg(arg_list, int);
            sqlite3_bind_int(stmt, i++, int_val);
            break;

        case DB_INT64:
            int64_val = va_arg(arg_list, long);
            sqlite3_bind_int64(stmt, i++, int64_val);
            break;

        case DB_TEXT:
            text_val = va_arg(arg_list, char*);

            if(text_val)
                sqlite3_bind_text(stmt, i, text_val, -1, SQLITE_TRANSIENT);

            i++;
            break;

        case DB_SKIP:
            i++;
            break;

        default:
            va_end(arg_list);
            return NULL;
        }
    }

    va_end(arg_list);

    GPtrArray* arr = g_ptr_array_new();
    cont = 1;
    i = 0;
    while (cont) {
        int code = sqlite3_step(stmt);
        switch (code) {
        /* Succes reques */
        case SQLITE_ROW:
            m = sqlite3_column_count(stmt); /* get fields number */
            char** string = calloc(m, sizeof(char*)); /* Allocate memory */

            for (j = 0; j < m; j++) {
                const char* result = sqlite3_column_text(stmt, j);
                if (result) {
                    int l = strlen(result);
                    string[j] = calloc(l + 1, sizeof(char)); /* Allocate memory for cell */
                    strcpy(string[j], result); 
                } else {
                    string[j] = NULL;
                }
            }
            g_ptr_array_add(arr, string);
            i++;
            break;

        /* Query done */
        case SQLITE_DONE:
            cont = 0;
            sqlite3_reset(stmt);
            break;
        /* Error */
        default:
            ev_db_check_step_errors(data->sqlite, stmt, error, code);
            sqlite3_reset(stmt);
            ev_db_sql_query_clear_2(arr, m);
            *n = *mm = 0;
            return NULL;
        }
    }
    if(n)
        *n = i;
    if (mm)
        *mm = m;
    /* If no results found, return NULL */
    if (!arr->len) {
        g_ptr_array_free(arr, TRUE);
        return NULL;
    }
    return arr;
}


/**
 * @brief Free results array
 * 
 * @param arr array to free
 * @param m number of fields
 */
void ev_db_sql_query_clear_2(GPtrArray* arr, int m)
{
    if (arr) {
        int i, j;
        for (i = 0; i < arr->len; i++) {
            char** string = g_ptr_array_index(arr, i);
            for (j = 0; j < m; j++) {
                if (string[j])
                    free(string[j]);
            }
            free(string);
        }
        g_ptr_array_free(arr, TRUE);
    }
}
