/*
 * Copyright (C) 2008-2009 Petrozavodsk State University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#define CTF_LOG_FORMAT_USE_TIME
#define CTF_THROW_EXIT

#include "events/events.h"
#include "../CTF/src/ctf.h"
#include <ui/callbacks.h>
#include <settings/settings.h>
#include <appdata.h>
#include <time.h>

#define BUF_SIZE 2048

/**
 * @brief Функция преобразует структуру GDate в time_t
 *
 * @param d дата
 * @return дата в time_t
 */
static time_t gdate2time_t(GDate* d)
{
    struct tm t;
    g_date_to_struct_tm(d, &t);
    t.tm_isdst = 0;
    return mktime(&t);
}


GLOBAL_DECL();
CTF_TEST_DATA_FILE_NAME_DECL;
CTF_STAT_FILE_NAME_DECL;

CTF_TEST_CASE(select_event_test) {
    CTF_TEST_NAME("select_event_test");
    AppData   *data;
    data = create_data();
    int i;
    conf_init(data);

    ev_db_date_id_list_free(data);
    
    GDate* start_d = g_date_new_dmy(1, 8, 2009);
    time_t start = gdate2time_t(start_d);

    GDate* end_d = g_date_new_dmy(31, 8, 2009);
    time_t end = gdate2time_t(end_d);
   
    GDate* d1_d = g_date_new_dmy(15, 8, 2009);

    GDate* d2_d = g_date_new_dmy(16, 8, 2009);
    
    ev_sel_get_by_period(start, end, data, NULL);
    CTF_REQUIRE_MSG(data->id_list, "Cannot get events by period");

    int ok = 1;
    GDate day;
    
    for (i = 0; data->id_list[i].date > 0; i++) {
        time_t d = data->id_list[i].date;
        g_date_clear(&day, 1);
        g_date_set_time_t(&day, d);
        
        GDateWeekday wd = g_date_get_weekday(&day);
        
        if (g_date_compare(start_d, &day) > 0 || g_date_compare(&day, end_d) > 0) {
            ok = 0;
            break;
        }

        if (wd == G_DATE_MONDAY || wd == G_DATE_WEDNESDAY || wd == G_DATE_FRIDAY) {
            continue;
        }

        if(!g_date_compare(&day, d1_d) || !g_date_compare(&day, d2_d)) {
            continue;
        }

        ok = 0;
        break;
    }

    CTF_CHECK_MSG(ok, "Wrong events has got by period");
    g_date_free(d1_d);
    g_date_free(d2_d);
    g_date_free(start_d);
    g_date_free(end_d);



    ev_db_date_id_list_free(data);    
    destroy_data(data);
}

CTF_TEST_CASE(search_event_test) {
    CTF_TEST_NAME("search_event_test");
    AppData* data;
    data = create_data();
    int i;
    conf_init(data);
    ev_db_date_id_list_free(data);

    GDate* d1_d = g_date_new_dmy(16, 8, 2009);
    time_t d1 = gdate2time_t(d1_d);

    GDate* d2_d = g_date_new_dmy(22, 8, 2009);
    time_t d2 = gdate2time_t(d2_d);

    GError* err = NULL;
    ev_sel_get_by_period(d1, d2, data, &err);
    if(err) g_error_free(err);

    err = NULL;
    GPtrArray* arr = ev_sel_filter(data, data->id_list, "test", &err);
    if(err) g_error_free(err);

    CTF_CHECK_MSG(arr->len == 1, "Search event test failed");
    ev_sel_event_array_free(arr);
    
    err = NULL;
    arr = ev_sel_filter(data, data->id_list, "e", &err);
    if(err) g_error_free(err);

    CTF_CHECK_MSG(arr->len == 2, "Wrong number of events found");
    ev_sel_event_array_free(arr);

    g_date_free(d1_d);
    g_date_free(d2_d);

    destroy_data(data);
}

CTF_TEST_CASE(view_event_test) {
    CTF_TEST_NAME("view_event_test");
    AppData* data;
    data = create_data();
    int i;
    conf_init(data);
    ev_db_date_id_list_free(data);

    GError *err= NULL;
    event *ev = ev_mgr_new(&err);
    CTF_REQUIRE_MSG(!err, "Cannot create event");

    err = NULL;
    int code = ev_sel_get_by_id(1024, ev, data, &err);
    CTF_CHECK(code);

    err = NULL;
    code = ev_sel_get_by_id(6, ev, data, &err);
    CTF_REQUIRE(!code);

    CTF_CHECK_MSG(!strcmp(ev->title, "Test"), "Error view event with id 6");

    ev_mgr_clear(ev, NULL);

    destroy_data(data);
}

CTF_TEST_CASE(edit_event_test) {
    CTF_TEST_NAME("edit_event_test");
    AppData* data;
    data = create_data();
    conf_init(data);
    ev_db_date_id_list_free(data);

    GError *err= NULL;
    event *ev = ev_mgr_new(&err);
    CTF_REQUIRE_MSG(!err, "Cannot create event");

    err = NULL;
    int code = ev_sel_get_by_id(6, ev, data, &err);
    CTF_REQUIRE(!code);

    /* Изменение описания */
    const char* new_desc = "new_desc";
    char* old_desc = ev->description;
    
    ev->description = malloc(strlen(new_desc) + 1);
    strcpy(ev->description, new_desc);

    err = NULL;
    ev_mgr_store(ev, data, &err);
    CTF_CHECK_MSG(!err, "Error storing event");
    ev_mgr_clear(ev, NULL);
    
    /* Повторное получение события */    
    err= NULL;
    ev = ev_mgr_new(&err);
    CTF_REQUIRE(!err);

    err = NULL;
    code = ev_sel_get_by_id(6, ev, data, &err);
    CTF_REQUIRE(!code);
    
    CTF_CHECK_MSG(!strcmp(ev->description, new_desc), "Event stores with error");
    
    /* Запись исходного события */
    free(ev->description);
    
    ev->description = old_desc;
    ev_mgr_store(ev, data, &err);

    ev_mgr_clear(ev, NULL);

    destroy_data(data);
}

CTF_TEST_CASE(create_event_test) {
   srand(time(NULL));
   CTF_TEST_NAME("create_and_remove_event_test");
   AppData   *data;
   int       i;

   data = create_data();
   conf_init(data);

   GError *err = NULL;
   event *ev = ev_mgr_new(&err);
   if (err) g_error_free(err);

   ev->id_type = 1;

   char title[512];
   sprintf(title, "testestesfkkksd90435849%d0843985438uudiuiui" , rand());

   ev->title = malloc(strlen(title) + 1);
   strcpy(ev->title, title);

   const char *location = "tesqqqqqqqqqqqqqqqqqqqqqqqqsssssssssssssssssssssssssszzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzzztestesfkkksd904358490843985438uudiuiui";
   ev->location = malloc(strlen(location) + 1);
   strcpy(ev->location, location);

   const char *description = "43985438uudiuiui";
   ev->description = malloc(strlen(description) + 1);
   strcpy(ev->description, description);

   ev->start_time = time(NULL);
   ev->end_time = time(NULL) + 3600;

   //Запись события в бд
   err = NULL;
   ev_mgr_store(ev, data, &err);
   if (err) g_error_free(err);

   int id = ev->id;

   err = NULL;
   ev_mgr_clear(ev, &err);
   if (err) g_error_free(err);

   //Поиск события в базе данных
   err = NULL;
   ev_sel_get_by_period(time(NULL) - 3600 * 48, time(NULL) + 3600 * 48, data, &err);
   if (err) g_error_free(err);

   err = NULL;
   GPtrArray* arr = ev_sel_filter(data, data->id_list, "zzzzzzzzzzzzzzzzzzt", &err);
   if (err) g_error_free(err);


   int ok = 0;
   for (i = 0; i < arr->len; i++) {
       ev = g_ptr_array_index(arr, i);
       if (!strcmp(ev->title, title)) {
           ok = 1;
           break;
       }
   }
   CTF_REQUIRE(ok);
   CTF_CHECK_MSG(id == ev->id, "Create event error");
   ev_sel_event_array_free(arr);


   ev_db_date_id_list_free(data);
   CTF_CHECK(data->id_list == NULL);

   //Повторный поиск события в базе данных
   if (ok) {
       err = NULL;
       ev_mgr_remove(id, data, &err);
       if(err) g_error_free(err);


       err = NULL;
       ev_sel_get_by_period(time(NULL) - 3600 * 48, time(NULL) + 3600 * 48, data, &err);
       if (err) g_error_free(err);

       err = NULL;
       GPtrArray* arr =  ev_sel_filter(data, data->id_list, "zzzzzzzzzzzzzzzzzzt", &err);
       if (err) g_error_free(err);

       ok = 0;
       for (i = 0; i < arr->len; i++) {
           ev = g_ptr_array_index(arr, i);
           if (!strcmp(ev->title, title)) {
               ok = 1;
               break;
           }
       }
   }
   CTF_CHECK_MSG(!ok, "Found deleted event");

   ev_db_date_id_list_free(data);
   CTF_CHECK_MSG(data->id_list == NULL, "Free date_id_list error");

   destroy_data(data);

}

CTF_TEST_CASE(alarm_test) {
    CTF_TEST_NAME("alarm_test");
    AppData* data;
    data = create_data();
    conf_init(data);
    ev_db_date_id_list_free(data);

    event* ev = ev_mgr_new(NULL);
    ev_sel_get_by_id(1, ev, data, NULL);

    ev->start_time = time(NULL) + 5;
    ev->remind = 1;

    /* Установка напоминания  */    
    ev_mgr_reminder_add(ev, time(NULL), data, NULL);
    
    int c = ev->alarm_cookie;
    /* Проверка  */
    CTF_CHECK_MSG(ev->alarm_cookie, "Add alarm error");

    alarm_event_t* al = alarmd_event_get(c);
    CTF_CHECK_MSG(al, "Check is alarmd added successful failed");
    //alarmd_event_free(al);

    ev_mgr_reminder_remove(ev, data, NULL);
    CTF_CHECK_MSG(!alarmd_event_get(c), "Alarm doesn't removes after ev_mgr_reminder_remove()");

    ev_mgr_clear(ev, NULL);
    destroy_data(data);
}


CTF_TEST_CASE(alarm_test2) {
    CTF_TEST_NAME("alarm_test");
    AppData* data;
    data = create_data();
    conf_init(data);
    ev_db_date_id_list_free(data);

    event* ev = ev_mgr_new(NULL);
    ev_sel_get_by_id(1, ev, data, NULL);

    /* Копирование события */
    ev->id = 0;
    ev_mgr_store(ev, data, NULL);

    ev->start_time = time(NULL) + 5;

    /* Установка напоминания  */
    ev_mgr_reminder_add(ev, time(NULL), data, NULL);
    ev_mgr_store(ev, data, NULL); 
    int c = ev->alarm_cookie;
    
    /* Удаление события из базы  */
    ev_mgr_remove(ev->id, data, NULL);
    
    /* Проверка что alarm удалился */
    CTF_CHECK_MSG(!alarmd_event_get(ev->alarm_cookie), "Can't remove alarm");

    ev_mgr_clear(ev, NULL);
    destroy_data(data);
}


