/**************************************************************************
 *  Kchmviewer for Maemo5 - a CHM file viewer with broad language support *
 *  Copyright (C) 2009-2010 Max Kiva , max.kiva@gmail.com                 *
 *                                                                        *
 *  This program is free software: you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation, either version 3 of the License, or     *
 *  (at your option) any later version.                                   *
 *															                                  	      *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 **************************************************************************/
#include <iostream>
#include <QtGui>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QtGui/QApplication>
#include <qfiledialog.h>
#include <qtimer.h>

#include "mainwindow.h"
#include "pageviewwidget.h"
#include "toolbarwidget.h"
#include "contentswidget.h"

#include "fullscreen.xpm"
#include "nextpage.xpm"
#include "prevpage.xpm"

/***************************************************************************
 */
inline static size_t screenWidth()
{
  return QApplication::desktop()->width();
}

/***************************************************************************
 */
inline static size_t screenHeight()
{
  return QApplication::desktop()->height();
}

/***************************************************************************
 */
inline static size_t notificationAreaHeight()
{
  const size_t NOTIF_AREA_HEIGHT = 56;
  return NOTIF_AREA_HEIGHT;
}

/***************************************************************************
 */
inline static size_t toolbarHeight()
{
  const size_t TOOLBAR_HEIGHT = 70;
  return TOOLBAR_HEIGHT;
}

/***************************************************************************
 */
inline static size_t toolbarX()
{
  const size_t TOOLBAR_X = 0;
  return TOOLBAR_X;
}
/***************************************************************************
 */
inline static size_t toolbarY()
{
  const int TOOLBAR_Y = screenHeight() - 
                        notificationAreaHeight() - 
                        toolbarHeight(); 
  return TOOLBAR_Y;
}

MainWindow::MainWindow(QWidget *parent, Qt::WindowFlags flags) 
  : QMainWindow(parent,flags),
    pageView_( new PageViewWidget(this) ),
    toolBar_( new ToolBarWidget(this) ),
    contents_(new ContentsWidget(this) ),
    toolBarSize_( screenWidth(),toolbarHeight()),
    pageSize_( screenWidth(), toolbarY() ),
    contentsSize_( screenWidth(), toolbarY() ),
    fullScreenOff_( new QPushButton( QPixmap( fullscreen_xpm ),0,this) ),
    nextPage_( new QPushButton( QPixmap( nextpage_xpm ),0,this) ),
    prevPage_( new QPushButton( QPixmap( prevpage_xpm ),0,this) ),
    fullScreenMode_(false)
{
  //TODO: disable initially all buttons apart from open
  toolBar_->move( toolbarX(),toolbarY());
  toolBar_->resize( toolBarSize_ ); 
  toolBar_->show();

  contents_->move(0,0);
  contents_->resize( contentsSize_ );
  contents_->hide();
  connect( contents_, SIGNAL(selectedContentsItem()), this, SLOT(pickedContent()) );
  
  pageView_->move(0,notificationAreaHeight());
  pageView_->setMinimumSize( pageSize_ );
  pageView_->setMaximumSize( pageSize_ );

  connect( pageView_,SIGNAL(cursorPositionChanged()),this,SLOT(showFullScreenControls()));

  // setup the full screen off button
  fullScreenOff_->move( screenWidth()-48-4,screenHeight()-48);
  fullScreenOff_->resize(48,48);
  fullScreenOff_->setMinimumSize( 48,48 );
  fullScreenOff_->setMaximumSize( 48,48 );
  fullScreenOff_->setFlat(true);
  fullScreenOff_->hide();
  connect( fullScreenOff_,SIGNAL(clicked()), this, SLOT(doFullScreenOff()) );
  
  prevPage_->move( 4, (screenHeight() - 48 )/2 );
  prevPage_->resize(48,48);
  prevPage_->setMinimumSize( 48,48 );
  prevPage_->setMaximumSize( 48,48 );
  prevPage_->setFlat(true);
  prevPage_->hide();
  connect( prevPage_,SIGNAL(clicked()), this, SLOT(doPrevPage()) );

  nextPage_->move( screenWidth() - 48 -4, (screenHeight() - 48 )/2 );
  nextPage_->resize(48,48);
  nextPage_->setMinimumSize( 48,48 );
  nextPage_->setMaximumSize( 48,48 );
  nextPage_->setFlat(true);
  nextPage_->hide();
  connect( nextPage_,SIGNAL(clicked()), this, SLOT(doNextPage()) );

  setCentralWidget(pageView_);
  grabZoomKeys();
}

MainWindow::~MainWindow()
{
  delete pageView_;

  delete toolBar_;
  delete contents_; 
  delete fullScreenOff_;
  delete nextPage_;
  delete prevPage_;
}

void MainWindow::grabZoomKeys(bool grab)
{
  if (!winId()) {
    qWarning("Can't grab keys unless we have a window id");
    exit(1);
    return;
  }

  unsigned long val = (grab) ? 1 : 0;
  Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
  if (!atom) {
    qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
        "on a Maemo 5 device!");
    return;
  }

  XChangeProperty (QX11Info::display(),
      winId(),
      atom,
      XA_INTEGER,
      32,
      PropModeReplace,
      reinterpret_cast<unsigned char *>(&val),
      1);
}


void MainWindow::keyPressEvent(QKeyEvent* event)
{
  switch (event->key()) {
    case Qt::Key_F7:
      doZoomIn();
      event->accept();
      break;

    case Qt::Key_F8:
      doZoomOut();
      event->accept();
      break;
  }
  QWidget::keyPressEvent(event);
}



void MainWindow::openFile( const QString& fileName )
{
  if (  pageView_->loadChmFile( fileName ) )
  {
    contents_->populate( pageView_->getContents() );
    toolBar_->enableButtons();
  }
}

void MainWindow::doOpenFile()
{
#ifdef __ARM_EABI__
  QString fileName = QFileDialog::getOpenFileName( this, 
      tr("Open Chm file"), tr(getenv("/")), tr("Chm Files (*.chm)"));
#else
  QString fileName = QFileDialog::getOpenFileName( this, 
      tr("Open Chm file"), tr(getenv("HOME")), tr("Chm Files (*.chm)"));
#endif
  //TODO: check validity of the file name and if 
  //TODO: it is not the one active already
  if ( fileName.length() )
  {
    openFile( fileName );  
  }

}

void MainWindow::doOpenTableOfContents()
{
  // create contents widget
  if ( !contents_ )
  {
    contents_ = new ContentsWidget(this);
  }
  // show it 
  if ( contents_->isVisible() )
  {
    contents_->hide();
  }
  else
  {
    contents_->show();
  }
}


void MainWindow:: doFullScreenOn()
{
  if( !fullScreenMode_ )
  {
    
#ifdef STD_TRACING
    std::cout << __PRETTY_FUNCTION__ 
              << ": full screen ON"
              << std::endl;
#endif
    
    toolBar_->hide();

    pageView_->move(0,0);
    pageView_->setMinimumSize( screenWidth(),screenHeight() );
    pageView_->setMaximumSize(  screenWidth(),screenHeight() );


    showFullScreen();
    // must be set before showing controls
    fullScreenMode_ = true;
    
    showFullScreenControls();


  }
}

void MainWindow:: doFullScreenOff()
{
  if( fullScreenMode_ )
  {
#ifdef STD_TRACING
    std::cout << __PRETTY_FUNCTION__ 
              << ": full screen OFF"
              << std::endl;
#endif

    pageView_->move(0,notificationAreaHeight());
    pageView_->setMinimumSize( pageSize_ );
    pageView_->setMaximumSize( pageSize_ );
  
    hideFullScreenControls();
    
    toolBar_->show();

    showNormal();
    

    fullScreenMode_ = false;
  }
}

void MainWindow:: doPrevPage()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif  
  toolBar_->enableButtons( false );
  pageView_->prevPage();
  toolBar_->enableButtons( true );
}

void MainWindow:: doNextPage()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif  
  toolBar_->enableButtons( false );
  pageView_->nextPage();
  toolBar_->enableButtons( true );
}

void MainWindow:: doZoomIn()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif  
  pageView_->zoomIn();
}

void MainWindow:: doZoomOut()
{
#ifdef STD_TRACING
  std::cout << __PRETTY_FUNCTION__ << std::endl;
#endif
  pageView_->zoomOut();
}



void MainWindow::pickedContent()
{

  size_t row = contents_->currentRow(); 
  contents_->hide();
  toolBar_->enableButtons( false );
  pageView_->openPageNum( row );
  toolBar_->enableButtons( true );
}

void MainWindow::minimizeApplication()
{
    setWindowState(Qt::WindowMinimized);
}


void MainWindow::showFullScreenControls()
{
    if( fullScreenMode_ )
    {
      prevPage_->show();
      nextPage_->show();

      fullScreenOff_->show();

      QTimer::singleShot (5000, this, SLOT(hideFullScreenControls()) );
    }
}

void MainWindow::hideFullScreenControls()
{
    fullScreenOff_->hide();
    nextPage_->hide();
    prevPage_->hide();
}


