/**************************************************************************
 *  Kchmviewer for Maemo5 - a CHM file viewer with broad language support *
 *  Copyright (C) 2009-2010 Max Kiva , max.kiva@gmail.com                 *
 *                                                                        *
 *  This program is free software: you can redistribute it and/or modify  *
 *  it under the terms of the GNU General Public License as published by  *
 *  the Free Software Foundation, either version 3 of the License, or     *
 *  (at your option) any later version.                                   *
 *															                                  	      *
 *  This program is distributed in the hope that it will be useful,       *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *  GNU General Public License for more details.                          *
 *                                                                        *
 *  You should have received a copy of the GNU General Public License     *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 **************************************************************************/

#include <iostream>
#include <QWidget>
#include <QString>
#include <QPalette>
#include <QColor>
#include <libchmurlfactory.h>
#include "pageviewwidget.h"

using namespace std;

static QString decodeUrl( const QString &input );

/****************************************************************************
 *
 *
 */
PageViewWidget::PageViewWidget( QWidget* parent )
  : QTextBrowser( parent ),
    chm_(  new LCHMFile() ),
    tableOfContents_(),
    baseUrl_("/"),
    page_(0)
{
  QTimer::singleShot (500, this, SLOT(runAutoTest()) );
}

/****************************************************************************
 *
 *
 */
PageViewWidget::~PageViewWidget()
{
  delete chm_;
}

void PageViewWidget::openPageNum( size_t page )
{

  size_t sz = static_cast<size_t>(tableOfContents_.size());

    std::cout << __PRETTY_FUNCTION__ 
              << ": request to move from page :" 
              << page_
              << " to page :"
              << page
              << std::endl;

  if ( page && page < static_cast<size_t>(tableOfContents_.size()) )
  {
    QString url = tableOfContents_[page].urls.join("|");
    openPage( url );
    page_ = page;

    std::cout << __PRETTY_FUNCTION__ 
              << ": new page :" 
              << page_
              << std::endl;
  }
}

/****************************************************************************
 *
 *
 */
bool PageViewWidget::openPage( const QString& url )
{
  bool result = true;
  
    std::cout << __PRETTY_FUNCTION__ 
              << " : decoding url"
              << std::endl;
  QString decodedUrl = decodeUrl( url );
    std::cout << __PRETTY_FUNCTION__ 
              << " : setting source"
              << std::endl;
  
  // call into baseclass: QTextBrowsr
  setSource( decodedUrl );
  
    std::cout << __PRETTY_FUNCTION__ 
              << " : reloading"
              << std::endl;
  reload();
    std::cout << __PRETTY_FUNCTION__ 
              << " : done"
              << std::endl;

  return result;
}



static QString decodeUrl( const QString &input )
{
	QString temp;
	int len = input.length();
	for( int i = 0 ; i < len; ++i )
	{
		QChar c = input[i];
		if (c == '%' && i + 2 < len)
		{
			int a = input[++i].unicode();
			int b = input[++i].unicode();

			if (a >= '0' && a <= '9')
				a -= '0';
			else if (a >= 'a' && a <= 'f')
				a = a - 'a' + 10;
			else if (a >= 'A' && a <= 'F')
				a = a - 'A' + 10;

			if (b >= '0' && b <= '9')
				b -= '0';
			else if (b >= 'a' && b <= 'f')
				b  = b - 'a' + 10;
			else if (b >= 'A' && b <= 'F')
				b  = b - 'A' + 10;

			temp.append( (QChar)((a << 4) | b ) );
		}
		else
		{
			temp.append( c );
		}
	}

   return temp;
}


bool PageViewWidget::loadChmFile( const QString& filename )
{
  
  bool loaded = chm_->loadFile( filename );

  if ( loaded )
  {
    tableOfContents_.clear(); 
    chm_->parseTableOfContents( &tableOfContents_ );


    if ( tableOfContents_.size()  )
    {
      openPage( tableOfContents_[0].urls.join("|") );
    }
  }

  return loaded;
  
}


QVariant PageViewWidget::loadResource(int type, const QUrl & name)
{
	QString data, file, path = name.toString( QUrl::StripTrailingSlash );

//  std::cout << __PRETTY_FUNCTION__ << ":" << name.toString().toStdString() << std::endl;
	// Retreive the data from chm file

	if ( !chm_ )
		return 0;

	int pos = path.indexOf('#');
	if ( pos != -1 )
		path = path.left (pos);
	
	path = makeURLabsolute( path, false );

	// To handle a single-image pages, we need to generate the HTML page to show 
	// this image. We did it in KCHMViewWindow::handleStartPageAsImage; now we need
	// to generate the HTML page, and set it.
	if ( LCHMUrlFactory::handleFileType( path, data ) )
		return QVariant( QString( data ) );
	
	if ( type == QTextDocument::HtmlResource || type == QTextDocument::StyleSheetResource )
	{
		if ( !chm_->getFileContentAsString( &data, path ) )
			qWarning( "Needed file %s is not present in this CHM archive\n", qPrintable( path ) );

		// A "fix" (actually hack) for CHM files people sent to me. I have no idea why QTextBrowser cannot show it.
		if ( type == QTextDocument::HtmlResource )
			data.remove( "<META http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">" );

		return QVariant( QString( data ) );
	}
	else if ( type == QTextDocument::ImageResource )
	{
		QImage img;
		QByteArray buf;
		
		QString fpath = decodeUrl( path );
		
		if ( chm_->getFileContentAsBinary( &buf, fpath ) )
		{
			if ( !img.loadFromData ( (const uchar *) buf.data(), buf.size() ) )
				qWarning( "Needed file %s is not present in this CHM archive\n", qPrintable( path ) );
		}
		
		return QVariant( img );
	}
	
//	qWarning("loadResource: Unknown type %d", type);
	return QVariant();
}
//////////////////////////////////////////////////////////////////////////////////////////////////
QString PageViewWidget::makeURLabsolute ( const QString & url, bool set_as_base )
{
	QString p1, p2, newurl = url;

	if ( !LCHMUrlFactory::isRemoteURL (url, p1)
	&& !LCHMUrlFactory::isJavascriptURL (url)
	&& !LCHMUrlFactory::isNewChmURL (url, p1, p2) )
	{
		newurl = QDir::cleanPath (url);

		// Normalize url, so it becomes absolute
		if ( newurl[0] != '/' )
			newurl = baseUrl_ + "/" + newurl;
	
		newurl = QDir::cleanPath (newurl);

		if ( set_as_base )
		{
			baseUrl_ = newurl;
		
			// and set up new baseurl
			int i = newurl.lastIndexOf('/');
			if ( i != -1 )
				baseUrl_ = QDir::cleanPath (newurl.left (i + 1));
		}
	}

	//qDebug ("ViewWindow::makeURLabsolute (%s) -> (%s)", url.ascii(), newurl.ascii());
	return newurl;
}


void PageViewWidget::nextPage()
{
  size_t pg = page_+1;
  openPageNum( pg );
}

void PageViewWidget::prevPage()
{
  size_t pg = page_-1;
  openPageNum( pg );
}

void PageViewWidget::zoomOut()
{
  QTextBrowser::zoomOut();
}

void PageViewWidget::zoomIn()
{
  QTextBrowser::zoomIn();
}






