/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Selector.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Value.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Slider.h>
#include <Lum/Table.h>
#include <Lum/TextValue.h>

#include "TeeEdit.h"

class TeasModelPainter : public Lum::TableCellPainter
{
public:
  void Draw(Lum::OS::DrawInfo* draw,
            int x, int y, size_t width, size_t height) const
  {
    const Tea tea=dynamic_cast<const TeasModel*>(GetModel())->GetEntry(GetRow());

    //
    // Name of Tea
    //

    DrawStringLeftAligned(draw,
                          x,y,width,height,
                          tea.name);

    //
    //
    // Time of tea

    std::wstring time=Lum::Base::NumberToWString(tea.minutes);

    time.append(L":");
    if (tea.seconds<10) {
      time.append(L"0");
    }
    time.append(Lum::Base::NumberToWString(tea.seconds));

    DrawStringRightAligned(draw,
                           x,y,width,height,
                           time);
  }
};

Selector::Selector(Lum::Model::Number* seconds)
 : seconds(seconds),
   list(new TeasModel(teas)),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   editAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action()),
   okAction(new Lum::Model::Action()),
   success(false)
{
  seconds->Push();

  editAction->Disable();
  removeAction->Disable();

  Observe(selection);

  Observe(addAction);
  Observe(editAction);
  Observe(removeAction);

  Observe(GetClosedAction());
  Observe(okAction);

  okAction->Disable();

  list->SetEmptyText(_(L"TBL_TEA_EMPTY",L"(No teas defined)"));
}

Selector::~Selector()
{
  seconds->Pop();
}

Lum::Object* Selector::GetContent()
{
  Lum::Panel            *panel,*horiz;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  panel=Lum::VPanel::Create(true,true);

  horiz=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"TBL_TEA_HEADER_TEA",L"Tea"),Lum::Base::Size::stdCharWidth,30,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetShowHeader(false);
  table->GetTableView()->SetAutoFitColumns(true);
  table->GetTableView()->SetAutoHSize(true);
  table->SetModel(list);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(okAction);
  table->SetPainter(new TeasModelPainter());
  horiz->Add(table);

  horiz->AddSpace();

  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(_(L"BTN_ADD",L"_Add"),addAction,true,false))
             ->Add(Lum::Button::Create(_(L"BTN_EDIT",L"_Edit"),editAction,true,false))
             ->Add(Lum::Button::Create(_(L"BTN_REMOVE",L"_Remove"),removeAction,true,false)));

  panel->Add(horiz);

  return panel;
}

void Selector::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
}

void Selector::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==selection &&
           dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      editAction->Enable();
      removeAction->Enable();
      okAction->Enable();

      seconds->Set(teas[selection->GetLine()-1].minutes*60+
                   teas[selection->GetLine()-1].seconds);
    }
    else {
      editAction->Disable();
      removeAction->Disable();
      okAction->Disable();
    }
  }
  else if (model==addAction &&
           addAction->IsFinished()) {
    Lum::Model::StringRef name=new Lum::Model::String(_(L"TEA_NAME_DEFAULT",L"Some tea"));
    Lum::Model::NumberRef time=new Lum::Model::Number(4ul*60ul);

    TeeEdit *teeEdit=new TeeEdit(name,time);

    teeEdit->SetParent(this);
    if (teeEdit->Open()) {
      teeEdit->EventLoop();
      teeEdit->Close();
    }

    if (teeEdit->Success()) {
      Tea tea;

      // The tea list
      tea.name=name->Get();
      tea.minutes=time->GetUnsignedLong()/60;
      tea.seconds=time->GetUnsignedLong()%60;

      // The selection list
      list->Append(tea);

      configurationChanged=true;
    }

    delete teeEdit;
  }
  else if (model==editAction &&
           editAction->IsFinished()) {
    assert(selection->HasSelection());

    Lum::Model::StringRef name=new Lum::Model::String(teas[selection->GetLine()-1].name);
    Lum::Model::ULongRef  time=new Lum::Model::ULong(teas[selection->GetLine()-1].minutes*60+
                                                     teas[selection->GetLine()-1].seconds);

    TeeEdit *teeEdit=new TeeEdit(name,time);

    teeEdit->SetParent(this);
    if (teeEdit->Open()) {
      teeEdit->EventLoop();
      teeEdit->Close();
    }

    if (teeEdit->Success()) {
      teas[selection->GetLine()-1].name=name->Get();
      teas[selection->GetLine()-1].minutes=time->Get()/60;
      teas[selection->GetLine()-1].seconds=time->Get()%60;

      seconds->Set(teas[selection->GetLine()-1].minutes*60+
                   teas[selection->GetLine()-1].seconds);

      list->RedrawRow(selection->GetLine());

      configurationChanged=true;
    }

    delete teeEdit;
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    assert(selection->HasSelection());

    list->Delete(selection->GetLine());

    configurationChanged=true;
  }
  else if (model==okAction && okAction->IsFinished()) {
    success=true;
    seconds->Save();
    Exit();
  }

  Lum::Dlg::ActionDialog::Resync(model,msg);
}

bool Selector::Success() const
{
  return success;
}

