/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/Audio/PlaySound.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Integer.h>

#include <Lum/OS/Main.h>

#include <Lum/OS/Manager/Repository.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>

#include "Configuration.h"
#include "Selector.h"
#include "StopWatch.h"

#include "config.h"

static Lum::Def::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  enum State {
    stateUninitialized,
    stateStoped,
    stateRunning,
    stateAlerting
  };

private:
  Lum::Model::ActionRef aboutAction;
  Lum::Model::ActionRef startAction;
  Lum::Model::ActionRef restartAction;
  Lum::Model::ActionRef stopAction;
  Lum::Model::ActionRef timer;
  Lum::Model::SizeTRef  seconds;
  size_t                lastSeconds;

  State                 state;
  Lum::Base::SystemTime alertTime;

  StopWatch             *value;
  Lum::Audio::PlaySound *beepSound;

public:
  MainDialog()
  : aboutAction(new Lum::Model::Action()),
    startAction(new Lum::Model::Action()),
    restartAction(new Lum::Model::Action()),
    stopAction(new Lum::Model::Action()),
    timer(new Lum::Model::Action()),
    seconds(new Lum::Model::SizeT()),
    lastSeconds(0),
    state(stateUninitialized),
    value(new StopWatch()),
    beepSound(NULL)
  {
    Lum::Base::Path soundPath;

    soundPath.SetNativeDir(Lum::Base::Path::GetCWD());
    soundPath.AppendDir(L"data");
    soundPath.AppendDir(L"sounds");
    soundPath.SetBaseName(L"alarm.wav");

#if defined(APP_DATADIR)
    if (!soundPath.Exists()) {
      soundPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
      soundPath.AppendDir(L"sounds");
      soundPath.SetBaseName(L"alarm.wav");
    }
#endif

    if (soundPath.Exists()) {
      beepSound=Lum::Audio::GetPlaySound();
      if (beepSound!=NULL) {
        beepSound->SetFilename(soundPath.GetPath(),Lum::Audio::PlaySound::typeWav);
      }
    }

    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    Observe(seconds);

    Observe(value->GetClickAction());

    Observe(aboutAction);
    Observe(startAction);
    Observe(restartAction);
    Observe(stopAction);
    Observe(timer);

    // Settings for initial state
    value->GetClickAction()->Enable();
    startAction->Disable();
    restartAction->Disable();
    stopAction->Disable();
  }

  ~MainDialog()
  {
    delete beepSound;
  }

  void SetValue()
  {
    if (seconds->IsNull()) {
      value->SetText(L"-:--");
    }
    else {
      std::wstring text;

      text=Lum::Base::NumberToWString(seconds->Get()/60);
      text.append(L":");
      if (seconds->Get()%60<10) {
        text.append(L"0");
      }
      text.append(Lum::Base::NumberToWString(seconds->Get()%60));

      value->SetText(text);
    }
  }

  void PlayAlert()
  {
    if (beepSound!=NULL) {
      beepSound->Play();
    }
    else {
      Lum::OS::display->Beep();
    }
  }

  void AllowScreenBlanking(bool allow)
  {
    Lum::OS::Manager::DisplayManager *displayManager=Lum::OS::Manager::repository->GetDisplayManager();

    if (displayManager!=NULL) {
      displayManager->AllowScreenBlanking(allow);
    }
  }

  void SetState(State state)
  {
    if (this->state==state) {
      return;
    }

    this->state=state;

    // Right

    switch (state) {
    case stateUninitialized:
      assert(false);
      break;
    case stateStoped:
      startAction->Enable();
      restartAction->Enable();
      stopAction->Disable();
      value->GetClickAction()->Enable();
      AllowScreenBlanking(true);
      break;
    case stateRunning:
      restartAction->Enable();
      value->GetClickAction()->Disable();
      AllowScreenBlanking(false);
    case stateAlerting:
      startAction->Enable();
      restartAction->Enable();
      stopAction->Enable();
      value->GetClickAction()->Disable();
      AllowScreenBlanking(false);
      break;
    }

    // Action

    switch (state) {
    case stateUninitialized:
      assert(false);
      break;
    case stateStoped:
      value->SetAlarm(false);
      Lum::OS::display->RemoveTimer(timer);
      break;
    case stateRunning:
      value->SetAlarm(false);
      lastSeconds=seconds->Get();
      alertTime.SetToNow();
      alertTime.Add(seconds->Get());
      Lum::OS::display->AddTimer(0,100000,timer);
      break;
    case stateAlerting:
      value->SetAlarm(true);
      seconds->Set(0);
      PlayAlert();
      alertTime.SetToNow();
      alertTime.Add(3);
      Lum::OS::display->RemoveTimer(timer);
      Lum::OS::display->AddTimer(0,100000,timer);
      break;
    }
  }

  virtual void PreInit()
  {
    value->SetFlex(true,true);

    SetMain(Lum::VPanel::Create(true,true)
            ->Add(value)
            ->AddSpace()
            ->Add(Lum::ButtonRow::Create(true,false)
                  ->Add(Lum::Button::Create(_(L"BUTTON_START",L"_Start"),startAction,true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_RESTART",L"_Restart"),restartAction,true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_STOP",L"Sto_p"),stopAction,true,true))));

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->ActionSettings(value->GetClickAction())
      ->End()
      ->GroupHelp()
        //->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      LoadConfig();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (configurationChanged && !SaveConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              _(L"ERR_CONFIG_SAVE_CAPTION",
                                L"Error while storing configuration..."),
                              _(L"ERR_CONFIG_SAVE_TEXT",
                                L"There was an error while storing the configuration!"));
      }
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==value->GetClickAction() &&
             value->GetClickAction()->IsFinished() &&
             (state==stateStoped || state==stateUninitialized)) {
      Selector *selector;

      selector=new Selector(seconds);
      selector->SetParent(GetWindow());
      if (selector->Open()) {
        selector->EventLoop();
        selector->Close();
      }

      delete selector;

      SetState(stateStoped);
    }
    else if (model==startAction &&
             startAction->IsFinished()) {
      SetState(stateRunning);
    }
    else if (model==restartAction &&
             restartAction->IsFinished()) {
      SetState(stateStoped);
      seconds->Set(lastSeconds);
      SetState(stateRunning);
    }
    else if (model==stopAction &&
             stopAction->IsFinished()) {
      SetState(stateStoped);
    }
    else if (model==timer &&
             timer->IsFinished()) {
      if (state==stateRunning) {
        Lum::Base::SystemTime now;

        if (now>=alertTime) {
          SetState(stateAlerting);
        }
        else {
          seconds->Set((alertTime.GetTime()-now.GetTime()));
          Lum::OS::display->AddTimer(0,100000,timer);
        }
      }
      else if (state==stateAlerting) {
        Lum::Base::SystemTime now;

        if (now.GetTime()>alertTime) {
          PlayAlert();
          alertTime.SetToNow();
          alertTime.Add(3);
        }

        Lum::OS::display->AddTimer(0,100000,timer);
      }
    }
    else if (model==seconds) {
      SetValue();
    }

    Dialog::Resync(model,msg);
  }

};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"A tea steeping timer"));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    FreeConfig();

    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,L"Kandis")
