/*

Kana of The Day -- Japanese kana learning application for Maemo Fremantle.
Copyright (C) 2012 M. Hossein Azadmanesh (mhazadmanesh2009@gmail.com)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/

#include "MainWindow.h"

#include "AboutDialog.h"
#include "HelpDialog.h"
#include "OptionsDialog.h"

#include "../libkanaoftheday/kanadb.h"
#include "../libkanaoftheday/KanaOTDLib.h"

#include <QCoreApplication>
#include <QDateTime>
#include <QDir>
#include <QKeyEvent>
#include <QFile>
#include <QFileInfo>
#include <QStringList>
#include <QTextStream>

#include <QDBusConnection>
#include <QDBusMessage>

#include <QAction>
#include <QBoxLayout>
#include <QComboBox>
#include <QLabel>
#include <QMenu>
#include <QMenuBar>
#include <QMessageBox>
#include <QPushButton>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5InformationBox>
#endif

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setWindowTitle("Kana of The Day");
    seed_random();

    bool fileError = false;
    if (ReadPermFromFile(current_kana, current_kana_set, current_permtab, fileError) == false)
    {
        HelpDialog* dlg = new HelpDialog(true, this);
        dlg->exec();
        delete dlg;
    }
    else if (fileError)
    {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, "\n<b>Error!</b>\n\nInvalid configuration file.\n"
                                           "Defaults loaded.\n", QMaemo5InformationBox::NoTimeout);
#else
        QMessageBox::warning(this, "Error", "Invalid configuration file.\nDefaults loaded.");
#endif
    }

    CreateMenus();
    CreateUI();
    orientationPreviouslyWas= 0;

    mouseStartX = 0;
    swipeSensitivity = landscapeSwipeSensitivity;

    ShowKana();
}

MainWindow::~MainWindow()
{

}

void MainWindow::setOrientation(ScreenOrientation orientation)
{
    Qt::WidgetAttribute attribute;
    switch (orientation) {
#if QT_VERSION < 0x040702
    // Qt < 4.7.2 does not yet have the Qt::WA_*Orientation attributes
    case ScreenOrientationLockPortrait:
        attribute = static_cast<Qt::WidgetAttribute>(128);
        break;
    case ScreenOrientationLockLandscape:
        attribute = static_cast<Qt::WidgetAttribute>(129);
        break;
    default:
    case ScreenOrientationAuto:
        attribute = static_cast<Qt::WidgetAttribute>(130);
        break;
#else // QT_VERSION < 0x040702
    case ScreenOrientationLockPortrait:
        attribute = Qt::WA_LockPortraitOrientation;
        break;
    case ScreenOrientationLockLandscape:
        attribute = Qt::WA_LockLandscapeOrientation;
        break;
    default:
    case ScreenOrientationAuto:
        attribute = Qt::WA_AutoOrientation;
        break;
#endif // QT_VERSION < 0x040702
    };
    setAttribute(attribute, true);

    //NOTE: Calling `RelayoutUI();` here crashed!
}

void MainWindow::showExpanded()
{
#if defined(Q_OS_SYMBIAN) || defined(Q_WS_SIMULATOR)
    showFullScreen();
#elif defined(Q_WS_MAEMO_5)
    showMaximized();
#else
    show();
#endif
}

void MainWindow::CreateMenus()
{
    QMenu* mainMenu = menuBar()->addMenu("Kana of The Day");
    mainMenu->addAction("Choose Kana Set...", this, SLOT(ShowKanaSetDialog()));
    mainMenu->addAction("Randomize Kana Order", this, SLOT(RandomizeKanaPermTab()));
    mainMenu->addAction("Help", this, SLOT(ShowHelp()));
    mainMenu->addAction("About", this, SLOT(ShowAbout()));
}

void MainWindow::CreateUI()
{
    lbl_kset = new QLabel("Kana Set: <b>" + QString(get_kana_set_name(current_kana_set)) + "</b>");

    QLabel* lbl_id_roma = new QLabel("Romaji:");
    lbl_roma = new QLabel("Wow!");
    btn_hide = new QPushButton("x");
    btn_hide->setFixedSize(50, 50);
    connect(btn_hide, SIGNAL(clicked()), this, SLOT(HideRomaji()));
    btn_reveal = new QPushButton("Reveal");
    btn_reveal->setFixedHeight(50);
    btn_reveal->setVisible(false);
    connect(btn_reveal, SIGNAL(clicked()), this, SLOT(ShowRomaji()));
    layh_roma = new QHBoxLayout();
    layh_roma->addStretch(1);
    layh_roma->addWidget(lbl_id_roma);
    layh_roma->addWidget(lbl_roma);
    layh_roma->addWidget(btn_hide);
    layh_roma->addWidget(btn_reveal);
    layh_roma->addStretch(1);

    QLabel* lbl_id_hira = new QLabel("Hiragana:");
    lbl_hira = new QLabel(QString::fromUtf8("ざぴ"));
    layv_hira = new QVBoxLayout();
    layv_hira->addWidget(lbl_id_hira, 0);
    layv_hira->addWidget(lbl_hira, 1);

    QLabel* lbl_id_kata = new QLabel("Katakana:");
    lbl_kata = new QLabel(QString::fromUtf8("がざ"));
    layv_kata = new QVBoxLayout();
    layv_kata->addWidget(lbl_id_kata, 0);
    layv_kata->addWidget(lbl_kata, 1);

    QPushButton* btn_prev = new QPushButton("<< Previous Kana");
    connect(btn_prev, SIGNAL(clicked()), this, SLOT(PrevKana()));
    QPushButton* btn_next = new QPushButton("Next Kana >>");
    connect(btn_next, SIGNAL(clicked()), this, SLOT(NextKana()));
    layh_btns = new QHBoxLayout();
    layh_btns->addWidget(btn_prev);
    layh_btns->addWidget(btn_next);

    lbl_kset->   setAlignment(Qt::AlignCenter);
    lbl_id_roma->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
    lbl_roma->   setAlignment(Qt::AlignLeft  | Qt::AlignVCenter);
    lbl_id_hira->setAlignment(Qt::AlignCenter);
    lbl_hira->   setAlignment(Qt::AlignCenter);
    lbl_id_kata->setAlignment(Qt::AlignCenter);
    lbl_kata->   setAlignment(Qt::AlignCenter);

    QFont romaFont = lbl_roma->font();
    romaFont.setBold(true);
    romaFont.setPixelSize(36);
    lbl_roma->setFont(romaFont);

    /*lbl_id_roma->setStyleSheet("background: yellow;");
    lbl_roma->setStyleSheet(lbl_id_roma->styleSheet());
    lbl_id_hira->setStyleSheet(lbl_id_roma->styleSheet());
    lbl_hira->setStyleSheet(lbl_id_roma->styleSheet());
    lbl_id_kata->setStyleSheet(lbl_id_roma->styleSheet());
    lbl_kata->setStyleSheet(lbl_id_roma->styleSheet());*/
}

void MainWindow::RelayoutUI()
{
    /*lbl_id_hira->setText(QString("*") + lbl_hira->text() + QString::number(size().width()) +
                      " " + QString::number(size().height()) + "\n");*/

    QVBoxLayout* mainLayout = new QVBoxLayout();

    //Crashes if we don't do this
    layh_roma->setParent(NULL);
    layv_hira->setParent(NULL);
    layv_kata->setParent(NULL);
    layh_btns->setParent(NULL);

    if (size().width() > size().height())
    {
        //Landscape
        setWindowTitle("Kana of The Day");
        swipeSensitivity = landscapeSwipeSensitivity;

        QFont cjkBigFont = lbl_hira->font();
        cjkBigFont.setPixelSize(160);
        lbl_hira->setFont(cjkBigFont);
        lbl_kata->setFont(cjkBigFont);

        QHBoxLayout* roma_kset_layout = new QHBoxLayout();
        roma_kset_layout->addLayout(layh_roma, 1);
        roma_kset_layout->addWidget(lbl_kset, 1);

        QHBoxLayout* kanas_layout = new QHBoxLayout();
        kanas_layout->addLayout(layv_hira);
        kanas_layout->addLayout(layv_kata);

        mainLayout->addLayout(roma_kset_layout);
        mainLayout->addStretch(1);
        mainLayout->addLayout(kanas_layout, 1);
        mainLayout->addStretch(1);
        mainLayout->addLayout(layh_btns);
    }
    else
    {
        //Portrait
        setWindowTitle("Kana OTD");
        swipeSensitivity = portraitSwipeSensitivity;

        QFont cjkBigFont = lbl_hira->font();
        cjkBigFont.setPixelSize(200);
        lbl_hira->setFont(cjkBigFont);
        lbl_kata->setFont(cjkBigFont);

        mainLayout->addLayout(layh_roma);
        mainLayout->addStretch(1);
        mainLayout->addLayout(layv_hira, 1);
        mainLayout->addStretch(1);
        mainLayout->addLayout(layv_kata, 1);
        mainLayout->addStretch(1);
        mainLayout->addWidget(lbl_kset);
        mainLayout->addStretch(1);
        mainLayout->addLayout(layh_btns);
    }

    QWidget* toBeCentralWidget = new QWidget(this);
    toBeCentralWidget->setLayout(mainLayout);
    setCentralWidget(toBeCentralWidget);
}

void MainWindow::resizeEvent(QResizeEvent*)
{
    bool isLandscape = size().width() > size().height();
    if ((isLandscape && orientationPreviouslyWas == 1) || (!isLandscape && orientationPreviouslyWas == 2))
        return;

    orientationPreviouslyWas = isLandscape ? 1 : 2;
    RelayoutUI();
}

void MainWindow::closeEvent(QCloseEvent* event)
{
    WritePermToFile(current_kana, current_kana_set, current_permtab);

    //Sending the kana index is just for convenience of the home widget, so it doesn't have to
    //  open the file and look again to see what kana it should show.
    //Also read the notes at `MainWindow::ShowKana` to know why we don't send this message there.
    int todayKanaIndex = TodayKanaIndex(current_kana, current_kana_set, current_permtab);
    QDBusMessage msg = QDBusMessage::createSignal("/org/kanaoftheday/kanaotd", "org.kanaoftheday.kanaotd", "PropagateKana");
    msg << todayKanaIndex;
    QDBusConnection::sessionBus().send(msg);

    //todo: These not needed for D-BUS, eh? What about top_window?
    //hide(); //For D-BUS
    //event->ignore(); //For D-BUS
}

void MainWindow::keyReleaseEvent(QKeyEvent* event)
{
    //Note: keyPressEvent did not catch Left and Right key events, but it did catch others.
    if (event->key() == Qt::Key_Left)
        PrevKana();
    else if (event->key() == Qt::Key_Right)
        NextKana();
    else
        event->ignore();
}

void MainWindow::mousePressEvent(QMouseEvent *event)
{
    if (event->button() == Qt::LeftButton)
    {
        swipeAction = 0;
        mouseStartX = event->x();
        initialBtnPrevWidth = layh_btns->itemAt(0)->widget()->width();
        initialBtnNextWidth = layh_btns->itemAt(1)->widget()->width();
    }
    else
        event->ignore();
}

void MainWindow::mouseReleaseEvent(QMouseEvent *event)
{
    if (event->button() == Qt::LeftButton)
    {
        mouseStartX = -1;

        QPushButton* btnPrev = qobject_cast<QPushButton*>(layh_btns->itemAt(0)->widget());
        QPushButton* btnNext = qobject_cast<QPushButton*>(layh_btns->itemAt(1)->widget());

        btnPrev->setCheckable(false);
        btnPrev->setChecked(false);
        btnNext->setCheckable(false);
        btnNext->setChecked(false);

        //Note: This is the trick to set to their default sizes. :)
        btnPrev->setMinimumWidth(0);
        btnPrev->setMaximumWidth(QWIDGETSIZE_MAX);
        btnNext->setMinimumWidth(0);
        btnNext->setMaximumWidth(QWIDGETSIZE_MAX);

        if (swipeAction == -1)
            PrevKana();
        else if (swipeAction == 1)
            NextKana();
    }
    else
        event->ignore();
}

//todo: the 'flicker' while resizing in PORTRAIT mode.

void MainWindow::mouseMoveEvent(QMouseEvent *event)
{
    if (!(event->buttons() & Qt::LeftButton) || mouseStartX == -1)
    {
        event->ignore();
        return;
    }

    QPushButton* btnPrev = qobject_cast<QPushButton*>(layh_btns->itemAt(0)->widget());
    QPushButton* btnNext = qobject_cast<QPushButton*>(layh_btns->itemAt(1)->widget());

    int diff = event->x() - mouseStartX;
    if (-10 <= diff && diff <= 10)
    {
        //Neutral area
        swipeAction = 0;

        btnPrev->setCheckable(false);
        btnPrev->setChecked(false);
        btnNext->setCheckable(false);
        btnNext->setChecked(false);

        //Note: This is the trick to set to their default sizes. :)
        btnPrev->setMinimumWidth(0);
        btnPrev->setMaximumWidth(QWIDGETSIZE_MAX);
        btnNext->setMinimumWidth(0);
        btnNext->setMaximumWidth(QWIDGETSIZE_MAX);
    }
    else if (10 < diff && diff < swipeSensitivity)
    {
        //Previous Kana, moving
        swipeAction = 0;
        if (btnPrev->isCheckable())
        {
            btnPrev->setCheckable(false);
            btnPrev->setChecked(false);
        }
        btnPrev->setFixedWidth(initialBtnPrevWidth + diff * initialBtnPrevWidth / swipeSensitivity);
        btnNext->setFixedWidth(initialBtnNextWidth - diff * initialBtnNextWidth / swipeSensitivity);
    }
    else if (diff >= swipeSensitivity)
    {
        //Previous Kana, highlight the button it
        swipeAction = -1;
        btnPrev->setCheckable(true);
        btnPrev->setChecked(true);
        btnPrev->setFixedWidth(2 * initialBtnPrevWidth); //Fill available space
        btnNext->setFixedWidth(0);
    }
    if (-10 > diff && diff > -swipeSensitivity)
    {
        //Next Kana, moving
        swipeAction = 0;
        if (btnNext->isCheckable())
        {
            btnNext->setCheckable(false);
            btnNext->setChecked(false);
        }
        btnPrev->setFixedWidth(initialBtnPrevWidth + diff * initialBtnPrevWidth / swipeSensitivity);
        btnNext->setFixedWidth(initialBtnNextWidth - diff * initialBtnNextWidth / swipeSensitivity);
    }
    else if (diff <= -swipeSensitivity)
    {
        //Next Kana, highlight the button it
        swipeAction = 1;
        btnNext->setCheckable(true);
        btnNext->setChecked(true);
        btnPrev->setFixedWidth(0);
        btnNext->setFixedWidth(2 * initialBtnPrevWidth); //Fill available space
    }

}

void MainWindow::ShowKanaSetDialog()
{
    OptionsDialog* dlg = new OptionsDialog(this);
    dlg->cbo_kset->setCurrentIndex(current_kana_set);
    if (dlg->exec() == QDialog::Accepted)
    {
        current_kana = 0;
        current_kana_set = dlg->cbo_kset->currentIndex();
        delete[] current_permtab;
        generate_kanaset(current_kana_set, current_permtab);

        lbl_kset->setText("Kana Set: <b>" + QString(get_kana_set_name(current_kana_set)) + "</b>");
        ShowKana();
    }
    delete dlg;
}

void MainWindow::RandomizeKanaPermTab()
{
    generate_permutation(current_kana_set, current_permtab);
    ShowKana();
}

void MainWindow::ShowHelp()
{
    HelpDialog* dlg = new HelpDialog(false, this);
    dlg->exec();
    delete dlg;
}

void MainWindow::ShowAbout()
{
    AboutDialog* dlg= new AboutDialog(this);
    dlg->exec();
    delete dlg;
}

void MainWindow::HideRomaji()
{
    lbl_roma->setVisible(false);
    btn_hide->setVisible(false);
    btn_reveal->setVisible(true);
}

void MainWindow::ShowRomaji()
{
    lbl_roma->setVisible(true);
    btn_hide->setVisible(true);
    btn_reveal->setVisible(false);
}

void MainWindow::ShowKana()
{
    int todayKanaIndex = TodayKanaIndex(current_kana, current_kana_set, current_permtab);
    lbl_roma->setText(QString::fromUtf8(get_kana_sign(todayKanaIndex, ROMAJI)));
    lbl_hira->setText(QString::fromUtf8(get_kana_sign(todayKanaIndex, HIRAGANA)));
    lbl_kata->setText(QString::fromUtf8(get_kana_sign(todayKanaIndex, KATAKANA)));

    //NOTE: If we want to show the kana here, we should also save the changes to the config file,
    //      since home widget refreshes every two hours and might peek the old setting from the file
    //      if we leave the file unchanged.
}

void MainWindow::NextKana()
{
    current_kana = (current_kana + 1) % get_kana_set_len(current_kana_set);
    ShowKana();
}

void MainWindow::PrevKana()
{
    current_kana -= 1;
    if (current_kana == -1)
        current_kana = get_kana_set_len(current_kana_set) - 1;
    ShowKana();
}

int MainWindow::top_application()
{
    show();
    return 0;
}
