/*

Kana of The Day -- Japanese kana learning application for Maemo Fremantle.
Copyright (C) 2012 M. Hossein Azadmanesh (mhazadmanesh2009@gmail.com)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/

#include "KanaOTDHomeWidget.h"

#include "../libkanaoftheday/KanaOTDLib.h"
#include "../libkanaoftheday/kanadb.h"

#include <QBoxLayout>
#include <QDBusConnection>
#include <QDBusMessage>
#include <QFontMetrics>
#include <QImageReader>
#include <QLabel>
#include <QMouseEvent>
#include <QPainter>
#include <QTimer>

//#include <QDebug>
//#include <QFile>

KanaOTDHomeWidget::KanaOTDHomeWidget(bool is_horizontal, QWidget *parent) :
    QWidget(parent)
{
    mousePressed = false;
    horizontal = is_horizontal;
    loadingAnim = false;
    loadingAnimTimer = NULL;

    lbl_roma = new QLabel();
    lbl_hira = new QLabel();
    lbl_kata = new QLabel();

    lbl_roma->setAlignment(Qt::AlignCenter);
    lbl_hira->setAlignment(Qt::AlignCenter);
    lbl_kata->setAlignment(Qt::AlignCenter);

    QFont romaFont = lbl_roma->font();
    romaFont.setBold(true);
    romaFont.setPixelSize(36);
    lbl_roma->setFont(romaFont);

    QFont cjkBigFont = lbl_hira->font();
    cjkBigFont.setPixelSize(80);
    lbl_hira->setFont(cjkBigFont);
    lbl_kata->setFont(cjkBigFont);

    if (horizontal)
        m_layout = new QHBoxLayout();
    else
        m_layout = new QVBoxLayout();
    m_layout->addWidget(lbl_roma);
    if (horizontal) m_layout->addSpacing(20);
    m_layout->addWidget(lbl_hira);
    if (horizontal) m_layout->addSpacing(20);
    m_layout->addWidget(lbl_kata);

    setLayout(m_layout);

    setAttribute(Qt::WA_TranslucentBackground);

    ReadKanaFromFile();

    QTimer* myTimer = new QTimer(this);
    myTimer->setInterval(2 * 60 * 60 * 1000); //Two hours.
    myTimer->start();
    connect(myTimer, SIGNAL(timeout()), this, SLOT(ReadKanaFromFile()));

    imgLoading = new QImageReader(":/loading.gif");

    //f = new QFile("/home/user/kotddebug.txt", this);
    //f->open(QIODevice::WriteOnly);
}

KanaOTDHomeWidget::~KanaOTDHomeWidget()
{
    delete imgLoading;
}

QSize KanaOTDHomeWidget::sizeHint() const
{
    if (horizontal)
        return QSize(300, 150);
    else
        return QSize(150,300);
}

void KanaOTDHomeWidget::moveEvent(QMoveEvent*)
{
    //This event only fires when dragging stops.
    update(); //For rounded border control.
}

void KanaOTDHomeWidget::paintEvent(QPaintEvent* event)
{
    QPainter p(this);
    if (!mousePressed)
        p.setBrush(QColor(0, 0, 0, 160));
    else
        p.setBrush(QColor(0, 128, 255, 160));
    p.setPen(Qt::NoPen);
    //If the widget is at the edges of the screen, don't draw rounded borders. 56 is the topmost
    //  location Maemo desktop puts the widget.
    if (pos().x() == 0 || pos().y() == 56 || pos().x() + width() == 800 || pos().y() + height() == 480)
        p.drawRect(rect());
    else
        p.drawRoundedRect(rect(), 15, 15);

    if (loadingAnim)
    {
        QImage nextImage = imgLoading->read();
        if (nextImage.isNull())
        {
            //Of course widget must be tapped more than once to cause the animation to reach
            //  the end:

            //imgLoading->jumpToImage(0);                   //Doesn't work
            delete imgLoading;                              //Does work
            imgLoading = new QImageReader(":/loading.gif"); //Does work

            nextImage = imgLoading->read();
        }
        p.drawImage((width()  - imgLoading->size().width() ) / 2,
                    (height() - imgLoading->size().height()) / 2,
                    nextImage);
    }
    else
    {
        QWidget::paintEvent(event); //Althoug this is not called in case of loadingAnim,
                                    //  children are painted! I discard it, since I don't
                                    //  want to tamper with visibility of children, etc, etc.
                                    //  Let children be visible when animation is running;
                                    //  it's not that ugly!
    }
}

void KanaOTDHomeWidget::mousePressEvent(QMouseEvent* event)
{
    if (event->button() == Qt::LeftButton)
    {
        mousePressed = rect().contains(event->pos());
        //QDebug(f) << "PRESS " << rect() << " " << event->pos() << " " << rect().contains(event->pos()) << "\n";
        update();
    }
    event->ignore();
}

//Note: As a home widget, Maemo doesn't send move events at all, it send only one move event
//      with empty event->buttons().
/*void KanaOTDHomeWidget::mouseMoveEvent(QMouseEvent *event)
{
    QDebug(f) << "MOVING " << event->buttons() << "\n";
    if (event->buttons() & Qt::LeftButton) //This is the way for mouseMoveEvent
    {
        //Without `rect().contains(event->pos())` the thing stays blue by tapping inside then
        //  dragging outside!
        mousePressed = rect().contains(event->pos());
        QDebug(f) << "MOVE  " << rect() << " " << event->pos() << " " << rect().contains(event->pos()) << "\n";
        update();
    }
    event->ignore();
}*/

void KanaOTDHomeWidget::mouseReleaseEvent(QMouseEvent* event)
{

    if (event->button() == Qt::LeftButton)
    {
        //<old>
        //We check mousePressed, so when user drags out then enters again, we don't run the program.
        //  Why we don't run the program? Since mouseMoveEvent doesn't trigger after user drags out!
        //If mouseMoveEvent didn't have that bug, we checked `rect().contains(event->pos())` instead.
        //</old>
        //Update: Since I removed mouseMoveEvent, now I check `rect().contains(event->pos())`.

        mousePressed = false;
        update();
        //QDebug(f) << "RELEA " << rect() << " " << event->pos() << " " << rect().contains(event->pos()) << "\n";

        if (rect().contains(event->pos()))
        {
            QTimer* stopLoadingAnimTimer = new QTimer(this);
            stopLoadingAnimTimer->setSingleShot(true);
            connect(stopLoadingAnimTimer, SIGNAL(timeout()), this, SLOT(stopLoadingAnimation()));
            stopLoadingAnimTimer->start(1000);

            loadingAnim = true;
            if (loadingAnimTimer)
                delete loadingAnimTimer;
            loadingAnimTimer = new QTimer(this);
            //We connect it to `repaint()` since `update()` wasn't fast enough.
            connect(loadingAnimTimer, SIGNAL(timeout()), this, SLOT(repaint()));
            loadingAnimTimer->start(50);

            //Can launch multiple instances:
            //QProcess::startDetached("/opt/kanaoftheday/bin/kanaoftheday");

            //Can launch multiple instances:
            //QProcess::startDetached("dbus-launch", QStringList("/opt/kanaoftheday/bin/kanaoftheday"));

            //Can NOT launch multiple instances:
            //  CORRECT BEHAVIOR and CORRECT WAY OF CODING :)
            //Actually Maemo handles it and doesn't run extra instance.
            QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.HildonDesktop.AppMgr",
                                                              "/com/nokia/HildonDesktop/AppMgr",
                                                              "com.nokia.HildonDesktop.AppMgr",
                                                              "LaunchApplication");
            msg << "kanaoftheday";
            QDBusConnection::sessionBus().send(msg);
        }
    }
    event->ignore();
}

void KanaOTDHomeWidget::ReadKanaFromFile()
{
    bool errors;
    ReadPermFromFile(current_kana, current_kana_set, current_permtab, errors);
    int todayKanaIndex = TodayKanaIndex(current_kana, current_kana_set, current_permtab);
    UpdateKana(todayKanaIndex);
}

int max3(int a, int b, int c)
{
    if (a > b)
        return a > c ? a : c;
    else
        return b > c ? b : c;
}

void KanaOTDHomeWidget::UpdateKana(int index)
{
    //Always set the maximum width and height for the widget, then cancel it to shrink the widget
    //  to the smallest needed space.
    //setMaximumSize(10, 10);

    int oldwidth = width();

    lbl_roma->setText(QString::fromUtf8(get_kana_sign(index, ROMAJI)));
    lbl_hira->setText(QString::fromUtf8(get_kana_sign(index, HIRAGANA)));
    lbl_kata->setText(QString::fromUtf8(get_kana_sign(index, KATAKANA)));

    //setMaximumSize(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX);

    //Wait! The above size setting doesn't work on Maemo so we calculate the size ourselves...
    //lbl_*->minimumSize() didn't work, so I used minimumSizeHint().
    /*int m_width, m_height;
    if (horizontal)
    {
        m_width = lbl_roma->minimumSizeHint().width() + lbl_hira->minimumSizeHint().width() + lbl_kata->minimumSizeHint().width() + 40; //+40 = spacers' sizes
        m_height = max3(lbl_roma->minimumSizeHint().height(), lbl_hira->minimumSizeHint().height(), lbl_kata->minimumSizeHint().height());
    }
    else
    {
        m_width = max3(lbl_roma->minimumSizeHint().width(), lbl_hira->minimumSizeHint().width(), lbl_kata->minimumSizeHint().width());
        m_height = lbl_roma->minimumSizeHint().height() + lbl_hira->minimumSizeHint().height() + lbl_kata->minimumSizeHint().height();
    }*/

    /*QSize sz_roma = QFontMetrics(lbl_roma->font()).size(Qt::TextSingleLine, lbl_roma->text());
    QSize sz_hira = QFontMetrics(lbl_hira->font()).size(Qt::TextSingleLine, lbl_hira->text());
    QSize sz_kata = QFontMetrics(lbl_kata->font()).size(Qt::TextSingleLine, lbl_kata->text());

    if (horizontal)
    {
        m_width = sz_roma.width() + sz_hira.width() + sz_kata.width() + 40; //+40 = spacers' sizes
        m_height = max3(sz_roma.height(), sz_hira.height(), sz_kata.height());
    }
    else
    {
        m_width = max3(sz_roma.width(), sz_hira.width(), sz_kata.width());
        m_height = sz_roma.height() + sz_hira.height() + sz_kata.height();
    }*/
    //NOTE: Had to take content margins into account too.
    //setFixedSize(m_width, m_height);

    //Finally: SIMPLE SOLUTION!!! setMaximumSize can't be used (it doesn't reduce widget's size).
    //This is the same as adding up labels' minimumSizeHint() (but NOT minimumSize()) or adding
    //  up font metrics and content margins of the labels.
    setFixedSize(m_layout->minimumSize());

    //We should see which way we want to dock (or keep near the edge) the home widget.
    bool KeepLeft = (pos().x() < 800-(pos().x() + width()));
    int diff = width() - oldwidth;
    if (!KeepLeft) //Well then keep it to the right side!
        move(pos().x() - diff, pos().y());

    //If after moving kana is long, the width of widget might increase and go out of the screen horizontally.
    //  Vertically this is not possible.
    if (pos().x() < 0)
        move(0, pos().y());
    else if (pos().x() + width() > 800)
        move(800 - width(), pos().y()); //Originally: move(pos().x() - (pos().x() + width() - 800), pos().y());
}

void KanaOTDHomeWidget::stopLoadingAnimation()
{
    loadingAnim = false;
    loadingAnimTimer->stop();
    update();
}
