#include <QContact>
#include <QContactName>
#include <QContactFetchRequest>
#include <QBuffer>
#include <QDataStream>
#include <QHash>
#include <QContactLocalIdFilter>
#include <QTime>
#include <QDesktopWidget>
#include <QContactThumbnail>
#include <QSettings>
#include <QDBusMessage>
#include <QDBusConnection>
#include <QPixmap>
#include <QColor>
#include <QFontMetrics>
#include <QErrorMessage>
#include "k9call.h"

#include <QtDebug>

using namespace QtMobility;

static const char* KEYPAD_IDX_PROPERTY = "keypadidx";

static const unsigned int CACHE_VERSION = 13;

static const int MAX_NAME_SIZE = 30 /*pt*/;

// The QtMobility framework doesn't handle unicode strings correctly
static inline QString fixUTF8String(const QString& str) {
	if(str.length() > 0) {
		// There is something strange about fromUtf8 that such that 
		// it doesn't work for strings of length 3 or less.
		const char* temp = ("AAAAA"+str).toLatin1().data();
		QString ustr = QString::fromUtf8(temp);
		return ustr.mid(5);
	}
	return str;
}

k9call::k9call(QWidget *parent) :
    QWidget(parent), ui(), m_Settings(NULL),
    m_FirstNameTrie(), m_LastNameTrie(), m_Accel(),
    m_ActiveList(), m_ActiveListItr(m_ActiveList.constBegin()),
    m_PhoneNumbers(), m_PhoneNumbersItr(m_PhoneNumbers.constBegin()),
    m_Manager(NULL), m_ContactWorker(NULL), 
    m_Initialized(false), m_SearchString()
{
    createStyle();
	
	setupUi();
    QTimer::singleShot(0, this, SLOT(populateList()));
}

void k9call::setupUi() {
	ui.setupUi(this);
	
	// Setup the keypad
	setupButton(KEYPAD_BUTTON_1, ui.button1);
	setupButton(KEYPAD_BUTTON_2, ui.button2);
	setupButton(KEYPAD_BUTTON_3, ui.button3);
	
	setupButton(KEYPAD_BUTTON_4, ui.button4);
	setupButton(KEYPAD_BUTTON_5, ui.button5);
	setupButton(KEYPAD_BUTTON_6, ui.button6);
	
	setupButton(KEYPAD_BUTTON_7, ui.button7);
	setupButton(KEYPAD_BUTTON_8, ui.button8);
	setupButton(KEYPAD_BUTTON_9, ui.button9);
	
	setupButton(KEYPAD_BUTTON_STAR, ui.buttonStar);
	setupButton(KEYPAD_BUTTON_0   , ui.button0);
	setupButton(KEYPAD_BUTTON_HASH, ui.buttonHash);
	
	// Setup misc buttons
	connect(ui.clear, SIGNAL(clicked()), this, SLOT(clear()));
	connect(ui.name, SIGNAL(clicked()), this, SLOT(nextPhoneNumber()));
	connect(ui.next, SIGNAL(clicked()), this, SLOT(nextContact()));
	connect(ui.call, SIGNAL(clicked()), this, SLOT(call()));
	
	ui.name->setStyleSheet(m_StyleLetters);
	ui.next->setStyleSheet(m_StyleLetters);
	ui.call->setStyleSheet(m_StyleCallButton);
	ui.clear->setStyleSheet(m_StyleClearButton);

	// Setup pic
	ui.pic->setScene(new QGraphicsScene());

	// Make the background TRANSPARENT!
	
	setStyleSheet("background: rgba(0,0,0,128)");
	// setAttribute(Qt::WA_TranslucentBackground);
}

void k9call::setupButton(KeypadButton idx, QPushButton* button) {
    button->setStyleSheet(m_StyleLetters);
    button->setProperty(KEYPAD_IDX_PROPERTY, idx);
    connect(button, SIGNAL(clicked()), this, SLOT(insertChar()));
}

void k9call::createStyle()
{
	m_StyleLetters =
	        "QPushButton {"
	        "	color: white;"
	        "	background-color: qlineargradient(x1: 0, y1: 0, "
	                                             "x2: 0, y2: 1, "
	                                             "stop: 0 #222, "
	                                             "stop: 1 #333);"
	        "	border-style: outset;"
	        "	border-radius: 3px;"
	        "	border-width: 1px;"
	        "	border-color: #000;"
	        "}"
	        "QPushButton:pressed {"
	        "	background-color: qlineargradient(x1: 0, y1: 0, "
	                                             "x2: 0, y2: 1, "
	                                             "stop: 0 #16B, "
	                                             "stop: 1 #2AC);"
	        "}";

	m_StyleCallButton =
	        "QPushButton {"
	        "	color: white;"
	        "	background-color: qlineargradient(x1: 0, y1: 0, "
	                                             "x2: 0, y2: 1, "
	                                             "stop: 0 #242, "
	                                             "stop: 1 #353);"
	        "	border-style: outset;"
	        "	border-radius: 3px;"
	        "	border-width: 1px;"
	        "	border-color: #020;"
	        "}"
	        "QPushButton:pressed {"
	        "	background-color: qlineargradient(x1: 0, y1: 0, "
	                                             "x2: 0, y2: 1, "
	                                             "stop: 0 #393, "
	                                             "stop: 1 #282);"
	        "}";

	m_StyleClearButton =
	        "QPushButton {"
	        "	color: white;"
	        "	background-color: qlineargradient(x1: 0, y1: 0, "
	                                             "x2: 0, y2: 1, "
	                                             "stop: 0 #422, "
	                                             "stop: 1 #533);"
	        "	border-style: outset;"
	        "	border-radius: 3px;"
	        "	border-width: 1px;"
	        "	border-color: #200;"
	        "}"
	        "QPushButton:pressed {"
	        "	background-color: qlineargradient(x1: 0, y1: 0, "
	                                             "x2: 0, y2: 1, "
	                                             "stop: 0 #933, "
	                                             "stop: 1 #822);"
	        "}";
}

void k9call::clear()
{
	m_SearchString = m_SearchString.left(m_SearchString.length()-1);
	updateQuery();
}

void k9call::updateQuery()
{
	ui.phoneNumber->setText(m_SearchString);

	QStringList qlist = m_SearchString.split("0", QString::SkipEmptyParts);
	QSet<QContactLocalId> set;
	if(qlist.empty())
		set = m_FirstNameTrie.lookup ( m_SearchString.data() );
	else {
		set = m_FirstNameTrie.lookup ( qlist[0].data() );
		if(qlist.size() == 1) {
			// There isn't a space, so just look up for last names
			set.unite(m_LastNameTrie.lookup(qlist[0].data()));
		}
		else {
			// There is a space, assume FirstName LastName pattern
			set.intersect(m_LastNameTrie.lookup(qlist[1].data()));
		}
	}

	if(set.count() < 15 && set.count() > 0) {
		m_ActiveList = QList<QContactLocalId>::fromSet(set);

		if(m_Manager == NULL)
			m_Manager = new QContactManager();
	}
	else {
		// Not filtered enough, just clear it
		m_ActiveList.clear();
		m_PhoneNumbers.clear();

		// Free up the manager
		delete m_Manager;
		m_Manager = NULL;
	}

	// Sort the result
	qSort(m_ActiveList);

	// Prepend accelerator result
	QPair<QContactLocalId, QContactLocalId>& accel = m_Accel[m_SearchString];
	if(accel.first != 0) {
		m_ActiveList.removeOne(accel.first);
		m_ActiveList.prepend(accel.first);
	}

	m_ActiveListItr = --m_ActiveList.constEnd();
	nextContact();
}

void k9call::nextContact() {
	if(m_ActiveList.size() == 0) {
		ui.name->setText("");
		ui.name->setEnabled(false);
		ui.pic->scene()->clear();
		return;
	}

	// Increment and wrap around (if necessary)
	m_ActiveListItr++;
	if(m_ActiveListItr == m_ActiveList.constEnd())
		m_ActiveListItr = m_ActiveList.constBegin();

	QContactLocalId id = *m_ActiveListItr;
	QContact contact = m_Manager->contact(id);
	QContactName name = contact.detail(QContactName::DefinitionName);
	QString display = fixUTF8String(name.firstName()+" "+name.lastName());

	// Figure out optimal font sizes
	QFont font = ui.name->font();
	QFontMetrics fontmetric(font);
	const int nameWidth = ui.name->width()-10;
	const int nameHeight = (ui.name->height()-10);
	// Enlarge the font as necessary
	while(fontmetric.width(display) < nameWidth &&
	      fontmetric.height() < nameHeight &&
	      font.pointSize() < MAX_NAME_SIZE) {
		// Font is too small, make it bigger
		font.setPointSize(font.pointSize()+1);
		fontmetric = QFontMetrics(font);
	}
	// Shrink to fit
	while(fontmetric.width(display) > nameWidth ||
	      fontmetric.height() > nameHeight || 
	      font.pointSize() == 1) {
		// Font is too small, make it bigger
		font.setPointSize(font.pointSize()-1);
		fontmetric = QFontMetrics(font);
	}
	ui.name->setFont(font);
	ui.name->setText(display);
	ui.name->setEnabled(true);

	// Setup code for thumbnail
	QContactThumbnail detail = contact.detail(QContactThumbnail::DefinitionName);
	QImage thumbnail = detail.thumbnail().convertToFormat(QImage::Format_RGB32);
	if(thumbnail.isNull()) {
		// No thumbnail so just clear the pic
		ui.pic->scene()->clear();
	}
	else {
		// Scale the image to fit (upscale until Nokia pushes the fix thorugh, same bug as below)
		thumbnail = thumbnail.scaled(ui.pic->size(), Qt::KeepAspectRatioByExpanding, Qt::SmoothTransformation);

		// Render Image
		QPixmap pix = QPixmap::fromImage(thumbnail);
		QGraphicsScene *scene = ui.pic->scene();
		scene->clear();
		scene->addPixmap(pix);
	}

	m_PhoneNumbers = contact.details<QContactPhoneNumber>();
	m_PhoneNumbersItr = m_PhoneNumbers.constBegin();

	// Reset the phone numbers list back to the search string
	ui.phoneNumber->setText(m_SearchString);
}

void k9call::nextPhoneNumber() {
	if(m_PhoneNumbers.size() == 0)
		return;
	
	ui.phoneNumber->setText(m_PhoneNumbersItr->number());
	
	// Increment and wrap around (if necessary)
	m_PhoneNumbersItr++;
	if(m_PhoneNumbersItr == m_PhoneNumbers.constEnd())
		m_PhoneNumbersItr = m_PhoneNumbers.constBegin();
}

void k9call::populateList()
{
	qDebug("Start Populating List");
	
	// Then load from cache
	if(!loadCache()) {
		qDebug("Load Cache failed, requesting new list of contacts");
		syncContacts();
	}

	// Clear the search string
	m_SearchString = "";
	updateQuery();

	// Finish setup components
	m_Initialized = true;
	
	qDebug("Finish initial list population");
}

void k9call::makeDbusGSMCall(QString number)
{
    QDBusMessage msg = QDBusMessage::createMethodCall(
        "com.nokia.csd.Call", // --dest
        "/com/nokia/csd/call", // destination object path
        "com.nokia.csd.Call", // message name (w/o method)
        "CreateWith" // method
    );
    msg << number;
    msg << 0;
    msg = QDBusConnection::systemBus().call(msg);
	
	if(msg.type() == QDBusMessage::ErrorMessage) {
		QErrorMessage* err = new QErrorMessage();
		err->showMessage(msg.errorMessage());
	}
}

bool k9call::doesPhoneNumberExist(QContact *i_Contact)
{
    QList<QContactDetail> details = i_Contact->details();
    for (int z=0;z < details.size(); z++)
    {
        QString temp = details.at(z).definitionName();
        if (temp == "PhoneNumber") {
			return true;
        }
    }

	return false;
}

void k9call::call()
{
	// Update accelerator
	QPair<QContactLocalId, QContactLocalId>& accel = m_Accel[m_SearchString];
	QContactLocalId id = *m_ActiveListItr;
	if(accel.first == 0 || accel.second == id)
		accel.first = id;
	else
		accel.second = id;

	// You want to dial whatever number is shown to the user
	QString number = ui.phoneNumber->text();

	// Clear the text
	m_SearchString = "";
	updateQuery();

	// Make the call
	makeDbusGSMCall(number);
}

void k9call::insertChar()
{
    QPushButton *tempButton = qobject_cast<QPushButton *> ( QObject::sender() );
    QVariant idx = tempButton->property ( KEYPAD_IDX_PROPERTY );
    if ( !idx.isValid() )
        return;

    switch ( idx.toInt() )
    {
    case KEYPAD_BUTTON_0:
		m_SearchString += "0";
        break;
    case KEYPAD_BUTTON_1:
		m_SearchString += "1";
        break;
    case KEYPAD_BUTTON_2:
		m_SearchString += "2";
        break;
    case KEYPAD_BUTTON_3:
		m_SearchString += "3";
        break;
    case KEYPAD_BUTTON_4:
		m_SearchString += "4";
        break;
    case KEYPAD_BUTTON_5:
		m_SearchString += "5";
        break;
    case KEYPAD_BUTTON_6:
		m_SearchString += "6";
        break;
    case KEYPAD_BUTTON_7:
		m_SearchString += "7";
        break;
    case KEYPAD_BUTTON_8:
		m_SearchString += "8";
        break;
    case KEYPAD_BUTTON_9:
		m_SearchString += "9";
        break;
    case KEYPAD_BUTTON_STAR:
		m_SearchString += "*";
        break;
    case KEYPAD_BUTTON_HASH:
		m_SearchString += "#";
        break;
    default:
        return;
    }
    
    updateQuery();
}

void k9call::syncContacts() 
{
	if(m_ContactWorker)
		return; // Already syncing

	m_ContactWorker = new ContactsThreadWorker(this);
	QTimer::singleShot(0, m_ContactWorker, SLOT(syncContacts()));
}

void k9call::updateContacts(QList<QContact> contacts) 
{
	qDebug() << "Got update contacts call!";
	// Delete the Contacts Thread Worker
	m_ContactWorker->quit();
	m_ContactWorker->wait();
	delete m_ContactWorker;
	m_ContactWorker = NULL;

	// Reset Tries
	m_FirstNameTrie.reset();
	m_LastNameTrie.reset();
	m_ActiveList.clear();
	
	int size = contacts.count();
	for (int i = 0; i < size; i++)
	{
		QContact &temp = contacts[i];
		QContactDisplayLabel displayLabel = temp.detail(QContactDisplayLabel::DefinitionName);

		if(!doesPhoneNumberExist(&temp) || displayLabel.isEmpty()) {
			contacts.removeAt(i);
			i--;
			size--;
			continue;
		}
	}

	saveCache(contacts);
	loadCache();

	qDebug("Done Updating Contacts");
}

void k9call::saveCache(const QList<QContact> &contacts)
{
    qDebug("Saving Cache");
	QHash<QContactLocalId, QList<QString> > cache;
	
	int size = contacts.count();
	for (int i = 0; i < size; i++)
	{
		const QContact &temp = contacts[i];
		QContactName name = temp.detail(QContactName::DefinitionName);
		
		QList<QString> contact;
		contact.append(fixUTF8String(name.firstName()));
		contact.append(fixUTF8String(name.lastName()));
		cache.insert(temp.localId(), contact);
	}
	
	QByteArray array;
	QDataStream ds(&array, QIODevice::WriteOnly);
	ds << cache << m_Accel;
	
	// Compress the array
	QByteArray carray = qCompress(array, 9);

	qDebug() << "Saving" << cache.count() << "entries in" << array.count() << "compressed in" << carray.count();
	QSettings settings;
	
	settings.setValue("cache", carray);
	settings.setValue("cache_version", CACHE_VERSION);
	settings.sync();
}

bool k9call::loadCache()
{
    qDebug("Loading Cache");
	QSettings settings;
	
	// Check cache version
	QVariant qversion = settings.value("cache_version");
	if(!qversion.canConvert(QVariant::UInt) || qversion.toUInt() != CACHE_VERSION)
		return false;

	// Get the cache
	QVariant qvalue = settings.value("cache");
	if(!qvalue.canConvert(QVariant::ByteArray))
		return false;
	
	QByteArray carray = qvalue.toByteArray();
	QByteArray array = qUncompress(carray);
	qDebug() << "Loading array of size " << array.count() << " compressed in " << carray.count();
	
	QHash<QContactLocalId, QList<QString> > cache;
	
	QDataStream ds(array);
	
	ds >> cache >> m_Accel;
	
	// First reset Tries
	m_FirstNameTrie.reset();
	m_LastNameTrie.reset();
	m_ActiveList.clear();
	
	QHashIterator<QContactLocalId, QList<QString> > i(cache);
	while(i.hasNext()) {
		i.next();
		
		const QString &firstName = i.value().at(0);
		const QString &lastName = i.value().at(1);

		m_FirstNameTrie.insert(firstName.data(), i.key());
		m_LastNameTrie.insert(lastName.data(), i.key());
		m_ActiveList.append(i.key());
	}
	
	m_SearchString = "";
	updateQuery();
	return true;
}

void k9call::openSettings() {
	Ui_Settings settings;

	if(m_Settings)
		closeSettings();

	// Setup new dialog
	m_Settings = new QDialog(0);
	settings.setupUi(m_Settings);

	// Connect relevant signals
	connect(settings.updateContacts, SIGNAL(clicked()), this, SLOT(syncContacts()));
	connect(settings.done, SIGNAL(clicked()), this, SLOT(closeSettings()));

	// Show settings dialog
	m_Settings->show();
}

void k9call::closeSettings() {
	if(!m_Settings)
		return;
	
	m_Settings->close();
	delete m_Settings;
	m_Settings = NULL;
}
