/*
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <QDebug>
#include "t9trie.h"

QString T9Trie::T9Value ( const QString& str )
{
	int i, len = str.length();
	QString result;
	result.reserve(len);
	for(i = 0; i < len; i++) {
		result.append(QVariant(T9Value(str[i])).toString());
	}
	return result;
}

bool T9Trie::insert ( const QChar* key, QtMobility::QContactLocalId id)
{
    if ( key->isNull() )
    {
        list.insert ( id );
        return false;
    }

    int t9 = T9Value ( *key );
    if ( t9 == -1 )
        return insert ( key+1, id );

    if ( children[t9] == NULL )
        children[t9] = new T9Trie();

    children[t9]->insert ( key+1, id );
    return true;
}

QSet<QtMobility::QContactLocalId> T9Trie::lookup ( const QChar* key )
{
    if ( key->isNull() )
    {
        QSet<QtMobility::QContactLocalId> set = list;
        for ( int i = 0; i < NUM_CHILDREN; i++ )
        {
            if ( children[i] )
                set = set.unite ( children[i]->lookup ( key ) );
        }
        return set;
    }

    int digit = key->digitValue();
    if ( digit == -1 || children[digit] == NULL )
        return QSet<QtMobility::QContactLocalId>();

    return children[digit]->lookup ( key+1 );
}

int T9Trie::T9Value(const QChar& c)
{
	switch ( c.toLower().toAscii() )
	{
	case '1':
		return 1;
	case '2':
	case 'a':
	case 'b':
	case 'c':
		return 2;
	case '3':
	case 'd':
	case 'e':
	case 'f':
		return 3;
	case '4':
	case 'g':
	case 'h':
	case 'i':
		return 4;
	case '5':
	case 'j':
	case 'k':
	case 'l':
		return 5;
	case '6':
	case 'm':
	case 'n':
	case 'o':
		return 6;
	case '7':
	case 'p':
	case 'q':
	case 'r':
	case 's':
		return 7;
	case '8':
	case 't':
	case 'u':
	case 'v':
		return 8;
	case '9':
	case 'w':
	case 'x':
	case 'y':
	case 'z':
				return 9;
	case '0':
	case ' ':
		return 0;
	case '\xE4' /* ä */:
		return 2;
	case '\xF6' /* ö */:
		return 6;
	case '\xFC' /* ü */:
		return 8;
	case '\xDF' /* ß */:
		return 7;
	case 0:
		switch(c.toLower().unicode()) {
		case 0x05D0 /* א */:
		case 0x05D1 /* ב */:
		case 0x05D2 /* ג */:
			return 3;
		case 0x05D3 /* ד */:
		case 0x05D4 /* ה */:
		case 0x05D5 /* ו */:
			return 2;
		case 0x05D6 /* ז */:
		case 0x05D7 /* ח */:
		case 0x05D8 /* ט */:
			return 6;
		case 0x05D9 /* י */:
		case 0x05DA /* ך */:
		case 0x05DB /* כ */:
		case 0x05DC /* ל */:
			return 5;
		case 0x05DD /* ם */:
		case 0x05DE /* מ */:
		case 0x05DF /* ן */:
		case 0x05E0 /* נ */:
			return 4;
		case 0x05E1 /* ס */:
		case 0x05E2 /* ע */:
		case 0x05E3 /* ף */:
		case 0x05E4 /* פ */:
			return 9;
		case 0x05E5 /* ץ */:
		case 0x05E6 /* צ */:
		case 0x05E7 /* ק */:
			return 8;
		case 0x05E8 /* ר */:
		case 0x05E9 /* ש */:
		case 0x05EA /* ת */:
			return 7;
		default:
			return -1;
		}
	default:
		return -1;
	}
}
