/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtGui/QMenu>
#include <QtGui/QApplication>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QCheckBox>
#include <QtGui/QLabel>
#include <QtGui/QScrollArea>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QPushButton>
#include <QtGui/QListWidget>
#include <QMaemo5InformationBox>
#include <QMaemo5ValueButton>
#include <QMaemo5TimePickSelector>
#include "themeschedulersettings.h"
#include "themescheduler.h"
#include "themepicker.h"
#include "buttonbox.h"

namespace
{
    QString const TIME_FORMAT = "hh:mm";
}

ThemeSchedulerSettings::ThemeSchedulerSettings(QWidget* parent):
QDialog(parent), currentWidget_(0), addDialog_(0), itemList_(0)
{
    setWindowTitle(tr("Theme scheduler"));

    QHBoxLayout* layout = new QHBoxLayout;
    layout_ = new QVBoxLayout;

    enabled_ = new QCheckBox(tr("Enabled"));
    enabled_->setChecked(ThemeScheduler::instance().isEnabled());

    QLabel* info = new QLabel(tr("Scheduled themes:"));

    layout_->addWidget(enabled_);
    layout_->addWidget(info);

    loadItems();

    ButtonBox* buttons = new ButtonBox;
    connect(buttons->addButton(tr("Save"), QDialogButtonBox::AcceptRole), SIGNAL(clicked(bool)), this, SLOT(saveSettings()));
    connect(buttons->addButton(tr("Add new theme"), QDialogButtonBox::ActionRole), SIGNAL(clicked(bool)), this, SLOT(openAddDialog()));
    connect(buttons->addButton(tr("Clear scheduler"), QDialogButtonBox::ActionRole), SIGNAL(clicked(bool)), this, SLOT(clearScheduler()));

    layout->addLayout(layout_, Qt::AlignLeft);
    layout->addWidget(buttons);

    setLayout(layout);

}

void ThemeSchedulerSettings::openAddDialog()
{
    if(!addDialog_)
    {
        addDialog_ = new QDialog(this);
        addDialog_->setWindowTitle("Add scheduled theme");
        QHBoxLayout* layout = new QHBoxLayout;
        QVBoxLayout* left = new QVBoxLayout;
        timeButton_ = new QMaemo5ValueButton(tr("Start time"));
        timeButton_->setValueLayout(QMaemo5ValueButton::ValueBesideText);
        pickSelector_ = new QMaemo5TimePickSelector;
        timeButton_->setPickSelector(pickSelector_);
        themePicker_ = new ThemePicker(tr("Theme"));

        ButtonBox* buttons = new ButtonBox;
        connect(buttons->addButton(tr("Add"), QDialogButtonBox::AcceptRole), SIGNAL(clicked(bool)), this, SLOT(addScheduledTheme()));

        left->addWidget(timeButton_);
        left->addWidget(themePicker_);

        layout->addLayout(left);
        layout->addWidget(buttons);

        addDialog_->setLayout(layout);

    }

    pickSelector_->setCurrentTime(QTime::currentTime());
    addDialog_->show();
}

void ThemeSchedulerSettings::addScheduledTheme()
{
    QTime time = QTime::fromString(timeButton_->valueText(), TIME_FORMAT);
    QString theme = themePicker_->value().toString();
    ThemeScheduler::instance().addItem(time, theme);
    loadItems();
    addDialog_->hide();
}

void ThemeSchedulerSettings::clearScheduler()
{
    if(!ThemeScheduler::instance().isEmpty())
    {
        ThemeScheduler::instance().clear();
        loadItems();
    }
}

void ThemeSchedulerSettings::loadItems()
{
    ThemeScheduler::instance().getItems(items_);
    itemList_ = 0;

    if(currentWidget_)
    {
        layout_->removeWidget(currentWidget_);
        delete currentWidget_;
        currentWidget_ = 0;
    }

    if(items_.isEmpty())
    {
        QLabel* label = new QLabel(tr("There are currently no scheduled themes."));
        label->setWordWrap(true);
        label->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        currentWidget_ = label;
        layout_->addWidget(label);
    }
    else
    {
        itemList_ = new QListWidget;
        itemList_->setContextMenuPolicy(Qt::CustomContextMenu);
        connect(itemList_, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(showContextMenu(const QPoint&)));

        for(int i = 0; i < items_.size(); i++)
        {
            const QTime* endTime = 0;

            if(i < items_.size() - 1)
            {
                endTime = &(items_.at(i + 1).time);
            }
            else
            {
                endTime = &(items_.at(0).time);
            }

            QString theme = items_.at(i).theme;
            QString text = items_.at(i).time.toString(TIME_FORMAT) + " - " +
                              endTime->toString(TIME_FORMAT) + ": " +
                              theme.at(0).toUpper() + theme.mid(1);
            QListWidgetItem* item = new QListWidgetItem(text);
            item->setData(Qt::UserRole, items_.at(i).time.toString(TIME_FORMAT));
            itemList_->addItem(item);
        }

        currentWidget_ = itemList_;
        layout_->addWidget(itemList_);
    }
}

void ThemeSchedulerSettings::saveSettings()
{
    bool enabled = enabled_->isChecked();

    if(enabled && ThemeScheduler::instance().isEmpty())
    {
        QMaemo5InformationBox::information(this, tr("Theme scheduler is empty."));
        return;
    }

    ThemeScheduler::instance().setEnabled(enabled);
    hide();

    if(enabled)
    {
        emit themeChanged();
    }
}

void ThemeSchedulerSettings::showContextMenu(QPoint const& point)
{
    if(!itemList_)
    {
        return;
    }

    QMenu* menu = new QMenu(itemList_);
    menu->addAction(tr("Remove"), this, SLOT(removeCurrent()));
    connect(menu, SIGNAL(aboutToHide()), this, SLOT(removeSelection()));
    menu->popup(itemList_->mapToGlobal(point));
}

void ThemeSchedulerSettings::removeCurrent()
{
    if(itemList_)
    {
        QListWidgetItem* item = itemList_->currentItem();

        if(item)
        {
            ThemeScheduler::instance().removeItem(QTime::fromString(item->data(Qt::UserRole).toString(), TIME_FORMAT));
            loadItems();
        }
    }
}

void ThemeSchedulerSettings::removeSelection()
{
    if(itemList_)
    {
        itemList_->clearSelection();
    }
}

void ThemeSchedulerSettings::setVisible(bool visible)
{
    if(visible)
    {
        enabled_->setChecked(ThemeScheduler::instance().isEnabled());
    }

    QDialog::setVisible(visible);
}
