/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QDebug>
#include "themescheduler.h"
#include "settings.h"

namespace
{
    QString const DEFAULT_THEME = "default";
}

ThemeScheduler::ThemeScheduler(): QObject(0), enabled_(false)
{
    loadConfig();
}

ThemeScheduler::~ThemeScheduler()
{
    store();
    clear();
}

ThemeScheduler& ThemeScheduler::instance()
{
    static ThemeScheduler instance;
    return instance;
}

void ThemeScheduler::addItem(QTime const& time, QString const& theme)
{
    removeItem(time);
    ItemDetails* details = new ItemDetails;
    details->time = time;
    details->theme = theme;

    int msecs = QTime::currentTime().msecsTo(time);

    if(msecs < 0)
    {
        msecs += 86400000;
    }

    details->timer.setSingleShot(true);
    details->timer.start(msecs + 500);
    connect(&details->timer, SIGNAL(timeout()), this, SLOT(emitThemeChanged()));

    for(int i = 0; i < items_.size(); i++)
    {
        if(items_.at(i)->time > time)
        {
            items_.insert(i, details);
            return;
        }
    }

    items_.push_back(details);
}

void ThemeScheduler::removeItem(QTime const& time)
{
    for(int i = 0; i < items_.size(); i++)
    {
        if(items_.at(i)->time == time)
        {
            delete items_.at(i);
            items_.removeAt(i);
            return;
        }
    }
}

void ThemeScheduler::store()
{
    Settings::instance().setValue("theme_scheduler_enabled", enabled_);

    if(items_.isEmpty())
    {
        Settings::instance().remove("theme_scheduler");
        return;
    }

    QList<QVariant> list;

    for(int i = 0; i < items_.size(); i++)
    {
        QMap<QString, QVariant> map;
        map["time"] = items_.at(i)->time.toString();
        map["theme"] = items_.at(i)->theme;
        list.push_back(map);
    }

    Settings::instance().setValue("theme_scheduler", list);
}

void ThemeScheduler::loadConfig()
{
    enabled_ = Settings::instance().value("theme_scheduler_enabled", false).toBool();

    QList<QVariant> list = Settings::instance().value("theme_scheduler").toList();

    for(int i = 0; i < list.size(); i++)
    {
        QMap<QString, QVariant> map = list.at(i).toMap();
        QTime time = QTime::fromString(map["time"].toString());
        addItem(time, map["theme"].toString());
    }
}

void ThemeScheduler::setEnabled(bool enabled)
{
    enabled_ = enabled;
}

bool ThemeScheduler::isEnabled() const
{
    return enabled_;
}

QString ThemeScheduler::currentTheme() const
{
    if(enabled_)
    {
        QTime time = QTime::currentTime();

        for(int i = 0; i < items_.size(); i++)
        {
            if(items_.at(i)->time >= time)
            {
                int index = i - 1;

                if(index < 0)
                {
                    index = items_.size() - 1;
                }

                return items_.at(index)->theme;
            }
        }

        if(items_.size() > 0)
        {
            return items_.at(items_.size() - 1)->theme;
        }
    }

    QString theme = Settings::instance().value("theme", DEFAULT_THEME).toString();

    if(theme.isEmpty())
    {
        return DEFAULT_THEME;
    }

    return theme;
}

QString const& ThemeScheduler::getDefaultTheme()
{
    return DEFAULT_THEME;
}

void ThemeScheduler::clear()
{
    for(int i = 0; i < items_.size(); i++)
    {
        delete items_.at(i);
    }

    items_.clear();
}

void ThemeScheduler::getItems(QList<SchedulerItem>& items)
{
    items.clear();

    for(int i = 0; i < items_.size(); i++)
    {
        SchedulerItem item;
        item.time = items_.at(i)->time;
        item.theme = items_.at(i)->theme;
        items.push_back(item);
    }
}

void ThemeScheduler::emitThemeChanged()
{
    if(enabled_)
    {
        emit themeChanged();
    }
}

bool ThemeScheduler::isEmpty() const
{
    return items_.isEmpty();
}
