/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QByteArray>
#include <QtCore/QDebug>
#include <QtGui/QGraphicsScene>
#include <QtGui/QFont>
#include <QtGui/QFontDatabase>
#include <QtGui/QGraphicsEffect>
#include <QtGui/QGraphicsDropShadowEffect>
#include <QtGui/QGraphicsItem>
#include "graphicselement.h"
#include "reader.h"
#include "textelement.h"
#include "imageelement.h"
#include "rectangle.h"
#include "pointer.h"
#include "compass.h"
#include "settings.h"
#include "effect.h"
#include "poialerts.h"
#include "speedalarm.h"

GraphicsElement::GraphicsElement(Reader* reader): QObject(0),
reader_(reader), error_(""), effect_(0), visibleWhen_(ALWAYS), canBeVisible_(true)
{
}

GraphicsElement* GraphicsElement::getElement(QString const& name, Reader* reader)
{
    if(name == "image")
    {
        return new ImageElement(reader);
    }

    if(name == "text")
    {
        return new TextElement(reader);
    }

    if(name == "rectangle")
    {
        return new Rectangle(reader);
    }

    if(name == "pointer")
    {
        bool animate = Settings::instance().value("animate_pointer", true).toBool();
        return new Pointer(reader, animate);
    }

    if(name == "compass")
    {
        bool animate = Settings::instance().value("animate_pointer", true).toBool();
        return new Compass(reader, animate);
    }

    qDebug() << "Element not found: " << name;
    return 0;
}

QString const& GraphicsElement::getError() const
{
    return error_;
}

void GraphicsElement::setError(QString const& error)
{
    error_ = error;
}

bool GraphicsElement::readFile(QString const& filename, QByteArray& data)
{
    if(!reader_->readFile(filename, data))
    {
        setError(reader_->errorString());
        return false;
    }

    return true;
}

bool GraphicsElement::getFont(QString const& name, QFont& font)
{
    QByteArray data;

    QMap<QString, QString>::const_iterator it = loadedFonts_.find(name);

    if(it != loadedFonts_.end())
    {
        font = QFont(it.value());
        return true;
    }

    if(!reader_->fileExists(name))
    {
        font = QFont(name);
        return true;
    }

    if(!readFile(name, data))
    {
        return false;
    }

    int id = QFontDatabase::addApplicationFontFromData(data);

    if(id == -1)
    {
        setError("Unable to load font: " + name);
        loadedFonts_[name] = "";
        return false;
    }

    QStringList families = QFontDatabase::applicationFontFamilies(id);

    if(families.size() == 0)
    {
        setError("No fonts found in " + name);
        return false;
    }

    loadedFonts_[name] = families.at(0);
    font = QFont(families.at(0));

    return true;
}

int GraphicsElement::getAttribute(QString const& name,
                                  QString const& value,
                                  const GraphicsElement::AttributeDetails details[],
                                  int count,
                                  int& intValue)
{
    QString lower = name.toLower();

    for(int i = 0; i < count; i++)
    {
        if(details[i].name == lower)
        {
            if(!details[i].isInt)
            {
                return i;
            }
            else
            {
                bool ok = true;
                int tmp = value.toInt(&ok);

                if(ok)
                {
                    intValue = tmp;
                    return i;
                }
                else
                {
                    setError("Value for " + name + " is not integer ("+value+")");
                    return -1;
                }
            }
        }
    }

    setError("Unknown attribute: " + name);
    return -1;
}

bool GraphicsElement::setEffect(QString const& name)
{
    effect_ = Effect::getEffect(name);

    if(!effect_)
    {
        return false;
    }

    return true;
}

bool GraphicsElement::setEffectAttribute(QString const& name, QString const& value)
{
    if(!effect_)
    {
        qDebug() << "Effect not set";
        return false;
    }

    return effect_->setAttribute(name, value);
}

void GraphicsElement::applyEffect()
{
    effect_->apply(this);
}

void GraphicsElement::setVisibleWhen(VisibleWhen when)
{
    if(when != visibleWhen_)
    {
        visibleWhen_ = when;

        switch(when)
        {
        case POI_VISIBLE:
            updateVisibility(false);
            connect(&(PoiAlerts::instance()), SIGNAL(visibilityChanged(bool)), this, SLOT(updateVisibility(bool)));
            break;
        case SPEED_EXCEEDED:
            updateVisibility(false);
            connect(&(SpeedAlarm::instance()), SIGNAL(speedExceedChanged(bool)), this, SLOT(updateVisibility(bool)));
            break;
        case ALWAYS:
            disconnect(&(PoiAlerts::instance()), SIGNAL(visibilityChanged(bool)), this, SLOT(updateVisibility(bool)));
            disconnect(&(SpeedAlarm::instance()), SIGNAL(speedExceedChanged(bool)), this, SLOT(updateVisibility(bool)));
            updateVisibility(true);
            break;
        }
    }
}

GraphicsElement::VisibleWhen GraphicsElement::strToVisibleWhen(QString const& str) const
{
    if(str == "poivisible")
    {
        return POI_VISIBLE;
    }
    else if(str == "speedexceeded")
    {
        return SPEED_EXCEEDED;
    }

    return ALWAYS;
}

void GraphicsElement::updateVisibility(bool visible)
{
    QGraphicsItem* item = getElement();

    if(item)
    {
        item->setVisible(visible);

        if(!visible)
        {
            canBeVisible_ = false;
        }
        else
        {
            canBeVisible_ = true;
        }
    }
}

bool GraphicsElement::canBeVisible() const
{
    return canBeVisible_;
}
