/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDir>
#include <QtCore/QDebug>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtGui/QFileDialog>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QPushButton>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QLabel>
#include <QtGui/QCheckBox>
#include <QtGui/QLineEdit>
#include <QtGui/QIntValidator>
#include <QMaemo5InformationBox>
#include <math.h>
#include "poisettings.h"
#include "fileselector.h"
#include "poialerts.h"
#include "settings.h"
#include "mediaplayer.h"
#include "poireader.h"
#include "buttonbox.h"
#include "soundselector.h"
#include "odometer.h"


PoiSettings::PoiSettings(QWidget* parent): QDialog(parent)
{
    setWindowTitle(tr("Speed camera alerts"));
    enabled_ = new QCheckBox(tr("Enabled"));
    enabled_->setChecked(Settings::instance().value("alert_enabled", false).toBool());

    QHBoxLayout* poiLayout = new QHBoxLayout;
    poiFileSelector_ = new FileSelector(tr("Poi file (.asc)"));
    QPushButton* importButton = new QPushButton(tr("Import"));
    connect(importButton, SIGNAL(clicked(bool)), this, SLOT(importFile()));
    poiLayout->addWidget(poiFileSelector_, Qt::AlignLeft);
    poiLayout->addWidget(importButton);

    soundSelector_ = new SoundSelector;

    distanceLabel_ = new QLabel;
    distance_ = new QLineEdit;
    distance_->setValidator(new QIntValidator(0, 5000, this));
    QHBoxLayout* distance = new QHBoxLayout;
    distance->addWidget(distanceLabel_);
    distance->addWidget(distance_);
    onlyOnRoute_ = new QCheckBox(tr("Alert only if poi is on route"));
    onlyOnRoute_->setChecked(Settings::instance().value("alert_only_on_route", true).toBool());

    ButtonBox* buttons = new ButtonBox;
    connect(buttons->addButton(tr("Save"), QDialogButtonBox::AcceptRole), SIGNAL(clicked(bool)), this, SLOT(saveSettings()));

    QHBoxLayout* layout = new QHBoxLayout;
    QVBoxLayout* left = new QVBoxLayout;

    left->addWidget(enabled_);
    left->addLayout(poiLayout);
    left->addLayout(soundSelector_);
    left->addLayout(distance);
    left->addWidget(onlyOnRoute_);

    layout->addLayout(left, Qt::AlignLeft);
    layout->addWidget(buttons);

    setLayout(layout);
}


void PoiSettings::importFile()
{
    if(!poiFileSelector_->importFile(PoiAlerts::getPoiDir(),
                                     "Poi files",
                                     PoiReader::getFormatPattern(),
                                     true))
    {
        qDebug() << "Unable to import";
    }
}

void PoiSettings::loadFiles()
{
    distanceLabel_->setText((tr("Alert distance (%1)").arg(Odometer::getMeterUnit())));
    int speedValue = round(Settings::instance().value("alert_distance", 300).toDouble() * Odometer::getMeterMultiplier());
    distance_->setText(QString::number(speedValue));

    QDir poiDir(PoiAlerts::getPoiDir());

    QString selectedSound = Settings::instance().value("alert_sound", "").toString();
    soundSelector_->load();
    soundSelector_->setValue(selectedSound);

    poiFileSelector_->clear();
    QString selectedPoi = Settings::instance().value("alert_poi_file", "").toString();
    poiFileSelector_->loadFiles(PoiAlerts::getPoiDir(), PoiReader::getFormatPattern());
    poiFileSelector_->selectByValue(selectedPoi);

}

void PoiSettings::setVisible(bool visible)
{
    if(visible)
    {
        loadFiles();
    }

    QDialog::setVisible(visible);
}

void PoiSettings::saveSettings()
{
    if(poiFileSelector_->currentIndex() < 0 && enabled_->isChecked())
    {
        QMaemo5InformationBox::information(this, tr("No poi file selected"));
        return;
    }

    double distance = distance_->text().toInt() / Odometer::getMeterMultiplier();

    Settings::instance().setValue("alert_enabled", enabled_->isChecked());
    Settings::instance().setValue("alert_only_on_route", onlyOnRoute_->isChecked());
    Settings::instance().setValue("alert_distance", distance);
    Settings::instance().setValue("alert_sound", soundSelector_->value());
    Settings::instance().setValue("alert_poi_file", poiFileSelector_->value());

    if(!PoiAlerts::instance().loadConfig())
    {
        QMaemo5InformationBox::information(this, tr("Unable to load poi file: %1.").arg(PoiAlerts::instance().error()),
                                           QMaemo5InformationBox::NoTimeout);
    }
    else
    {
        hide();
    }
}
