/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtGui/QApplication>
#include <QtGui/QDesktopWidget>
#include "graphicsscreen.h"
#include "toolbaritem.h"
#include "graphicsscene.h"
#include "odometer.h"

namespace
{
    int const PADDING = 6;
}

GraphicsScreen::GraphicsScreen(QWidget* parent): QGraphicsView(parent),
AbstractScreen(), scene_(0), isFlipped_(false)
{
    QRect rect = QApplication::desktop()->availableGeometry();

    scene_ = new GraphicsScene(QRect(QPoint(0, 0), rect.size()));
    setScene(scene_);

    connect(scene_, SIGNAL(clicked()), this, SIGNAL(clicked()));

    minimizeButton_ = new ToolbarItem(":/resources/minimize.png", ":/resources/minimize_active.png");
    minimizeButton_->setZValue(9999);
    scene_->addItem(minimizeButton_);
    connect(minimizeButton_, SIGNAL(clicked()), this, SIGNAL(minimizePressed()));

    imageWidth_ = minimizeButton_->pixmap().width();

    closeButton_ = new ToolbarItem(":/resources/close.png", ":/resources/close_active.png");
    closeButton_->setZValue(9999);
    scene_->addItem(closeButton_);
    connect(closeButton_, SIGNAL(clicked()), this, SIGNAL(closePressed()));

    settingsButton_ = new ToolbarItem(":/resources/settings.png", ":/resources/settings_active.png");
    settingsButton_->setZValue(9999);
    scene_->addItem(settingsButton_);
    connect(settingsButton_, SIGNAL(clicked()), this, SIGNAL(settingsPressed()));

    reArrange();

    connect(&(Odometer::instance()), SIGNAL(dataUpdated()), this, SLOT(update()));
    connect(&(Odometer::instance()), SIGNAL(unitChanged()), this, SLOT(update()));
}

GraphicsScreen::~GraphicsScreen()
{
}

void GraphicsScreen::reArrange()
{
    QRect rect = QApplication::desktop()->screenGeometry();
    scene_->setSceneRect(rect);
    minimizeButton_->setPos(PADDING, PADDING);
    closeButton_->setPos(rect.width() - imageWidth_ - PADDING, PADDING);
    settingsButton_->setPos((rect.width() / 2) - (imageWidth_ / 2), PADDING);
}

void GraphicsScreen::flip()
{
    if(isFlipped_)
    {
        resetMatrix();
        isFlipped_  = false;
    }
    else
    {
        setTransform(QTransform(1, 0, 0, 0, -1, 0, 0, 0, 1));
        isFlipped_ = true;
    }
}

GraphicsScene* GraphicsScreen::getScene() const
{
    return scene_;
}

void GraphicsScreen::forceRepaint()
{
    QApplication::processEvents();
    update();
    QApplication::processEvents();
}
