/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDateTime>
#include <QtCore/QDebug>
#include <QtGui/QLabel>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QMenuBar>
#include "logwindow.h"

LogWindow::LogWindow(QWidget* parent): QMainWindow(parent), list_(0)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowTitle(tr("Incoming call log"));
    menuBar()->addAction(tr("Clear log"), this, SLOT(clearLog()));
}

void LogWindow::setVisible(bool visible)
{
    if(visible)
    {
        loadLogItems();
    }

    QMainWindow::setVisible(visible);
}

void LogWindow::loadLogItems()
{
    if(!list_)
    {
        list_ = new ListWidget(this);
        setCentralWidget(list_);
        connect(list_, SIGNAL(itemClicked(int)), this,
                SLOT(itemClicked(int)));
    }
    else
    {
        list_->clear();
    }

    QList<Cache::LogDetails> logList;

    Cache::instance().getLogItems(logList, LOG_MAX_ITEMS);

    if(logList.size() == 0)
    {
        QLabel* info = new QLabel(tr("There are currently no logged calls"));
        info->setAlignment(Qt::AlignCenter);
        setCentralWidget(info);
        list_ = 0;
    }
    else
    {
        for(int i = 0; i < logList.size(); i++)
        {
            QMap <QString, QVariant> data;
            data["name"] = QVariant(logList.at(i).result.name);
            data["street"] = QVariant(logList.at(i).result.street);
            data["city"] = QVariant(logList.at(i).result.city);
            data["number"] = QVariant(logList.at(i).result.number);
            data["country"] = QVariant(logList.at(i).result.country);

            list_->addWidget(createWidget(logList.at(i)), data);
        }
    }
}

void LogWindow::itemClicked(int index)
{
    if(!list_)
    {
        return;
    }

    QMap <QString, QVariant> data = list_->getData(index).toMap();
    Source::Result details;
    details.name = data["name"].toString();

    if(details.name.isEmpty())
    {
        return;
    }

    details.street = data["street"].toString();
    details.city = data["city"].toString();
    details.number = data["number"].toString();
    details.country = data["country"].toString();

    emit logItemSelected(details);
}

QWidget* LogWindow::createWidget(Cache::LogDetails const& details)
{
    QWidget* widget = new QWidget;
    QHBoxLayout* layout = new QHBoxLayout;

    QIcon icon;

    if(details.missed)
    {
        icon = QIcon::fromTheme("general_missed");
    }
    else
    {
        icon = QIcon::fromTheme("general_received");
    }

    QLabel* label = new QLabel;
    label->setPixmap(icon.pixmap(48, 48));

    layout->addWidget(label);

    QVBoxLayout* content = new QVBoxLayout;

    QString text;

    if(details.result.name.isEmpty())
    {
        text = tr("%1 (no search results)").arg(details.result.number);
    }
    else
    {
        text = details.result.name;

        if(!details.result.street.isEmpty())
        {
            text += ", " + details.result.street;
        }

        if(!details.result.city.isEmpty())
        {
            text += ", " + details.result.city;
        }
    }

    QLabel* nameLabel = new QLabel(text);

    QDateTime date = QDateTime::fromTime_t(details.time);
    QLabel* dateLabel = new QLabel(date.toString(Qt::SystemLocaleShortDate));
    QFont font;
    font.setPointSize(11);
    dateLabel->setFont(font);

    content->addWidget(nameLabel);
    content->addWidget(dateLabel);

    layout->addLayout(content, Qt::AlignLeft);

    widget->setLayout(layout);

    return widget;
}

void LogWindow::clearLog()
{
    Cache::instance().clearLog();

    loadLogItems();
}
