/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QMenuBar>
#include <QtGui/QLabel>
#include <QtGui/QPushButton>
#include <QtGui/QWidget>
#include <QtGui/QHBoxLayout>
#include <QtGui/QMessageBox>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QTextEdit>
#include <QtCore/QDebug>
#include <QMaemo5InformationBox>
#include "mainwindow.h"
#include "settingsdialog.h"
#include "searchdialog.h"
#include "daemon.h"
#include "settings.h"
#include "db.h"
#include "source.h"
#include "sourcecoreconfig.h"

namespace
{
    const QString START_ICON = ":/icons/start.png";
    const QString CLOSE_ICON = ":/icons/stop.png";
}

MainWindow::MainWindow(QWidget* parent): QMainWindow(parent),
searchResults_(0), settingsDialog_(0), running_(false),
toggleButton_(0), searchDialog_(0), aboutDialog_(0), warning_(0),
logWindow_(0)
{
    setWindowTitle(tr("Jenirok"));
    setAttribute(Qt::WA_Maemo5StackedWindow);
    QWidget* mainWidget = new QWidget(this);

    if(Daemon::isRunning())
    {
        toggleButton_ = createButton(tr("Stop daemon"));
        toggleButton_->setIcon(QIcon(CLOSE_ICON));
        running_ = true;
    }
    else
    {
        toggleButton_ = createButton(tr("Start daemon"));
        toggleButton_->setIcon(QIcon(START_ICON));
        running_ = false;
    }

    QSizePolicy policy;

    policy.setHorizontalPolicy(QSizePolicy::Preferred);

    toggleButton_->setSizePolicy(policy);

    QToolButton* searchButton = createButton(tr("Search"));
    searchButton->setIcon(QIcon::fromTheme("general_search"));
    searchButton->setSizePolicy(policy);

    QToolButton* logButton = createButton(tr("Log"));
    logButton->setIcon(QIcon::fromTheme("general_call"));
    logButton->setSizePolicy(policy);

    QSize size(64, 64);
    searchButton->setIconSize(size);
    toggleButton_->setIconSize(size);
    logButton->setIconSize(size);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(toggleButton_, Qt::AlignHCenter);
    buttonLayout->addWidget(searchButton, Qt::AlignHCenter);
    buttonLayout->addWidget(logButton, Qt::AlignHCenter);

    mainWidget->setLayout(buttonLayout);

    connect(toggleButton_, SIGNAL(pressed()), this, SLOT(toggleDaemon()));
    connect(searchButton, SIGNAL(pressed()), this, SLOT(openSearch()));
    connect(logButton, SIGNAL(pressed()), this, SLOT(openLog()));

    setCentralWidget(mainWidget);
    menuBar()->addAction(tr("Settings"), this, SLOT(showSettings()));
    menuBar()->addAction(tr("About"), this, SLOT(showAbout()));
}

MainWindow::~MainWindow()
{
    DB::removeDatabase();
}

void MainWindow::showSettings()
{
    if(warning_ && warning_->isVisible())
    {
        warning_->hide();
    }

    if(!settingsDialog_)
    {
        settingsDialog_ = new SettingsDialog(this);

        if(searchDialog_)
        {
            connect(settingsDialog_, SIGNAL(saved()), searchDialog_, SLOT(loadSearchTypes()));
        }
    }

    settingsDialog_->show();
}

void MainWindow::toggleDaemon()
{
    QString readyText;
    QString failText;
    QString buttonText;
    bool ret = false;

    if(running_)
    {
        readyText = tr("Daemon was successfully stopped.");
        failText = tr("Unable to stop daemon.");
        buttonText = tr("Start daemon");
        ret = Daemon::stop();
    }
    else
    {
        if(Settings::instance()->getConnectionType() == Settings::ALWAYS_ASK)
        {
            if(!warning_)
            {
                warning_ = new QDialog(this);
                warning_->setWindowTitle(tr("Unable to start daemon"));
                QHBoxLayout* warningLayout = new QHBoxLayout;
                QTextEdit* text = new QTextEdit(tr("Daemon cannot be started because it's not allowed to connect to the Internet. You have to either allow automatic Internet connection in Jenirok settings or in global Maemo settings."));
                text->setReadOnly(true);
                QDialogButtonBox* buttons = new QDialogButtonBox;
                buttons->setOrientation(Qt::Vertical);
                QPushButton* settingsButton = new QPushButton(tr("Open settings"));
                connect(settingsButton, SIGNAL(pressed()), this, SLOT(showSettings()));
                QPushButton* okButton = new QPushButton(tr("Close"));
                connect(okButton, SIGNAL(pressed()), warning_, SLOT(hide()));
                buttons->addButton(settingsButton, QDialogButtonBox::YesRole);
                buttons->addButton(okButton, QDialogButtonBox::AcceptRole);
                warningLayout->addWidget(text);
                warningLayout->addWidget(buttons);
                warning_->setLayout(warningLayout);
            }

            warning_->show();

            return;
        }

        readyText = tr("Daemon was successfully started.");
        failText = tr("Unable to start daemon.");
        buttonText = tr("Stop daemon");
        ret = Daemon::start();
    }

    if(!ret)
    {
        QMessageBox::critical(this, tr("Error"), failText);
    }
    else
    {
        QMaemo5InformationBox::information(this, readyText, 800);
        toggleButton_->setText(buttonText);
        toggleButton_->setIcon(QIcon(running_ ? START_ICON : CLOSE_ICON));
        running_ = !running_;
    }

}

void MainWindow::openSearch()
{
    Source::SourceId sourceId = Source::stringToId(Settings::instance()->get("source"));
    SourceCoreConfig* config = SourceCoreConfig::getCoreConfig(sourceId);

    Q_ASSERT(config != 0);

    bool readyToSearch = config->readyToSearch();
    delete config;

    if(!readyToSearch)
    {
        QMessageBox::information(this, tr("Info"), tr("You need to set login details or other options in settings before using this feature."));
        return;
    }

    if(!searchDialog_)
    {
        searchDialog_ = new SearchDialog(this);
        connect(searchDialog_, SIGNAL(search(SearchDialog::SearchDetails&)),
                this, SLOT(handleSearch(SearchDialog::SearchDetails&)));

        if(settingsDialog_)
        {
            connect(settingsDialog_, SIGNAL(saved()), searchDialog_, SLOT(loadSearchTypes()));
        }
    }

    searchDialog_->show();
}

void MainWindow::openLog()
{
    if(!logWindow_)
    {
        logWindow_ = new LogWindow(this);
        connect(logWindow_, SIGNAL(logItemSelected(Source::Result const&)), this, SIGNAL(logItemSelected(Source::Result const&)));
    }

    logWindow_->show();
}

QToolButton* MainWindow::createButton(QString const& text)
{
    QToolButton* button = new QToolButton();
    button->setText(text);
    button->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
    return button;
}

void MainWindow::handleSearch(SearchDialog::SearchDetails& details)
{
    emit search(details);
}

void MainWindow::showAbout()
{
    if(!aboutDialog_)
    {
        aboutDialog_ = new AboutDialog(this);
    }

    aboutDialog_->show();

}
