/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtGui/QLabel>
#include <QtGui/QPushButton>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QIntValidator>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QTabWidget>
#include <QMaemo5ValueButton>
#include <QMaemo5InformationBox>
#include "settingsdialog.h"
#include "settings.h"
#include "db.h"
#include "daemon.h"
#include "cache.h"
#include "buttonselector.h"
#include "connectionselector.h"
#include "sourceguiconfig.h"

QList<Source::SourceDetails> SettingsDialog::sources_;

SettingsDialog::SettingsDialog(QWidget* parent): QDialog(parent),
sourceConfig_(0), cacheInput_(0), sourceSelector_(0),
autostartSelector_(0), connectionSelector_(0), tabs_(0)
{
    setWindowTitle(tr("Settings"));

    QVBoxLayout* general = new QVBoxLayout;
    QVBoxLayout* daemon = new QVBoxLayout;
    QHBoxLayout* mainLayout = new QHBoxLayout;

    QHBoxLayout* cache = new QHBoxLayout;

    currentSource_ = Settings::instance()->get("source");
    Source::SourceId sourceId = Source::stringToId(currentSource_);
    sourceConfig_ = SourceGuiConfig::getGuiConfig(sourceId, this);

    Q_ASSERT(sourceConfig_ != 0);

    QLabel* cacheLabel = new QLabel(tr("Cache size (numbers)"));
    cacheInput_ = new QLineEdit(Settings::instance()->get("cache_size"));
    cacheInput_->setValidator(new QIntValidator(0, 10000, this));
    QPushButton* cacheResetButton = new QPushButton(tr("Clear"), this);
    connect(cacheResetButton, SIGNAL(pressed()), this, SLOT(resetCache()));

    languageSelector_ = new ButtonSelector(tr("Language"), this);
    languageSelector_->addItem(tr("Automatic"), "");

    selectedLanguage_ = Settings::instance()->get("language");

    QList<Settings::Language> langs;
    Settings::getLanguages(langs);

    for(int i = 0; i < langs.size(); i++)
    {
        languageSelector_->addItem(langs.at(i).name, langs.at(i).id);

        if(langs.at(i).id == selectedLanguage_)
        {
            languageSelector_->setCurrentIndex(i + 1);
        }
    }

    sourceSelector_ = new ButtonSelector(tr("Phonebook"), this);

    if(sources_.isEmpty())
    {
        Source::getSources(sources_);
    }

    for(int i = 0; i < sources_.size(); i++)
    {
        sourceSelector_->addItem(sources_.at(i).name, sources_.at(i).id);

        if(sources_.at(i).id == currentSource_)
        {
            sourceSelector_->setCurrentIndex(i);
        }
    }

    connect(sourceSelector_, SIGNAL(selected(unsigned int, QString const&, QVariant const&)),
            this, SLOT(onSourceSelected(unsigned int, QString const&, QVariant const&)));

    autostartSelector_ = new ButtonSelector(tr("Autostart"), this);
    QString autostart = Settings::instance()->get("autostart");
    autostartSelector_->addItem(tr("Enabled"), "1");
    autostartSelector_->addItem(tr("Disabled"), "0");
    autostartSelector_->setCurrentIndex(autostart == "1" ? 0 : 1);

    connectionSelector_ = new ConnectionSelector(tr("Connect automatically on"), this);
    QString selectedConnection = Settings::instance()->get("connection");

    if(connectionSelector_->selectByValue(selectedConnection) &&
       selectedConnection == "gprs")
    {
        connectionSelector_->updateConnections();
    }

    QPushButton* submitButton = new QPushButton(tr("Save"), this);
    connect(submitButton, SIGNAL(pressed()), this, SLOT(saveSettings()));

    cache->addWidget(cacheLabel);
    cache->addWidget(cacheInput_);
    cache->addWidget(cacheResetButton);
    general->addLayout(cache);
    general->addWidget(languageSelector_);
    general->addWidget(sourceSelector_);

    daemon->addWidget(autostartSelector_);
    daemon->addWidget(connectionSelector_);

    QDialogButtonBox* buttons = new QDialogButtonBox;
    buttons->setCenterButtons(false);
    buttons->addButton(submitButton, QDialogButtonBox::AcceptRole);

    tabs_ = new QTabWidget;

    QWidget* generalTab = new QWidget;
    generalTab->setLayout(general);

    QWidget* daemonTab = new QWidget;
    daemonTab->setLayout(daemon);

    tabs_->addTab(generalTab, tr("General"));
    tabs_->addTab(daemonTab, tr("Daemon"));
    tabs_->addTab(sourceConfig_, tr("Phonebook settings"));

    mainLayout->addWidget(tabs_);
    mainLayout->addWidget(buttons);

    setLayout(mainLayout);

}

void SettingsDialog::saveSettings()
{
    hide();

    Settings::instance()->startEdit();

    Settings::instance()->set("cache_size", cacheInput_->text());
    QString source = sourceSelector_->value().toString();
    Settings::instance()->set("source", source);
    QString autostart = autostartSelector_->value().toString();
    Settings::instance()->set("autostart", autostart);
    QString connection = connectionSelector_->value().toString();
    Settings::instance()->set("connection", connection);
    QString language = languageSelector_->value().toString();
    Settings::instance()->set("language", language);
    sourceConfig_->save();

    Settings::instance()->endEdit();

    bool infoboxShown = false;

    if(language != selectedLanguage_)
    {
        QMaemo5InformationBox::information(this, tr("You need to restart Jenirok for language change to take effect."));
        selectedLanguage_ = language;
        infoboxShown = true;
    }

    if(Daemon::isRunning())
    {
        if(!infoboxShown)
        {
            QMaemo5InformationBox::information(this, tr("Restarting daemon..."), 1500);
        }

        Daemon::restart();
    }

}

void SettingsDialog::resetCache()
{
    int ret = Cache::instance().clear();

    if(ret >= 0)
    {
        QMaemo5InformationBox::information(this, tr("%n number(s) were deleted from cache", "", ret));
    }
}

void SettingsDialog::onSourceSelected(unsigned int index,
                                      QString const& text,
                                      QVariant const& value)
{
    Q_UNUSED(index);
    Q_UNUSED(text);

    QString source = value.toString();

    if(source != currentSource_)
    {
        int tabId = tabs_->indexOf(sourceConfig_);

        if(tabId >= 0)
        {
            tabs_->removeTab(tabId);
        }

        delete sourceConfig_;
        Source::SourceId sourceId = Source::stringToId(value.toString());
        sourceConfig_ = SourceGuiConfig::getGuiConfig(sourceId, this);
        Q_ASSERT(sourceConfig_ != 0);
        tabs_->addTab(sourceConfig_, tr("Phonebook settings"));
        currentSource_ = source;
    }
}
