/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtCore/QVariant>
#include <QtCore/QString>
#include <QtGui/QLabel>
#include <QtGui/QListWidgetItem>
#include <QtGui/QMessageBox>
#include "resultwindow.h"
#include "settings.h"
#include "db.h"
#include "cache.h"
#include "source.h"
#include "sourcecoreconfig.h"

ResultWindow::ResultWindow(QWidget* parent): QMainWindow(parent),
source_(0), list_(0), connectionManager_(0)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowTitle(tr("Search results"));
}

ResultWindow::~ResultWindow()
{
    delete connectionManager_;
    connectionManager_ = 0;
}

void ResultWindow::search(SearchDialog::SearchDetails& details)
{
    if(!list_)
    {
        list_ = new QListWidget(this);
        setCentralWidget(list_);
        connect(list_, SIGNAL(itemClicked(QListWidgetItem*)), this,
                SLOT(itemClicked(QListWidgetItem*)));
    }
    else
    {
        list_->clear();
    }

    Source::SourceId id = Source::stringToId(Settings::instance()->get("source"));

    if(!source_ || id != sourceId_)
    {
        sourceId_ = id;

        if(source_)
        {
            delete source_;
            source_ = 0;
        }

        source_ = Source::getSource(sourceId_);
        Q_ASSERT(source_ != 0);
        source_->setTimeout(REQUEST_TIMEOUT);

        connect(source_, SIGNAL(resultAvailable(Source::Result const&,
                                               Source::SearchDetails const&)),
                                               this, SLOT(resultAvailable(Source::Result const&,
                                                                          Source::SearchDetails const&)));

        connect(source_, SIGNAL(requestFinished(QVector <Source::Result> const&,
                                               Source::SearchDetails const&, bool)),
                                               this, SLOT(requestFinished(QVector <Source::Result> const&,
                                                                          Source::SearchDetails const&, bool)));
    }

    SourceCoreConfig* config = SourceCoreConfig::getCoreConfig(sourceId_);

    Q_ASSERT(config != 0);

    config->apply(source_);
    delete config;

    Source::SearchType type;

    switch(details.type)
    {
    case 0:
        type = Source::PERSONS;
        break;
    case 1:
        type = Source::YELLOW_PAGES;
        break;
    default:
        qDebug() << "Unknown search type: " << details.type;
        return;
    }

    show();
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);

    if(!connectionManager_)
    {
        connectionManager_ = new ConnectionManager();
    }

    connectionManager_->connect();

    source_->abort();
    source_->search(Source::SearchDetails(details.name, details.location, type));

}

void ResultWindow::resultAvailable(Source::Result const& result,
                                   Source::SearchDetails const& details)
{
    Q_UNUSED(details);

    if(!result.number.isEmpty())
    {
        Cache::instance().addItem(result);
    }

    QString row = result.name;

    if(!result.street.isEmpty())
    {
        row += ", " + result.street;
    }

    if(!result.city.isEmpty())
    {
        row += ", " + result.city;
    }

    QListWidgetItem* item = new QListWidgetItem(row, list_);
    QMap <QString, QVariant> data;
    data["name"] = QVariant(result.name);
    data["street"] = QVariant(result.street);
    data["city"] = QVariant(result.city);
    data["number"] = QVariant(result.number);

    item->setData(Qt::UserRole, QVariant(data));

    list_->addItem(item);
}

void ResultWindow::requestFinished(QVector <Source::Result> const& results,
                                   Source::SearchDetails const& details,
                                   bool error)
{
    Q_UNUSED(details);

    if(error)
    {
        QString errorString;
        Source::Error error = source_->error();

        switch(error)
        {
        case Source::CONNECTION_FAILURE:
            errorString = tr("Connection to server failed");
            break;
        case Source::INVALID_LOGIN:
            errorString = tr("Invalid login details");
            break;
        case Source::TIMEOUT:
            errorString = tr("Request timed out");
            break;
        default:
            errorString = tr("Searching failed:") + " " + source_->errorString();
            break;
        }

        QMessageBox::critical(this, tr("Error"), errorString);
    }

    if(results.size() == 0)
    {
        QLabel* info = new QLabel(tr("No results found"));
        info->setAlignment(Qt::AlignCenter);
        setCentralWidget(info);
        list_ = 0;
    }

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

}

void ResultWindow::itemClicked(QListWidgetItem* item)
{
    QMap <QString, QVariant> data = item->data(Qt::UserRole).toMap();
    Source::Result details;
    details.name = data["name"].toString();
    details.street = data["street"].toString();
    details.city = data["city"].toString();
    details.number = data["number"].toString();

    emit itemSelected(details);
}

void ResultWindow::setVisible(bool visible)
{
    QMainWindow::setVisible(visible);

    if(!visible && source_)
    {
        source_->abort();


    }
}



