/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#include <QtGui/QMessageBox>
#include <QtGui/QLabel>
#include <QtGui/QClipboard>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QApplication>
#include <QMaemo5ValueButton>
#include <QMaemo5InformationBox>
#include "detailwindow.h"
#include "contactmanager.h"

DetailWindow::DetailWindow(QWidget* parent): QMainWindow(parent), addDialog_(0)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    area_ = new QWidget(this);
    layout_ = new QVBoxLayout;
    QHBoxLayout* top = new QHBoxLayout;
    QHBoxLayout* bottom = new QHBoxLayout;
    QHBoxLayout* actions1 = new QHBoxLayout;
    QHBoxLayout* actions2 = new QHBoxLayout;

    QPushButton* addButton = new QPushButton(QIcon::fromTheme("general_contacts"), tr("Add to contacts"));
    QPushButton* copyButton = new QPushButton(tr("Copy number to clipboard"));
    QPushButton* callButton = new QPushButton(QIcon::fromTheme("general_call"), tr("Call"));
    QPushButton* smsButton = new QPushButton(QIcon::fromTheme("general_sms"), tr("Send SMS"));

    connect(addButton, SIGNAL(pressed()), this, SLOT(showAddToContactsDialog()));
    connect(copyButton, SIGNAL(pressed()), this, SLOT(copyToClipboard()));
    connect(callButton, SIGNAL(pressed()), this, SLOT(makeCall()));
    connect(smsButton, SIGNAL(pressed()), this, SLOT(sendSMS()));

    nameButton_ = new QMaemo5ValueButton(QIcon::fromTheme("general_default_avatar"),
                                         tr("Name"), this);
    streetButton_ = new QMaemo5ValueButton(tr("Street"), this);
    cityButton_ = new QMaemo5ValueButton(tr("City"), this);
    numberButton_ = new QMaemo5ValueButton(QIcon::fromTheme("general_call"),
                                           tr("Phone number"), this);

    connect(numberButton_, SIGNAL(pressed()), this, SLOT(makeCall()));

    top->addWidget(nameButton_);
    bottom->addWidget(streetButton_);
    top->addWidget(numberButton_);
    bottom->addWidget(cityButton_);
    actions1->addWidget(callButton);
    actions1->addWidget(smsButton);
    actions2->addWidget(addButton);
    actions2->addWidget(copyButton);
    layout_->addLayout(top);
    layout_->addLayout(bottom);
    layout_->addLayout(actions1);
    layout_->addLayout(actions2);
    area_->setLayout(layout_);
    setCentralWidget(area_);
}

void DetailWindow::loadData(Source::Result const& details)
{
    setWindowTitle(details.name);
    nameButton_->setValueText(details.name);
    streetButton_->setValueText(details.street);
    cityButton_->setValueText(details.city);
    numberButton_->setValueText(details.number);
    show();
}

void DetailWindow::makeCall()
{
    QString number = numberButton_->valueText();

    if(number.isEmpty())
    {
        return;
    }

    QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.csd",
                                                      "/com/nokia/csd/call",
                                                      "com.nokia.csd.Call",
                                                      "CreateWith");
    QList<QVariant> arguments;

    arguments.append(QVariant(number));
    arguments.append(QVariant(0));

    msg.setArguments(arguments);

    if(!QDBusConnection::systemBus().send(msg))
    {
        QMessageBox::critical(this, tr("Error"), tr("Unable make call"));
    }

}

void DetailWindow::showAddToContactsDialog()
{
    if(!addDialog_)
    {
        addDialog_ = new QDialog(this);
        addDialog_->setWindowTitle(tr("Add to contacts"));
        addContactInput_ = new QLineEdit();
        QHBoxLayout* layout = new QHBoxLayout();
        QLabel* name = new QLabel(tr("Name"));
        QPushButton* button = new QPushButton(tr("Add"));

        QDialogButtonBox* buttons = new QDialogButtonBox;
        buttons->setCenterButtons(false);
        buttons->addButton(button, QDialogButtonBox::AcceptRole);
        connect(button, SIGNAL(pressed()), this, SLOT(addToContacts()));

        QHBoxLayout* left = new QHBoxLayout();
        left->addWidget(name);
        left->addWidget(addContactInput_);

        layout->addLayout(left, Qt::AlignLeft);
        layout->addWidget(buttons);
        addDialog_->setLayout(layout);
    }

    addContactInput_->setText(nameButton_->valueText());
    addDialog_->show();
}

void DetailWindow::addToContacts()
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);

    ContactManager cm;
    ContactManager::Contact contact;
    contact.name = addContactInput_->text();
    contact.number = numberButton_->valueText();

    addDialog_->hide();

    if(cm.addContact(contact))
    {
        QMaemo5InformationBox::information(this, tr("Contact was successfully added to contacts."));
    }
    else
    {
        QMessageBox::critical(this, tr("Error"), tr("Unable to add contact."));
    }

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

}

void DetailWindow::copyToClipboard()
{
    QApplication::clipboard()->setText(numberButton_->valueText());
    QMaemo5InformationBox::information(this, tr("Number was successfully copied to clipboard."));
}

void DetailWindow::sendSMS()
{
    QString number = numberButton_->valueText();

    if(number.isEmpty())
    {
        return;
    }

    QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.MessagingUI",
                                                      "/com/nokia/MessagingUI",
                                                      "com.nokia.MessagingUI",
                                                      "messaging_ui_interface_start_sms");
    QList<QVariant> arguments;

    arguments.append(QVariant("sms:" + number));

    msg.setArguments(arguments);

    if(!QDBusConnection::systemBus().send(msg))
    {
        QMessageBox::critical(this, tr("Error"), tr("Unable to open SMS application"));
    }

}
