/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef SOURCE_H
#define SOURCE_H

#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QTimerEvent>
#include <QtCore/QUrl>
#include <QtCore/QByteArray>
#include <QtNetwork/QHttp>

class Source : public QObject
{

    Q_OBJECT

public:

    struct Result
    {
        QString name;
        QString street;
        QString city;
        QString number;
    };

    enum SearchType {YELLOW_PAGES, PERSONS, BOTH};

    struct SearchDetails
    {
        QString query;
        QString location;
        SearchType type;
        SearchDetails(QString const& query = "",
                      QString const& location = "",
                      SearchType type = PERSONS);
    };


    enum Error {NO_ERROR, CONNECTION_FAILURE, INVALID_LOGIN, TIMEOUT};
    enum SourceId {ENIRO, MOBIL1881};
    static int const SOURCE_COUNT = 2;

    struct SourceDetails
    {
        SourceId type;
        QString name;
        QString id;
    };

    static unsigned int const DEFAULT_MAX_RESULTS = 30;

    static Source* getSource(SourceId id, QObject* parent = 0);
    static void getSources(QList<SourceDetails>& list);
    static SourceId stringToId(QString const& str);
    Source(QObject* parent = 0);
    virtual ~Source();
    static Source* getSource();
    virtual void abort();
    virtual void search(SearchDetails const& details) = 0;
    void setMaxResults(unsigned int results);
    unsigned int getMaxResults() const;
    void setTimeout(unsigned int ms);
    unsigned int getTimeout() const;
    void resetTimeout();
    void setFindNumber(bool value);
    bool getFindNumber() const;
    Error error() const;
    const QString& errorString() const;

signals:
    void resultAvailable(Source::Result const& result, Source::SearchDetails const& details);
    void requestFinished(QVector <Source::Result> const& results, Source::SearchDetails const& details, bool error);

protected:
    void setError(Error error, QString const& errorString = "");
    virtual void timerEvent(QTimerEvent *te);
    static QString ucFirst(QString& string);
    static QString& cleanUpNumber(QString& number);
    static QString& stripTags(QString& string);
    static QString& htmlEntityDecode(QString& string);
    static bool isPhoneNumber(QString const& string);
    void fixUrl(QUrl& url);
    QHttp http_;

private slots:
    void httpReady(int id, bool error);

private:
    Q_DISABLE_COPY(Source);
    virtual void handleHttpData(int id, QByteArray const& data) = 0;
    virtual void handleHttpError(int id) = 0;
    unsigned int maxResults_;
    unsigned int timeout_;
    int timerId_;
    bool findNumber_;
    Error error_;
    QString errorString_;
    QString username_;
    QString password_;
    bool loggedIn_;

    static QRegExp numberCleaner_;
    static QRegExp tagStripper_;

};

#endif
