/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include <QtNetwork/QHttpResponseHeader>
#include "source.h"
#include "eniro.h"
#include "mobil1881.h"
#include "dastelefonbuch.h"

namespace
{
    static const QString SOURCE_NAMES[Source::SOURCE_COUNT] =
    {
         "Eniro (FI, SE, DK)",
         "1881 Mobil (NO)",
         "Das Telefonbuch (DE)"
    };

    static const QString SOURCE_IDS[Source::SOURCE_COUNT] =
    {
         "eniro",
         "1881mobil",
         "dastelefonbuch"
    };

}

// Regexp used to remove everything except numbers from string
QRegExp Source::numberCleaner_ = QRegExp("([^0-9]+)");

// Removes html tags from string
QRegExp Source::tagStripper_ = QRegExp("<([^>]+)>");

Source* Source::getSource(Source::SourceId id, QObject* parent)
{
    switch(id)
    {
    case ENIRO:
        return new Eniro(parent);
        break;
    case MOBIL1881:
        return new Mobil1881(parent);
        break;
    case DASTELEFONBUCH:
        return new DasTelefonbuch(parent);
        break;
    default:
        qDebug() << "Unknown source:" << id;
    }

    return 0;
}

Source::SourceId Source::stringToId(QString const& str)
{
    for(int i = 0; i < SOURCE_COUNT; i++)
    {
        if(SOURCE_IDS[i] == str || SOURCE_NAMES[i] == str)
        {
            return static_cast<SourceId>(i);
        }
    }

    return ENIRO;
}

void Source::getSources(QList<SourceDetails>& list)
{
    for(int i = 0; i < SOURCE_COUNT; i++)
    {
        SourceDetails details;
        details.type = static_cast<SourceId>(i);
        details.name = SOURCE_NAMES[i];
        details.id = SOURCE_IDS[i];
        list.push_back(details);
    }
}

Source::Source(QObject* parent): QObject(parent),
maxResults_(DEFAULT_MAX_RESULTS), timeout_(0), timerId_(0), findNumber_(true),
error_(NO_ERROR), loggedIn_(false)
{
    connect(&http_, SIGNAL(requestFinished(int, bool)), this, SLOT(httpReady(int, bool)));
}

Source::~Source()
{
    abort();
}

void Source::abort()
{
    http_.abort();
}

void Source::setMaxResults(unsigned int results)
{
    maxResults_ = results;
}

unsigned int Source::getMaxResults() const
{
    return maxResults_;
}

void Source::getSearchTypes(QList<SearchType>& types) const
{
    types.clear();
    types.push_back(PERSONS);
    types.push_back(YELLOW_PAGES);
}

void Source::setTimeout(unsigned int timeout)
{
    timeout_ = timeout;
    resetTimeout();
}

unsigned int Source::getTimeout() const
{
    return timeout_;
}

void Source::resetTimeout()
{
    if(timerId_)
    {
        killTimer(timerId_);
    }
    if(timeout_)
    {
        timerId_ = startTimer(timeout_);
    }
}

void Source::timerEvent(QTimerEvent* t)
{
    Q_UNUSED(t);
}

void Source::setFindNumber(bool value)
{
    findNumber_ = value;
}

bool Source::getFindNumber() const
{
    return findNumber_;
}

Source::Error Source::error() const
{
    return error_;
}

const QString& Source::errorString() const
{
    return errorString_;
}

void Source::setError(Source::Error error, QString const& errorString)
{
    error_ = error;
    errorString_ = errorString;
}

void Source::httpReady(int id, bool error)
{
    QByteArray result = http_.readAll();

    if(error)
    {
        if(http_.error() == QHttp::Aborted)
        {
            return;
        }

        qDebug() << "Error: " << http_.errorString();
        handleHttpError(id);
    }
    else
    {
        handleHttpData(id, result);
    }
}

QString Source::ucFirst(QString& str)
{
    if (str.size() < 1) {
        return "";
    }

    QStringList tokens = str.split(" ");
    QList<QString>::iterator tokItr;

    for (tokItr = tokens.begin(); tokItr != tokens.end(); ++tokItr)
    {
        (*tokItr) = (*tokItr).at(0).toUpper() + (*tokItr).mid(1);
    }

    return tokens.join(" ");
}

QString& Source::cleanUpNumber(QString& number)
{
    return number.replace(numberCleaner_, "");
}

QString& Source::stripTags(QString& string)
{
    return string.replace(tagStripper_, "");
}

QString& Source::htmlEntityDecode(QString& string)
{
    static const QString entities[] =
    {
        "quot",
        "apos",
        "amp",
        "lt",
        "gt",
        "nbsp",
        "Agrave",
        "Aacute",
        "Acirc",
        "Atilde",
        "Auml",
        "Aring",
        "AElig",
        "Ccedil",
        "Egrave",
        "Eacute",
        "Ecirc",
        "Euml",
        "Igrave",
        "Iacute",
        "Icirc",
        "Iuml",
        "ETH",
        "Ntilde",
        "Ograve",
        "Oacute",
        "Ocirc",
        "Otilde",
        "Ouml",
        "Oslash",
        "Ugrave",
        "Uacute",
        "Ucirc",
        "Uuml",
        "Yacute",
        "THORN",
        "szlig",
        "agrave",
        "aacute",
        "acirc",
        "atilde",
        "auml",
        "aring",
        "aelig",
        "ccedil",
        "egrave",
        "eacute",
        "ecirc",
        "euml",
        "igrave",
        "iacute",
        "icirc",
        "iuml",
        "eth",
        "ntilde",
        "ograve",
        "oacute",
        "ocirc",
        "otilde",
        "ouml",
        "oslash",
        "ugrave",
        "uacute",
        "ucirc",
        "uuml",
        "yacute",
        "thorn",
        "yuml"
    };

    static const int entityValues[] =
    {
        34,
        39,
        38,
        60,
        62,
        32,
        192,
        193,
        194,
        195,
        196,
        197,
        198,
        199,
        200,
        201,
        202,
        203,
        204,
        205,
        206,
        207,
        208,
        209,
        210,
        211,
        212,
        213,
        214,
        216,
        217,
        218,
        219,
        220,
        221,
        222,
        223,
        224,
        225,
        226,
        227,
        228,
        229,
        230,
        231,
        232,
        233,
        234,
        235,
        236,
        237,
        238,
        239,
        240,
        241,
        242,
        243,
        244,
        245,
        246,
        248,
        249,
        250,
        251,
        252,
        253,
        254,
        255
    };

    static int const COUNT = sizeof(entityValues) / sizeof(entityValues[0]);

    for(int i = 0; i < COUNT; i++)
    {
        string = string.replace("&" + entities[i] + ";", QChar(entityValues[i]));
    }

    static QRegExp entityCleaner("&#([0-9]{1,3});");
    entityCleaner.setMinimal(true);

    int pos = 0;

    while((pos = entityCleaner.indexIn(string, pos)) != -1)
    {
        QString match = entityCleaner.cap(1);

        int value = match.toInt();

        if(value >= 1 && value <= 255)
        {
            string = string.replace(pos, match.length() + 3, QChar(value));
        }

        pos += entityCleaner.matchedLength();
    }


   return string;
}

void Source::fixUrl(QUrl& url)
{
    QByteArray path = url.encodedQuery().replace('+', "%2B");
    url.setEncodedQuery(path);
}

bool Source::isPhoneNumber(QString const& string)
{
    static QRegExp check("^([0-9 -]{7,25})$");
    return check.exactMatch(string);
}

Source::SearchDetails::SearchDetails(QString const& q,
                                     QString const& loc,
                                     SearchType t)
{
    query = q;
    location = loc;
    type = t;
}
