/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include "enirocoreconfig.h"
#include "eniro.h"
#include "settings.h"

namespace
{
    QString const COLUMN_PREFIX = "eniro_";
    QString const USERNAME_COLUMN = COLUMN_PREFIX + "username";
    QString const PASSWORD_COLUMN = COLUMN_PREFIX + "password";
    QString const SITE_COLUMN = COLUMN_PREFIX + "site";
}

EniroCoreConfig::EniroCoreConfig(): SourceCoreConfig(), loaded_(false)
{
}

EniroCoreConfig::~EniroCoreConfig()
{
}

bool EniroCoreConfig::apply(Source* source)
{
    Eniro* eniro = dynamic_cast<Eniro*>(source);

    if(!eniro)
    {
        return false;
    }

    load();

    QString username = config_[USERNAME_COLUMN];
    QString password = config_[PASSWORD_COLUMN];

    if(!username.isEmpty() && !password.isEmpty())
    {
        eniro->login(username, password);
    }

    eniro->setSite(Eniro::stringToSite(config_[SITE_COLUMN]));

    return true;
}

void EniroCoreConfig::getConfig(QMap<QString, QString>& config)
{
    load();

    config = config_;
}

void EniroCoreConfig::loadFromConfig(QMap<QString, QString> const& config)
{
    config_[USERNAME_COLUMN] = config[USERNAME_COLUMN];
    config_[PASSWORD_COLUMN] = config[PASSWORD_COLUMN];
    config_[SITE_COLUMN] = config[SITE_COLUMN];

    loaded_ = true;
}

void EniroCoreConfig::store()
{
    Settings::instance()->set(USERNAME_COLUMN, config_[USERNAME_COLUMN]);
    Settings::instance()->set(PASSWORD_COLUMN, config_[PASSWORD_COLUMN]);
    Settings::instance()->set(SITE_COLUMN, config_[SITE_COLUMN]);
}

bool EniroCoreConfig::readyToSearch()
{
    load();

    return !config_[USERNAME_COLUMN].isEmpty() && !config_[PASSWORD_COLUMN].isEmpty();
}

void EniroCoreConfig::setUsername(QString const& username)
{
    config_[USERNAME_COLUMN] = username;
}

void EniroCoreConfig::setPassword(QString const& password)
{
    config_[PASSWORD_COLUMN] = password;
}

void EniroCoreConfig::setSite(QString const& site)
{
    config_[SITE_COLUMN] = site;
}

QString EniroCoreConfig::getUsername() const
{
    return config_[USERNAME_COLUMN];
}

QString EniroCoreConfig::getPassword() const
{
    return config_[PASSWORD_COLUMN];
}

QString EniroCoreConfig::getSite() const
{
    return config_[SITE_COLUMN];
}

void EniroCoreConfig::load()
{
    if(loaded_)
    {
        return;
    }

    Settings::instance()->startEdit();

    config_[USERNAME_COLUMN] = Settings::instance()->get(USERNAME_COLUMN);
    config_[PASSWORD_COLUMN] = Settings::instance()->get(PASSWORD_COLUMN);
    config_[SITE_COLUMN] = Settings::instance()->get(SITE_COLUMN);

    Settings::instance()->endEdit();

    loaded_ = true;
}
