/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QDebug>
#include <QtCore/QVariant>
#include <QtCore/QString>
#include <QtGui/QLabel>
#include <QtGui/QListWidgetItem>
#include <QtGui/QMessageBox>
#include "resultwindow.h"
#include "settings.h"
#include "db.h"
#include "cache.h"

ResultWindow::ResultWindow(QWidget* parent): QMainWindow(parent),
eniro_(0), list_(0), connectionManager_(0)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowTitle(tr("Search results"));
}

ResultWindow::~ResultWindow()
{
    delete connectionManager_;
    connectionManager_ = 0;
}

void ResultWindow::search(SearchDialog::SearchDetails& details)
{
    if(!list_)
    {
        list_ = new QListWidget(this);
        setCentralWidget(list_);
        connect(list_, SIGNAL(itemClicked(QListWidgetItem*)), this,
                SLOT(itemClicked(QListWidgetItem*)));
    }
    else
    {
        list_->clear();
    }

    DB::connect();

    Eniro::Site selectedSite = Eniro::stringToSite(Settings::instance()->get("site"));

    if(!eniro_)
    {
        eniro_ = new Eniro(selectedSite, this);
        eniro_->setTimeout(REQUEST_TIMEOUT);

        connect(eniro_, SIGNAL(resultAvailable(Eniro::Result const&,
                                               Eniro::SearchDetails const&)),
                                               this, SLOT(resultAvailable(Eniro::Result const&,
                                                                          Eniro::SearchDetails const&)));

        connect(eniro_, SIGNAL(requestFinished(QVector <Eniro::Result> const&,
                                               Eniro::SearchDetails const&, bool)),
                                               this, SLOT(requestFinished(QVector <Eniro::Result> const&,
                                                                          Eniro::SearchDetails const&, bool)));
    }

    QString username = Settings::instance()->get("eniro_username");
    QString password = Settings::instance()->get("eniro_password");

    if(!username.isEmpty() && !password.isEmpty())
    {
        eniro_->login(username, password);
    }

    eniro_->setSite(selectedSite);

    DB::disconnect();

    Eniro::SearchType type;

    switch(details.type)
    {
    case 0:
        type = Eniro::PERSONS;
        break;
    case 1:
        type = Eniro::YELLOW_PAGES;
        break;
    default:
        return;
    }

    show();
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);

    if(!connectionManager_)
    {
        connectionManager_ = new ConnectionManager();
    }

    connectionManager_->connect();

    eniro_->abort();
    eniro_->search(Eniro::SearchDetails(details.name, details.location, type));

}

void ResultWindow::resultAvailable(Eniro::Result const& result,
                                   Eniro::SearchDetails const& details)
{
    Q_UNUSED(details);

    if(!result.number.isEmpty())
    {
        Cache::instance().addItem(result);
    }

    QString row = result.name;

    if(!result.street.isEmpty())
    {
        row += ", " + result.street;
    }

    if(!result.city.isEmpty())
    {
        row += ", " + result.city;
    }

    QListWidgetItem* item = new QListWidgetItem(row, list_);
    QMap <QString, QVariant> data;
    data["name"] = QVariant(result.name);
    data["street"] = QVariant(result.street);
    data["city"] = QVariant(result.city);
    data["number"] = QVariant(result.number);

    item->setData(Qt::UserRole, QVariant(data));

    list_->addItem(item);
}

void ResultWindow::requestFinished(QVector <Eniro::Result> const& results,
                                   Eniro::SearchDetails const& details,
                                   bool error)
{
    Q_UNUSED(details);

    if(error)
    {
        QString errorString;
        Eniro::Error error = eniro_->error();

        switch(error)
        {
        case Eniro::CONNECTION_FAILURE:
            errorString = tr("Connection to server failed");
            break;
        case Eniro::INVALID_LOGIN:
            errorString = tr("Invalid login details");
            break;
        case Eniro::TIMEOUT:
            errorString = tr("Request timed out");
            break;
        default:
            errorString = tr("Searching failed:") + " " + eniro_->errorString();
            break;
        }

        QMessageBox::critical(this, tr("Error"), errorString);
    }

    if(results.size() == 0)
    {
        QLabel* info = new QLabel(tr("No results found"));
        info->setAlignment(Qt::AlignCenter);
        setCentralWidget(info);
        list_ = 0;
    }

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

}

void ResultWindow::itemClicked(QListWidgetItem* item)
{
    QMap <QString, QVariant> data = item->data(Qt::UserRole).toMap();
    Eniro::Result details;
    details.name = data["name"].toString();
    details.street = data["street"].toString();
    details.city = data["city"].toString();
    details.number = data["number"].toString();

    emit itemSelected(details);
}


