/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef ENIRO_H
#define ENIRO_H

#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QVector>
#include <QtCore/QSet>
#include <QtCore/QRegExp>
#include <QtCore/QUrl>
#include <QtCore/QTimerEvent>
#include <QtNetwork/QHttp>

class Eniro: public QObject
{
    Q_OBJECT

public:

    enum Site {FI, SE, DK};
    static const int SITE_COUNT = 3;

    enum SearchType {YELLOW_PAGES, PERSONS};

    enum Error {NO_ERROR, CONNECTION_FAILURE, INVALID_LOGIN, TIMEOUT};

    struct Result
    {
        QString name;
        QString street;
        QString city;
        QString number;
    };

    struct SearchDetails
    {
        QString query;
        QString location;
        SearchType type;
        SearchDetails(QString const& query = "",
                      QString const& location = "",
                      SearchType type = PERSONS);
    };

    struct SiteDetails
    {
        QString name;
        QString id;
    };

    static unsigned int const DEFAULT_MAX_RESULTS = 30;

    Eniro(Site site, QObject *parent = 0);

    ~Eniro();

    void login(QString const& username, QString const& password);
    void logout();
    void testLogin();
    void setSite(Site);
    void setMaxResults(unsigned int value);
    void setFindNumber(bool value);
    void setTimeout(unsigned int ms);
    bool search(SearchDetails const& details);
    void abort();
    Error error() const;
    const QString& errorString() const;
    static QMap<Site, SiteDetails> getSites();
    static Site stringToSite(QString const& str);

signals:
    void resultAvailable(Eniro::Result const& result, Eniro::SearchDetails const& details);
    void requestFinished(QVector <Eniro::Result> const& results, Eniro::SearchDetails const& details, bool error);
    void loginStatus(bool success);

private slots:
    void httpReady(int id, bool error);

private:

    Q_DISABLE_COPY(Eniro);

    struct NumberData
    {
        int searchId;
        int index;
    };

    struct SearchData
    {
        SearchDetails details;
        QVector <Result> results;
        unsigned int foundNumbers;
        unsigned int numbersTotal;
    };

    QUrl createUrl(QString const& query, QString const& location);
    void loadResults(int id, QString const& data);
    void loadNumber(int id, QString const& data);
    void getNumberForResult(int id, int index, SearchDetails const& details);
    void emitRequestFinished(int key, SearchData* data, bool error);
    void resetTimeout();
    void timerEvent(QTimerEvent *te);
    QString ucFirst(QString& string);
    QString& cleanUpNumber(QString& number);
    QString& stripTags(QString& string);

    QHttp http_;
    Site site_;
    QString username_;
    QString password_;
    bool loggedIn_;
    Error error_;
    QString errorString_;
    unsigned int maxResults_;
    unsigned int timeout_;
    int timerId_;
    bool findNumber_;
    typedef QMap <int, SearchData*> searchMap;
    typedef QMap <int, NumberData*> numberMap;

    searchMap pendingSearches_;
    numberMap pendingNumberRequests_;
    QSet <int> pendingLoginRequests_;

    static QRegExp numberCleaner_;
    static QRegExp tagStripper_;

};

#endif // ENIRO_H

