/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDebug>
#include "eniro.h"

namespace
{
    static const QString SITE_URLS[Eniro::SITE_COUNT] =
    {
            "http://wap.eniro.fi/",
            "http://wap.eniro.se/",
            "http://wap.eniro.dk/"
    };

    static const QString SITE_NAMES[Eniro::SITE_COUNT] =
    {
         "finnish",
         "swedish",
         "danish"
    };

    static const QString SITE_IDS[Eniro::SITE_COUNT] =
    {
         "fi",
         "se",
         "dk"
    };

    static const QString INVALID_LOGIN_STRING = QObject::tr("Invalid login details");
    static const QString TIMEOUT_STRING = QObject::tr("Request timed out");
    static const QString PERSON_REGEXP = "<td class=\"hTd2\">(.*)<b>(.*)</td>";
    static const QString YELLOW_REGEXP = "<td class=\"hTd2\">(.*)<span class=\"gray\"\\}>(.*)</td>";
    static const QString NUMBER_REGEXP = "<div class=\"callRow\">(.*)</div>";
    static const QString LOGIN_CHECK = "<input class=\"inpTxt\" id=\"loginformUsername\"";
}

// Regexp used to remove numbers from string
QRegExp Eniro::numberCleaner_ = QRegExp("([^0-9]+)");

// Removes html tags from string
QRegExp Eniro::tagStripper_ = QRegExp("<([^>]+)>");

Eniro::Eniro(Site site, QObject *parent): QObject(parent), site_(site),
username_(""), password_(""), loggedIn_(false), error_(NO_ERROR),
errorString_(""), maxResults_(DEFAULT_MAX_RESULTS), timeout_(0), timerId_(0),
findNumber_(true), pendingSearches_(), pendingNumberRequests_()
{
    connect(&http_, SIGNAL(requestFinished(int, bool)), this, SLOT(httpReady(int, bool)));
}

Eniro::~Eniro()
{
    abort();
}

void Eniro::abort()
{
    http_.abort();

    for(searchMap::iterator sit = pendingSearches_.begin();
    sit != pendingSearches_.end(); sit++)
    {
        if(sit.value() != 0)
        {
            delete sit.value();
            sit.value() = 0;
        }
    }

    pendingSearches_.clear();

    for(numberMap::iterator nit = pendingNumberRequests_.begin();
    nit != pendingNumberRequests_.end(); nit++)
    {
        if(nit.value() != 0)
        {
            delete nit.value();
            nit.value() = 0;
        }
    }

    pendingNumberRequests_.clear();
    pendingLoginRequests_.clear();
}

void Eniro::setMaxResults(unsigned int value)
{
    maxResults_ = value;
}

void Eniro::setFindNumber(bool value)
{
    findNumber_ = value;
}

void Eniro::setSite(Eniro::Site site)
{
    site_ = site;
}

void Eniro::setTimeout(unsigned int ms)
{
    timeout_ = ms;
    resetTimeout();
}

void Eniro::resetTimeout()
{
    if(timerId_)
    {
        killTimer(timerId_);
    }
    if(timeout_)
    {
        timerId_ = startTimer(timeout_);
    }
}

void Eniro::timerEvent(QTimerEvent* t)
{
    if(t->timerId() == timerId_)
    {
        int currentId = http_.currentId();

        if(currentId)
        {
            searchMap::const_iterator it = pendingSearches_.find(currentId);

            if(it != pendingSearches_.end())
            {
                QVector <Eniro::Result> results = it.value()->results;
                SearchDetails details = it.value()->details;

                abort();

                error_ = TIMEOUT;
                errorString_ = TIMEOUT_STRING;

                emit requestFinished(results, details, true);
            }
        }

    }
}

void Eniro::login(QString const& username,
                  QString const& password)
{
    username_ = username;
    password_ = password;
    loggedIn_ = true;
}

void Eniro::logout()
{
    username_ = "";
    password_ = "";
    loggedIn_ = false;
}

void Eniro::testLogin()
{
    QUrl url = createUrl("", "");

    url.addQueryItem("what", "mobwp");
    http_.setHost(url.host(), url.port(80));
    int id = http_.get(url.encodedPath() + '?' + url.encodedQuery());

    pendingLoginRequests_.insert(id);
}

bool Eniro::search(SearchDetails const& details)
{
    resetTimeout();

    SearchType type = details.type;

    // Only logged in users can use other than person search
    if(!loggedIn_)
    {
        type = PERSONS;
    }

    QUrl url = createUrl(details.query, details.location);
    QString what;

    if(loggedIn_)
    {
        switch(type)
        {
        case YELLOW_PAGES:
            what = "mobcs";
            break;

        case PERSONS:
            what = "mobwp";
            break;

        default:
            what = "moball";
        }

    }
    else
    {
        what = "moball";
    }

    url.addQueryItem("what", what);

    http_.setHost(url.host(), url.port(80));
    int id = http_.get(url.encodedPath() + '?' + url.encodedQuery());

    QVector <Result> results;

    // Store search data for later identification
    SearchData* newData = new SearchData;
    newData->details = details;
    newData->results = results;
    newData->foundNumbers = 0;
    newData->numbersTotal = 0;

    // Store request id so that it can be identified later
    pendingSearches_[id] = newData;

    return true;
}

Eniro::Error Eniro::error() const
{
    return error_;
}

const QString& Eniro::errorString() const
{
    return errorString_;
}

void Eniro::httpReady(int id, bool error)
{
    if(error)
    {
        qDebug() << "Error: " << http_.errorString();
    }

    searchMap::const_iterator searchIt;
    numberMap::const_iterator numberIt;

    // Check if request is pending search request
    if((searchIt = pendingSearches_.find(id)) !=
        pendingSearches_.end())
    {
        if(error)
        {
            error_ = CONNECTION_FAILURE;
            errorString_ = http_.errorString();
            emitRequestFinished(id, searchIt.value(), true);
            return;
        }

        QString result(http_.readAll());

        // Load results from html data
        loadResults(id, result);
    }

    // Check if request is pending number requests
    else if((numberIt = pendingNumberRequests_.find(id)) !=
        pendingNumberRequests_.end())
    {
        if(error)
        {
            error_ = CONNECTION_FAILURE;
            errorString_ = http_.errorString();
            delete pendingNumberRequests_[id];
            pendingNumberRequests_.remove(id);
            return;
        }

        QString result(http_.readAll());

        // Load number from html data
        loadNumber(id, result);
    }

    // Check for login request
    else if(pendingLoginRequests_.find(id) !=
        pendingLoginRequests_.end())
    {
        bool success = true;

        if(!error)
        {
            QString result(http_.readAll());

            // If html source contains LOGIN_CHECK, login failed
            if(result.indexOf(LOGIN_CHECK) != -1)
            {
                success = false;
            }
        }
        else
        {
            success = false;
        }

        emit loginStatus(success);
    }

}

// Loads results from html source code
void Eniro::loadResults(int id, QString const& httpData)
{
    searchMap::iterator it = pendingSearches_.find(id);
    QString expr;

    switch(it.value()->details.type)
    {
    case YELLOW_PAGES:
        expr = YELLOW_REGEXP;
        break;
    case PERSONS:
        expr = PERSON_REGEXP;
        break;
    default:
        return;
    }

    QRegExp rx(expr);
    rx.setMinimal(true);

    bool requestsPending = false;
    int pos = 0;
    QString data;

    // Find all matches
    while((pos = rx.indexIn(httpData, pos)) != -1)
    {
        pos += rx.matchedLength();

        data = rx.cap(2);
        data = stripTags(data);
        QStringList rows = data.split('\n');

        for(int i = 0; i < rows.size(); i++)
        {
            // Remove white spaces
            QString trimmed = rows.at(i).trimmed().toLower();

            // Remove empty strings
            if(trimmed.isEmpty())
            {
                rows.removeAt(i);
                i--;
            }
            else
            {
                // Convert words to uppercase
                rows[i] = ucFirst(trimmed);
            }
        }

        Result result;

        int size = rows.size();

        switch(size)
        {
        case 1:
            result.name = rows[0];
            break;

        case 2:
            result.name = rows[0];
            result.city = rows[1];
            break;

        case 3:
            result.name = rows[0];
            result.street = rows[1];
            result.city = rows[2];
            break;

        case 4:
            result.name = rows[0];
            // Remove slashes and spaces from number
            result.number = cleanUpNumber(rows[1]);
            result.street = rows[2];
            result.city = rows[3];
            break;

        default:
            continue;

        }

        it.value()->results.push_back(result);

        unsigned int foundResults = ++(it.value()->numbersTotal);

        // If phone number searh is enabled, we have to make another
        // request to find it out
        if(findNumber_ && size < 4 && loggedIn_ &&
                it.value()->details.type != YELLOW_PAGES)
        {
            requestsPending = true;
            getNumberForResult(id, it.value()->results.size() - 1, it.value()->details);
        }
        // Otherwise result is ready
        else
        {
            emit resultAvailable(result, it.value()->details);
        }

        // Stop searching if max results is reached
        if(maxResults_ && (foundResults >= maxResults_))
        {
            break;
        }
    }

    // If there were no results or no phone numbers needed to
    // be fetched, the whole request is ready
    if(it.value()->numbersTotal == 0 || !requestsPending)
    {
        bool error = false;

        if(httpData.indexOf(LOGIN_CHECK) != -1)
        {
            error_ = INVALID_LOGIN;
            errorString_ = INVALID_LOGIN_STRING;
            error = true;
        }

        emitRequestFinished(it.key(), it.value(), error);
    }
}

// Loads phone number from html source
void Eniro::loadNumber(int id, QString const& result)
{
    numberMap::iterator numberIt = pendingNumberRequests_.find(id);

    // Make sure that id exists in pending number requests
    if(numberIt == pendingNumberRequests_.end() || numberIt.value() == 0)
    {
        return;
    }

    searchMap::iterator searchIt = pendingSearches_.find(numberIt.value()->searchId);

    if(searchIt == pendingSearches_.end() || searchIt.value() == 0)
    {
        return;
    }

    QRegExp rx(NUMBER_REGEXP);
    rx.setMinimal(true);

    int pos = 0;
    bool error = true;

    if((pos = rx.indexIn(result, pos)) != -1)
    {
        QString data = rx.cap(1);
        data = stripTags(data);

        QString trimmed = data.trimmed();

        if(!trimmed.isEmpty())
        {
            // Remove whitespaces from number
            searchIt.value()->results[numberIt.value()->index].number = cleanUpNumber(trimmed);

            emit resultAvailable(searchIt.value()->results[numberIt.value()->index], searchIt.value()->details);

            unsigned int found = ++searchIt.value()->foundNumbers;

            // Check if all numbers have been found
            if(found >= searchIt.value()->numbersTotal)
            {
                emitRequestFinished(searchIt.key(), searchIt.value(), false);
            }

            // If number was found, there was no error
            error = false;
        }
    }

    if(error)
    {
        error_ = INVALID_LOGIN;
        errorString_ = INVALID_LOGIN;
        emitRequestFinished(searchIt.key(), searchIt.value(), true);
    }

    // Remove number request
    int key = numberIt.key();

    delete pendingNumberRequests_[key];
    pendingNumberRequests_[key] = 0;
    pendingNumberRequests_.remove(key);

}

QUrl Eniro::createUrl(QString const& query, QString const& location)
{
    QUrl url(SITE_URLS[site_] + "query");

    if(!query.isEmpty())
    {
        url.addQueryItem("search_word", query);
    }

    if(!location.isEmpty())
    {
        url.addQueryItem("geo_area", location);
    }

    if(maxResults_)
    {
        url.addQueryItem("hpp", QString::number(maxResults_));
    }
    if(loggedIn_)
    {
        url.addQueryItem("login_name", username_);
        url.addQueryItem("login_password", password_);
    }

    QByteArray path = url.encodedQuery().replace('+', "%2B");
    url.setEncodedQuery(path);

    return url;
}

// Creates a new request for phone number retrieval
void Eniro::getNumberForResult(int id, int index, SearchDetails const& details)
{
    QUrl url = createUrl(details.query, details.location);
    url.addQueryItem("what", "mobwpinfo");
    url.addQueryItem("search_number", QString::number(index + 1));

    http_.setHost(url.host(), url.port(80));
    int requestId = http_.get(url.encodedPath() + '?' + url.encodedQuery());
    NumberData* number = new NumberData;
    number->searchId = id;
    number->index = index;
    pendingNumberRequests_[requestId] = number;

}

void Eniro::emitRequestFinished(int key, SearchData* data, bool error)
{

    // Do not emit "Request aborted" error
    if(!(error && (http_.error() == QHttp::Aborted)))
    {
        emit requestFinished(data->results, data->details, error);
    }

    delete pendingSearches_[key];
    pendingSearches_[key] = 0;
    pendingSearches_.remove(key);
}

QString Eniro::ucFirst(QString& str)
{
    if (str.size() < 1) {
        return "";
    }

    QStringList tokens = str.split(" ");
    QList<QString>::iterator tokItr;

    for (tokItr = tokens.begin(); tokItr != tokens.end(); ++tokItr)
    {
        (*tokItr) = (*tokItr).at(0).toUpper() + (*tokItr).mid(1);
    }

    return tokens.join(" ");
}

QString& Eniro::cleanUpNumber(QString& number)
{
    return number.replace(numberCleaner_, "");
}

QString& Eniro::stripTags(QString& string)
{
    return string.replace(tagStripper_, "");
}

QMap <Eniro::Site, Eniro::SiteDetails> Eniro::getSites()
{
    QMap <Site, SiteDetails> sites;
    SiteDetails details;

    for(int i = 0; i < SITE_COUNT; i++)
    {
        SiteDetails details;
        details.name = SITE_NAMES[i];
        details.id = SITE_IDS[i];
        sites[static_cast<Site>(i)] = details;
    }

    return sites;
}

Eniro::Site Eniro::stringToSite(QString const& str)
{
    Site site = FI;
    QString lower = str.toLower();

    for(int i = 0; i < SITE_COUNT; i++)
    {
        if(lower == SITE_NAMES[i] || lower == SITE_IDS[i])
        {
            site = static_cast <Site> (i);
        }
    }

    return site;
}

Eniro::SearchDetails::SearchDetails(QString const& q,
                                    QString const& loc,
                                    SearchType t)
{
    query = q;
    location = loc;
    type = t;
}
