/*
 * This file is part of Jenirok.
 *
 * Jenirok is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Jenirok is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jenirok.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QDebug>
#include <QApplication>
#include <QtCore/QTimerEvent>
#include <QtCore/QVariant>
#include <QtDBus/QDBusArgument>
#include <icd/dbus_api.h>
#include "connectionmanager.h"


ConnectionManager::ConnectionManager(QObject* parent): QObject(parent),
blocking_(true), stateReady_(false), connectionReady_(false), scanReady_(false),
connected_(false), timeout_(false), numberOfConnections_(0),
scannedConnections_(0), timer_(0), connections_(false)
{
    QDBusConnection systemBus = QDBusConnection::systemBus();

    icd2interface_ = new QDBusInterface(ICD_DBUS_API_INTERFACE,
                                        ICD_DBUS_API_PATH, ICD_DBUS_API_INTERFACE,
                                        systemBus, this);

    systemBus.connect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH,
                      ICD_DBUS_API_INTERFACE, ICD_DBUS_API_STATE_SIG,
                      this, SLOT(stateChange(const QDBusMessage&)));

    systemBus.connect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH,
                      ICD_DBUS_API_INTERFACE, ICD_DBUS_API_CONNECT_SIG,
                      this, SLOT(connectionChange(const QDBusMessage&)));

    systemBus.connect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH,
                      ICD_DBUS_API_INTERFACE, ICD_DBUS_API_SCAN_SIG,
                      this, SLOT(scanResult(const QDBusMessage&)));

}

ConnectionManager::~ConnectionManager()
{
    QDBusConnection systemBus = QDBusConnection::systemBus();

    systemBus.disconnect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH,
                         ICD_DBUS_API_INTERFACE, ICD_DBUS_API_STATE_SIG,
                         this, SLOT(stateChange(const QDBusMessage&)));

    systemBus.disconnect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH,
                         ICD_DBUS_API_INTERFACE, ICD_DBUS_API_CONNECT_SIG,
                         this, SLOT(connectionChange(const QDBusMessage&)));

    systemBus.disconnect(ICD_DBUS_API_INTERFACE, ICD_DBUS_API_PATH,
                         ICD_DBUS_API_INTERFACE, ICD_DBUS_API_SCAN_SIG,
                         this, SLOT(scanResult(const QDBusMessage&)));
}

void ConnectionManager::setBlocking(bool value)
{
    blocking_ = value;
}

bool ConnectionManager::connect()
{
    connectionReady_ = false;
    unsigned int flags = static_cast<unsigned int>(ICD_CONNECTION_FLAG_USER_EVENT);
    icd2interface_->call(ICD_DBUS_API_CONNECT_REQ, QVariant(flags));

    if(blocking_)
    {
        waitSignal(&connectionReady_);
        return connected_;
    }

    return true;
}

bool ConnectionManager::connect(ConnectionManager::Connection const& connection)
{
    connectionReady_ = false;
    QDBusArgument arg;
    arg.beginStructure();
    arg << connection.serviceType
        << connection.serviceAttributes
        << connection.serviceID
        << connection.networkType
        << connection.networkAttributes
        << connection.networkID;
    arg.endStructure();

    unsigned int flags = static_cast<unsigned int>(ICD_CONNECTION_FLAG_USER_EVENT);
    QDBusMessage rep = icd2interface_->call(ICD_DBUS_API_CONNECT_REQ,
                                            flags, arg.asVariant());

    qDebug() << rep.errorName() << rep.errorMessage();

    if(blocking_)
    {
        waitSignal(&connectionReady_);
        return connected_;
    }

    return true;
}

bool ConnectionManager::disconnect(bool force)
{
    // Forced disconnect is not allowed if connection
    // was not initialized by this class
    if(!connected_ && force)
    {
        return false;
    }

    connectionReady_ = false;
    unsigned int flags;

    if(force)
    {
        flags = static_cast<unsigned int>(ICD_CONNECTION_FLAG_UI_EVENT);
    }
    else
    {
        flags = static_cast<unsigned int>(ICD_CONNECTION_FLAG_USER_EVENT);
    }

    icd2interface_->call(ICD_DBUS_API_DISCONNECT_REQ, QVariant(flags));
    connected_ = false;
    return true;
}

bool ConnectionManager::isConnected()
{
    stateReady_ = false;
    QDBusMessage rep = icd2interface_->call(ICD_DBUS_API_STATE_REQ);

    unsigned int numOfReplies = rep.arguments().value(0).value<unsigned int>();

    if(numOfReplies == 0)
    {
        emit isConnectedReply(false);
        return false;
    }

    if(blocking_)
    {
        waitSignal(&stateReady_);
        return connected_;
    }

    return true;
}

bool ConnectionManager::scanConnections(QList<ConnectionManager::Connection>& connections)
{
    unsigned int flags = static_cast<unsigned int>(ICD_SCAN_REQUEST_ACTIVE);
    scanReady_ = false;
    scannedConnections_ = 0;
    connections_ = &connections;
    QDBusMessage rep = icd2interface_->call(ICD_DBUS_API_SCAN_REQ, QVariant(flags));

    numberOfConnections_ = rep.arguments().value(0).toList().size();

    if(numberOfConnections_ == 0)
    {
        connections_ = 0;
        return false;
    }

    if(blocking_)
    {
        bool ret = waitSignal(&scanReady_);
        connections_ = 0;
        return ret;
    }

    return true;
}

void ConnectionManager::stateChange(const QDBusMessage& rep)
{
    unsigned int status = rep.arguments().value(7).value<unsigned int>();

    switch(status)
    {
    case ICD_STATE_CONNECTING:
        break;
    case ICD_STATE_CONNECTED:
        connected_ = true;
        stateReady_ = true;
        break;
    case ICD_STATE_DISCONNECTING:
        break;
    case ICD_STATE_DISCONNECTED:
        connected_ = false;
        stateReady_ = true;
        break;
    case ICD_STATE_LIMITED_CONN_ENABLED:
        connected_ = true;
        stateReady_ = true;
        break;
    case ICD_STATE_LIMITED_CONN_DISABLED:
        connected_ = false;
        stateReady_ = true;
        break;
    case ICD_STATE_SEARCH_START:
        break;
    case ICD_STATE_SEARCH_STOP:
        break;
    case ICD_STATE_INTERNAL_ADDRESS_ACQUIRED:
        break;
    default:
        qDebug() << "Unknown connection status";
        break;
    }

    if(stateReady_)
    {
        emit isConnectedReply(connected_);
    }

}

void ConnectionManager::connectionChange(const QDBusMessage& rep)
{
    unsigned int status = rep.arguments().value(6).value<unsigned int>();

    switch(status)
    {
    case ICD_CONNECTION_SUCCESSFUL:
        connected_ = true;
        connectionReady_ = true;
        break;
    case ICD_CONNECTION_NOT_CONNECTED:
        connected_ = false;
        connectionReady_ = true;
        break;
    case ICD_CONNECTION_DISCONNECTED:
        connected_ = false;
        connectionReady_ = true;
        break;
    default:
        qDebug() << "Unknown connection status";
        break;
    }

    if(connectionReady_)
    {
        emit connectReply(connected_);
    }
}

void ConnectionManager::scanResult(const QDBusMessage& rep)
{
    if(!connections_)
    {
        return;
    }

    QList<QVariant> args = rep.arguments();

    unsigned int status = args.value(0).value<unsigned int>();

    if(status == ICD_SCAN_COMPLETE)
    {
        scannedConnections_++;
    }

    if(scannedConnections_ >= numberOfConnections_)
    {
        scanReady_ = true;
        emit scanReady();
        return;
    }

    if(status != ICD_SCAN_NEW)
    {
        return;
    }

    Connection connection;
    connection.serviceType = args.value(2).toString();
    connection.serviceAttributes = args.value(4).value<unsigned int>();
    connection.serviceID = args.value(5).toString();
    connection.networkName = args.value(8).toString();
    connection.networkType = args.value(7).toString();
    connection.networkAttributes = args.value(9).value<unsigned int>();
    connection.networkID = args.value(10).toByteArray();

    emit newConnection(connection);

    connections_->push_back(connection);
}

bool ConnectionManager::waitSignal(bool* ready)
{
    timeout_ = false;
    timer_ = startTimer(TIMEOUT);

    while(!*ready && !timeout_)
    {
        QCoreApplication::processEvents(QEventLoop::WaitForMoreEvents);
    }

    killTimer(timer_);

    return *ready || !timeout_;
}

void ConnectionManager::timerEvent(QTimerEvent* event)
{
    Q_UNUSED(event);
    killTimer(timer_);
    timeout_ = true;
    timer_ = 0;

    qDebug() << "Connection request timed out";
}
