/**sequencer.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is center of 7-12 years full sequencer view.
View contains three part:
 *wheel-game (loop-view)
 *track-view
 *general-view (settings)

*/

#include <glib-object.h>
//#include <clutter/clutter.h>
#include <math.h>
#include <string.h>

#include <tangle.h>

#include "../../meam/jammo-meam.h"
#include "../../meam/jammo-slider-event.h"
#include "../../meam/jammo-slider-track.h"
#include "../../meam/jammo-backing-track.h"
#include "../../meam/jammo-metronome-track.h"

#include "../jammo.h"
#include "../jammo-mentor.h"
#include "../jammo-track-view.h"
#include "../jammo-miditrack-view.h"


#include "sequencer.h"
#include "sequencer_loop.h"
#include "sequencer_track_view.h"
#include "startmenu.h"

//Copy pasted from jammo-backing-track.c
static gchar* str_replace(gchar* string, gsize position, gsize length, const gchar* replacement, gchar** replacement_end_return) {
	gsize string_length;
	gsize replacement_length;
	gchar* s;
	
	string_length = strlen(string);
	replacement_length = strlen(replacement);
	s = (gchar*)g_malloc(string_length - length + replacement_length + 1);
	g_memmove(s, string, position);
	g_memmove(s + position, replacement, replacement_length);
	g_memmove(s + position + replacement_length, string + position + length, string_length - position - length + 1);

	if (replacement_end_return) {
		*replacement_end_return = s + position + replacement_length;
	}

	g_free(string);

	return s;
}

/*
When sequencer starts: it calls load_state_from_file(load_only_backingtrack=TRUE)
 It starts loading backing-track.
 When backing_track is loaded (on_duration_notify)
  It creates sequencer and sequencer-track-view-area.
  Then load_state_from_file(load_only_backingtrack=FALSE) loads another tracks.
*/

//This is The sequencer of the game. All 'play'-buttons should play this.
//All new tracks are added to this
//If metronome is enabled it is added to this
JammoSequencer* static_sequencer;


static void on_duration_notify(GObject* object, GParamSpec* pspec, gpointer track);

static void load_state_from_file(gchar* filename, gboolean load_only_backingtrack) {

JsonParser *parser;
parser = json_parser_new ();
g_assert (JSON_IS_PARSER (parser));

GError *error = NULL;
if (!json_parser_load_from_file (parser, filename, &error))
	{
		g_print ("Error: %s\n", error->message);
		g_error_free (error);
		g_object_unref (parser);
		return;
	}

JsonNode *root;
JsonObject *object;
JsonNode *node;

g_assert (NULL != json_parser_get_root (parser));

root = json_parser_get_root (parser);
g_assert_cmpint (JSON_NODE_TYPE (root), ==, JSON_NODE_OBJECT);

object = json_node_get_object (root);
g_assert (object != NULL);

	gint tempo = 110; //'normal' tempo
	gchar* pitch ="D";  //'normal' pitch


	//gint slot;    //TODO: not used
	//gint loop_id; //TODO: function missing

	//META data: Load these only once
if (load_only_backingtrack){
	//int
	node = json_object_get_member (object, "tempo");
	if (node!=NULL && JSON_NODE_TYPE (node) == JSON_NODE_VALUE){
		tempo = json_node_get_int (node);
		printf("tempo: '%d'\n",tempo);
	}
	//string
	node = json_object_get_member (object, "pitch");
	if (node!=NULL && JSON_NODE_TYPE (node) == JSON_NODE_VALUE){
		pitch = (gchar*)  json_node_get_string (node);
		printf("pitch: '%s'\n",pitch);
	}

	jammo_sequencer_set_tempo(static_sequencer,tempo);
	jammo_sequencer_set_pitch(static_sequencer,pitch);

	//Correct GUI-elements:
	gchar* name_of_tempo_button;
	if (tempo==130) name_of_tempo_button="tempo-button1";
	else if (tempo==110) name_of_tempo_button="tempo-button2";
	else /*(tempo==90)*/ name_of_tempo_button="tempo-button3";
	tangle_button_set_selected(TANGLE_BUTTON(jammo_get_actor_by_id(name_of_tempo_button)),TRUE);

	gchar* name_of_pitch_button;
	if (pitch[0]=='G') name_of_pitch_button="pitch-button1";
	else if (pitch[0]=='D') name_of_pitch_button="pitch-button2";
	else /* (pitch[0]=='A')*/ name_of_pitch_button="pitch-button3";
	tangle_button_set_selected(TANGLE_BUTTON(jammo_get_actor_by_id(name_of_pitch_button)),TRUE);
}


	ClutterActor* track_view = NULL; //Will be jammo_trac_view or jammo_miditrack_view or TangleWidget
	JammoTrack* track = NULL;

	//Array
	node = json_object_get_member (object, "tracks");
	JsonArray* track_array;
	guint length_tracks = 0;
	if (node!=NULL && JSON_NODE_TYPE (node) == JSON_NODE_ARRAY){
		track_array =  json_node_get_array (node);
		length_tracks = json_array_get_length(track_array);
	}
	else {printf("Not any tracks\n"); }

	printf("length %d\n",length_tracks);
	int j;
	for (j=0;j<length_tracks;j++) {
		track_view = NULL;
		track = NULL;

		//Object
		JsonNode* track_node;
		track_node = json_array_get_element(track_array,j);

		if (track_node!=NULL && JSON_NODE_TYPE(track_node) == JSON_NODE_OBJECT){
			printf("Found track!\n");
			gchar* track_type="";

			JsonObject* sub_object = json_node_get_object(track_node);
			JsonNode *sub_node;
			sub_node = json_object_get_member (sub_object, "track-type");
			if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
				track_type =(gchar*)  json_node_get_string (sub_node);
				printf(" track-type: '%s'\n",track_type);
				}

			//boolean. We need this to construct track_view
			gboolean muted=FALSE; //default
			sub_node = json_object_get_member (sub_object, "muted");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					muted = json_node_get_boolean (sub_node);
					printf(" muted: '%d'\n",muted);
				}

			if(load_only_backingtrack) {
			//BACKING-TRACK
			if (strncmp(track_type, "BackingTrack", 12) == 0) {

					printf("Found mandatory BackingTrack\n");
					gchar* backing_track_filename="";
					//string
					sub_node = json_object_get_member (sub_object, "audio-file");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						backing_track_filename =(gchar*)  json_node_get_string (sub_node);
						printf("backing_track_filename: '%s'\n",backing_track_filename);

						track = JAMMO_TRACK(jammo_backing_track_new(backing_track_filename));
						// Create other tracks after the duration of the backing track is known.
						JammoSample* sample;
						printf("pitch '%s', tempo '%d'\n",pitch,tempo);

						//This code is copy-pasted from jammo-backing-track.c:287
						gchar* s;
						gchar* filename_for_sample = s = g_strdup(backing_track_filename);
						gchar* tempo_string;
						while ((s = strchr(s, '$'))) {
							switch (s[1]) {
								case '$':
									filename_for_sample = str_replace(filename_for_sample, s - filename_for_sample, 2, "$", &s);
									break;
								case 't':
									tempo_string = g_strdup_printf("%u", tempo);
									filename_for_sample = str_replace(filename_for_sample, s - filename_for_sample, 2, tempo_string, &s);
									g_free(tempo_string);
									break;
								case 'p':
									filename_for_sample = str_replace(filename_for_sample, s - filename_for_sample, 2, pitch, &s);
									break;
								default:
									g_warning("Unexpected parameter in the backing track filename: '$%c', skipped.", s[1]);
									s += 2;
									break;
							}
						}
						sample = jammo_sample_new_from_file(filename_for_sample);
						g_free(filename_for_sample);
						g_signal_connect(sample, "notify::duration", G_CALLBACK(on_duration_notify), track);
	
					}
					printf("Ending looking for BackingTrack\n");

				}
			} //End load_only_backingtrack

			else {
			//EDITING-TRACK
			if(strncmp(track_type, "EditingTrack", 12) == 0) {
				gint e_track_type=-1;
				gchar* editing_track_type="";
				//string
				sub_node = json_object_get_member (sub_object, "editing-track-type");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					editing_track_type =(gchar*)  json_node_get_string (sub_node);
					printf("editing-track-type: '%s'\n",editing_track_type);
				}
				ClutterColor sample_color =  { 0, 0, 0, 255 };
				if (strncmp(editing_track_type,"Rhythm",6) == 0 ) {
					e_track_type = RHYTHM_TRACK;
					sample_color.red   = 255;
					sample_color.green = 155;
					sample_color.blue  = 0;
				}
				else if (strncmp(editing_track_type,"Melody",6) == 0 ) {
					e_track_type = MELODY_TRACK;
					sample_color.red   = 0;
					sample_color.green = 0;
					sample_color.blue  = 155;
				}
				else if (strncmp(editing_track_type,"Harmony",7) == 0 ) {
					e_track_type = HARMONY_TRACK;
					sample_color.red   = 200;
					sample_color.green = 0;
					sample_color.blue  = 0;
				}
				else if (strncmp(editing_track_type,"Effect",6) == 0 ) {
					e_track_type = EFFECT_TRACK;
					sample_color.red    = 0;
					sample_color.green  = 155;
					sample_color.blue   = 0;
				}
				track_view=sequencer_track_view_add_with_type(EDITING_TRACK_VIEW,muted,e_track_type);
				g_object_set(track_view, "sample-color", &sample_color, NULL);


				//Array
				sub_node = json_object_get_member (sub_object, "samples");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_ARRAY){
					JsonArray* sample_array =  json_node_get_array (sub_node);

					guint length = json_array_get_length(sample_array);
					printf("length %d\n",length);
					int i;
					for (i=0;i<length;i++) {
						JsonNode* sample_node;
						sample_node = json_array_get_element(sample_array,i);
						if (sample_node!=NULL && JSON_NODE_TYPE(sample_node) == JSON_NODE_OBJECT){
							JsonObject* sample_object = json_node_get_object(sample_node);
							gint loop_id=-1;
							gint slot=-1;

							//int
							sub_node = json_object_get_member (sample_object, "loop_id");
							if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
								loop_id =  json_node_get_int (sub_node);
								printf("loop_id: '%d'\n",loop_id);
							}
							//int
							sub_node = json_object_get_member (sample_object, "slot");
							if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
								slot =  json_node_get_int (sub_node);
								printf("slot: '%d'\n",slot);
							}
							if (track_view==NULL)
									printf("track_view==NULL\n");
							ClutterActor* sample_button=NULL;// = theme_give_sample_button_for_this_theme_and_variation_for_this_id(theme_name,variation,loop_id);
							if (sample_button==NULL)
								printf("sample_button==NULL\n");
							//jammo_track_view_add_jammo_sample_button(track_view,JAMMO_SAMPLE_BUTTON(sample_button),slot);

						} //This sample-object is ready

					} //Foreach in sample-array ready

				} //sample-array is ready
			} //Editing-track over

			//VirtualInstrumentTrack
			else if(strncmp(track_type, "VirtualInstrumentTrack", 22) == 0) {
				//int
				gint instrument=0; //default instrument
				sub_node = json_object_get_member (sub_object, "instrument");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						instrument = json_node_get_int (sub_node);
						printf("instrument: '%d'\n",instrument);
					}

				track_view=sequencer_track_view_add_with_type(INSTRUMENT_TRACK_VIEW,muted,instrument);

				//string
				sub_node = json_object_get_member (sub_object, "note-file");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					gchar* note_file = (gchar*) json_node_get_string (sub_node);
					printf("loading notes from file '%s'\n",note_file);
					GList* events = jammomidi_file_to_glist(note_file);
					jammo_miditrack_view_add_event_list(JAMMO_MIDITRACK_VIEW(track_view),events);
				}
			} //VirtualInstrumentTrack over


			//SliderTrack
			else if(strncmp(track_type, "SliderTrack", 11) == 0) {
				//int
				gint instrument=0; //default instrument
				sub_node = json_object_get_member (sub_object, "instrument");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						instrument = json_node_get_int (sub_node);
						printf("instrument: '%d'\n",instrument);
					}

				track_view=sequencer_track_view_add_with_type(SLIDER_TRACK_VIEW,muted,instrument);
				//In slider-case: track_view is TangleWidget.
				track = g_object_get_data(G_OBJECT(track_view), "track");

				//string
				sub_node = json_object_get_member (sub_object, "note-file");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					gchar* note_file = (gchar*) json_node_get_string (sub_node);
					printf("loading notes from file '%s'\n",note_file);
					GList* events = jammo_slider_event_file_to_glist(note_file);
					jammo_slider_track_set_event_list(JAMMO_SLIDER_TRACK(track), events); //We need access to track here!
				}
			} //SliderTrack over

			} //End else- load_only_backingtrack


			//Tune volume, mute, etc.
			//In case of slider and backing_track, we already have 'track'
			if (!track) {
				if (track_view && (JAMMO_IS_TRACK_VIEW(track_view) || JAMMO_IS_MIDITRACK_VIEW(track_view)))
					g_object_get(track_view,"track",&track,NULL);
			}

			//If we didn't got track, skip this
			if (track) {
				//double / float
				gfloat volume=1.0; //default volume is 100%
				sub_node = json_object_get_member (sub_object, "volume");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						volume = (gfloat) json_node_get_double (sub_node);
						printf("volume: '%f'\n",volume);
					}
				jammo_playing_track_set_volume(JAMMO_PLAYING_TRACK(track), volume);

				//This is asked earlier:
				jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(track), muted);
			}

		} //Track is ready

	} //Next track

g_object_unref(parser);

}



gboolean sequencer_change_to_loop_view(TangleActor *actor, gpointer data) {
	printf("changing view to loop \n");
	ClutterActor* scrolling_actor = jammo_get_actor_by_id("fullsequencer-scroller");
	clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_CIRC, 400, "scrolling-offset-y", 0.0, NULL);
	return TRUE;
}

gboolean sequencer_change_to_sequencer_view(TangleActor *actor, gpointer data) {
	printf("changing view to sequencer \n");
	ClutterActor* scrolling_actor = jammo_get_actor_by_id("fullsequencer-scroller");
	clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_CIRC, 400, "scrolling-offset-y", 480.0, NULL);
	return TRUE;
}

gboolean sequencer_change_to_bottom_view (TangleActor *actor, gpointer data) {
	printf("changing view to bottom \n");
	ClutterActor* scrolling_actor = jammo_get_actor_by_id("fullsequencer-scroller");
	clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor),   CLUTTER_EASE_IN_CIRC, 400, "scrolling-offset-y", 960.0, NULL);
	return TRUE;
}



static void on_duration_notify(GObject* object, GParamSpec* pspec, gpointer backing_track) {

	JammoSample* sample;
	sample = JAMMO_SAMPLE(object);

	// The duration of backing track is used to set fixed duration for other tracks
	guint64 duration;
	duration = jammo_sample_get_duration(sample);
	g_print("Duration: %" G_GUINT64_FORMAT "\n", duration);

	gint tempo_factory = 4;
	//We want operate whole time with guint64.
	guint64 temp=60*tempo_factory;
	guint64 one_second = 1000000000L;
	guint64 big = temp * one_second;
	guint64 duration_of_one_slot = big / jammo_sequencer_get_tempo(static_sequencer);

	gint number_of_slots = duration/duration_of_one_slot;
	printf("*number_of_slots in backing_track: %d\n",number_of_slots);
	gfloat slot_width = 40.0;
	gfloat height = 50.0;
	//sequencer_track_view uses internally this just created area.
	sequencer_track_view_create_area_for_tracks(static_sequencer,number_of_slots, duration, slot_width,height);


	ClutterActor* track_view = NULL;
	track_view=sequencer_track_view_add_backing_track(backing_track, jammo_playing_track_get_muted(backing_track));

	//continue loading
	load_state_from_file(inputFilename,FALSE);

	// Do not call this function ever again!
	g_signal_handlers_disconnect_by_func(object, G_CALLBACK(on_duration_notify), static_sequencer);
}

JammoMetronomeTrack* static_metronome_track;

/*
This actor is visible when starting jammo_get_actor_by_id("fullsequencer-view");
*/
void start_sequencer(gpointer filename)
{
	static_sequencer = JAMMO_SEQUENCER(jammo_get_object_by_id("fullsequencer-the-sequencer"));
	static_metronome_track = jammo_metronome_track_new();
	jammo_sequencer_add_track(static_sequencer, JAMMO_TRACK(static_metronome_track));
	jammo_metronome_track_set_time_signature_beats(static_metronome_track,4);
	jammo_metronome_track_set_time_signature_note_value(static_metronome_track,4);
	jammo_metronome_track_set_accent(static_metronome_track,FALSE);
	jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(static_metronome_track), TRUE);

	printf("Starting sequencer GUI\n");
	//inputFilename is global! (from sequencer.h)
	if (filename == NULL)
		inputFilename = NULL;
	else {
		inputFilename = malloc(strlen((gchar*) filename) + 1);
		memcpy(inputFilename, (gchar*) filename, strlen((gchar*) filename) + 1);
		printf("The filename is '%s'\n",inputFilename);
	}


	tangle_actor_hide_animated(TANGLE_ACTOR(jammo_mentor_get_default()));


	//Tune some things that can't be done with JSON only ("clamp-scrolling-offset-y")
	sequencer_loop_tune_wheels();

	//Loading tracks from file:
	if(inputFilename) {
		FILE *ifp;
		ifp = fopen(inputFilename, "r"); //currently open only for reading

		if(ifp != NULL) {
			load_state_from_file(inputFilename,TRUE);
		}
	} else {
		printf("No file specified for sequencer.\n");
	}


	sequencer_change_to_sequencer_view(NULL,NULL); //Start to middle view.
}

/*
void leave_sequencer()
{
	clutter_actor_hide(CLUTTER_ACTOR(sequencer_view));

	start_startmenu();
}
*/



/*
 * CallBack for JSON
 */
void fullsequencer_general_arrow_clicked (TangleButton *tanglebutton, gpointer user_data){
printf("up\n");
sequencer_change_to_sequencer_view(NULL,NULL);
}


void fullsequencer_general_tempo_clicked(TangleButton *tanglebutton, gpointer user_data) {
	printf("tempo clicked '%s'\n",clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)));

	int new_tempo = 110;
	if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"fast",4)==0)
			new_tempo=130;
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"normal",6)==0)
			new_tempo=110;
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"slow",4)==0)
			new_tempo=90;
	printf("new tempo:%d\n",new_tempo);
	jammo_sequencer_set_tempo(static_sequencer, new_tempo);
}


void fullsequencer_general_pitch_clicked(TangleButton *tanglebutton, gpointer user_data) {
	printf("pitch clicked '%s'\n",clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)));

	gchar* new_pitch = "G";
	if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"high",4)==0)
			new_pitch = "G";
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"normal",6)==0)
			new_pitch = "D";
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"low",3)==0)
			new_pitch = "A";
	printf("new pitch:'%s'\n",new_pitch);

	jammo_sequencer_set_pitch(static_sequencer, new_pitch);
}




void fullsequencer_metronome_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Metronome pressed\n");

	jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(static_metronome_track), !tangle_button_get_selected(tanglebutton));
}
