/**sequencer_track_view.c is part of JamMo.
License: GPLv2, read more from COPYING

First call this: sequencer_sequencer_create_sequencer_container(JammoSequencer*)
And then sequencer_track_view_add_with_type(sequencerTrackViewTYpe)
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>
#include <tangle.h>


#include "../../meam/jammo-track.h"
#include "../../meam/jammo-editing-track.h"
#include "../../meam/jammo-instrument-track.h"
#include "../../meam/jammo-slider-track.h"
#include "../../meam/jammo-midi.h"
#include "../../meam/jammo-meam.h" //JAMMO_DURATION_INVALID

#include "../jammo-track-view.h"
#include "../jammo-miditrack-view.h"
#include "../jammo.h"
#include "../jammo-cursor.h"


#include "sequencer_track_view.h"
#include "sequencer.h"
#include "midi_editor.h"
#include "instrument_gui.h"


//We get these as parameters
static int number_of_slots = -1;
static int width_of_slot = -1;
static guint64 duration_of_tracks = JAMMO_DURATION_INVALID; //they all must have same duration
static gfloat track_height = -1;


static ClutterActor* texture_of_cursor;
static SequencerTrackView *track_views[6] = {NULL};
static gint tracks_in_use = 0;

static JammoSequencer* sequencer; // for playing the tracks;

static gfloat VOLUME_MAX = 100; //? FIX

static gboolean METRONOME_STATE = FALSE;


static gboolean pressed_sequencer_button_mute(ClutterActor *actor, ClutterEvent *event, gpointer data);
static ClutterAnimation * animate_buttonpress_release(ClutterActor *actor);
static gboolean pressed_sequencer_button_volume(ClutterActor *actor, ClutterEvent *event, gpointer data);
static gboolean pressed_sequencer_button_track_label(ClutterActor *actor, ClutterEvent *event, gpointer data);





/*
 * Creates and returns SequencerTrackView struct
 */
static SequencerTrackView *sequencer_track_view_new(ClutterActor *label,
    ClutterActor *track_view, char *track_view_type, void* track) {

	ClutterActor* mute_button = clutter_texture_new_from_file("/opt/jammo/sequencer/button_mute.png", NULL);
	clutter_actor_set_size(mute_button, track_height-4,track_height-4); //it is square. Four pixels marginal
	ClutterActor* volume_button = clutter_texture_new_from_file("/opt/jammo/sequencer/button_volume.png", NULL);
	clutter_actor_set_size(volume_button, track_height-4,track_height-4); //it is square. Four pixels marginal

	clutter_actor_set_height(label,track_height-4); //this is not square, so change oly height

	SequencerTrackView *view = malloc(sizeof(SequencerTrackView));
	view->control_panel = tangle_widget_new();
	view->label = label;
	view->mute_button = mute_button;
	view->volume_button = volume_button;
	view->track_view = track_view;
	view->position = -1;   //This is used only for volume_slider: FIXME: remove
	view->muted = FALSE;   //Track will know if it is muted. FIXME: remove
	view->volume_pressed_down = FALSE;
	view->volume_level = VOLUME_MAX;  //Track will know own volume_level. FIXME: remove
	view->track_type = track_view_type;

	tangle_widget_add(TANGLE_WIDGET(view->control_panel), view->label,NULL);
	tangle_widget_add(TANGLE_WIDGET(view->control_panel), view->mute_button,NULL);
	tangle_widget_add(TANGLE_WIDGET(view->control_panel), view->volume_button,NULL);

	gfloat x=0;
	clutter_actor_set_position(view->label, x, 0.0);
	x += 2 +  clutter_actor_get_width(view->label);

	clutter_actor_set_position(view->mute_button, x, 0.0);
	x += 2.0 + clutter_actor_get_width(view->mute_button);

	clutter_actor_set_position(view->volume_button, x, 0.0);

	g_signal_connect(CLUTTER_ACTOR(view->label), "button_press_event", G_CALLBACK(pressed_sequencer_button_track_label), view);
	g_signal_connect(CLUTTER_ACTOR(view->mute_button), "button_press_event", G_CALLBACK(pressed_sequencer_button_mute), view);
	g_signal_connect(CLUTTER_ACTOR(view->volume_button), "button_press_event", G_CALLBACK(pressed_sequencer_button_volume), view);



	clutter_actor_set_reactive(view->label, TRUE);
	clutter_actor_set_reactive(view->mute_button, TRUE);
	clutter_actor_set_reactive(view->volume_button, TRUE);

	return view;
}

/* Adds STV into array & container and positions it.
 */
static void sequencer_track_view_add(SequencerTrackView *track_view) {
	track_views[tracks_in_use] = track_view;

	//Add this tracks control_panel (volume buttons etc)
	clutter_container_add_actor(CLUTTER_CONTAINER(jammo_get_actor_by_id("fullsequencer-track-control-panel")), track_view->control_panel);
	clutter_actor_set_y(track_view->control_panel, track_height * tracks_in_use);

	//Add track to container
	ClutterActor* track_container = jammo_get_actor_by_id("fullsequencer-container-for-tracks");
	clutter_container_add_actor(CLUTTER_CONTAINER(track_container), track_view->track_view);
	clutter_actor_set_x(track_view->track_view, clutter_actor_get_x(track_view->track_view)+1);  //unless left border is not visible
	clutter_actor_set_y(track_view->track_view, track_height * (tracks_in_use)+1);  //unless bottom border is not visible

	//Raise container's and scroller's height
	clutter_actor_set_height(track_container,clutter_actor_get_height(track_container)+track_height);
	ClutterActor* scrolling_actor = jammo_get_actor_by_id("fullsequencer-scroller-for-tracks");
	clutter_actor_set_height(scrolling_actor, clutter_actor_get_height(track_container));
	clutter_actor_set_height(texture_of_cursor, clutter_actor_get_height(track_container));

	track_view->position = tracks_in_use;
	tracks_in_use++;
	printf("Sequencer Track View placed in Position y: %f\n",
         clutter_actor_get_y(track_view->control_panel));
}



/*
 * Adds given type of track view into sequencer view
 * Returns track_view_actor (jammo_track_view) of newly created sequencer_track_view
 * Returns NULL is askes invalid type.
 */
ClutterActor* sequencer_track_view_add_with_type(gchar* track_view_type ) {
	char *twt = track_view_type;
	printf("Making '%s' track (duration: %" G_GUINT64_FORMAT ")\n",track_view_type, duration_of_tracks);

	guint64 duration_of_one_slot = duration_of_tracks / number_of_slots;

	ClutterActor *label = clutter_texture_new();
	JammoTrack* track;

	ClutterActor* track_view_actor = NULL; //This will be returned
	SequencerTrackView *sequencer_track_view;

	//Choose label-image:
	if (strncmp(track_view_type,"DRUM_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_drum.png", NULL);

	else if (strncmp(track_view_type,"SLIDER_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_slider.png", NULL);

	else if (strncmp(track_view_type,"AUDIO_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_audio.png", NULL);

	else if (strncmp(track_view_type,"LOOP_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_loop.png", NULL);

	else if (strncmp(track_view_type,"BACKING_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_backingtrack.png", NULL);

	else if (strncmp(track_view_type,"KEYBOARD_TRACK_VIEW",7)==0)
		label= clutter_texture_new_from_file("/opt/jammo/sequencer/label_keyboard.png", NULL);


	else  { //Error-case
		printf("Not detected track-view type\n");
		return NULL;
	}


	//Make instrument_track and miditrack_view
	if (strncmp(track_view_type,"KEYBOARD_TRACK_VIEW",19)==0)
		{
		track = JAMMO_TRACK(jammo_instrument_track_new(JAMMO_INSTRUMENT_TYPE_UD)); //TODO. instrument_type
		jammo_instrument_track_set_realtime(JAMMO_INSTRUMENT_TRACK(track),FALSE);

		// TODO set transpose. Key of the song should be stored somewhere
		instrument_gui_set_transpose('c');

		//TODO: we need to know highest_note and lowest_note for this instrument!
		track_view_actor = jammo_miditrack_view_new(JAMMO_INSTRUMENT_TRACK(track),
          number_of_slots, duration_of_one_slot, width_of_slot, track_height,60,24); //height can be anything, we zoom it
		jammo_miditrack_view_zoom_to_fit(JAMMO_MIDITRACK_VIEW(track_view_actor), number_of_slots*width_of_slot,track_height);
		jammo_miditrack_view_set_show_grid(JAMMO_MIDITRACK_VIEW(track_view_actor),FALSE);
		}



	//Make slider_track and container for it
	else if (strncmp(track_view_type,"SLIDER_TRACK_VIEW",13)==0)
		{
		track = JAMMO_TRACK(jammo_slider_track_new(JAMMO_SLIDER_TYPE_KARPLUS));//TODO. instrument_type
		jammo_instrument_track_set_realtime(JAMMO_INSTRUMENT_TRACK(track),FALSE);

		track_view_actor = tangle_widget_new();
		g_object_set_data (G_OBJECT(track_view_actor),"track",track);
		//TODO: set size and some visualization.
		}

	//Make editing_track and track_view
	else
	{
		track = JAMMO_TRACK(jammo_editing_track_new_fixed_duration(duration_of_tracks));

		track_view_actor = jammo_track_view_new(JAMMO_EDITING_TRACK(track),
          number_of_slots, duration_of_one_slot, width_of_slot, track_height);
		g_object_set(track_view_actor, "line-every-nth-slot", 1, NULL);
	}
	
	/* TODO: This is a quick test. Please, fix me. */
	ClutterColor sample_color = { 255, 255, 0, 255 };
	g_object_set(track_view_actor, "sample-color", &sample_color, NULL);

	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(track));
	sequencer_track_view = sequencer_track_view_new(label, track_view_actor, twt, track);
	sequencer_track_view_add(sequencer_track_view);
	return track_view_actor;
}

static ClutterAnimation * animate_buttonpress_release(ClutterActor *actor) {
	ClutterAnimation *anim = clutter_actor_animate (actor,          /* the actor to animate */
			CLUTTER_LINEAR, /* the easing mode */
			100,           /* the duration of the animation */
			"scale-x", 1.0, /* final horizontal scaling factor */
			"scale-y", 1.0, /* final vertical scaling factor */
			"scale-gravity",CLUTTER_GRAVITY_CENTER,
			NULL);
	return anim;
}

static ClutterAnimation * animate_buttonpress_down(ClutterActor *actor) {
	ClutterAnimation *anim = clutter_actor_animate (actor,          /* the actor to animate */
			CLUTTER_LINEAR, /* the easing mode */
			100,           /* the duration of the animation */
			"scale-x", 0.80, /* final horizontal scaling factor */
			"scale-y", 0.80, /* final vertical scaling factor */
			"scale-gravity",CLUTTER_GRAVITY_CENTER,
			NULL);
	return anim;
}


/*
 * Event for pressing label on track, data points to track structure
 */
static gboolean pressed_sequencer_button_track_label(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	SequencerTrackView* track = (SequencerTrackView*)data;

	printf("Pressed track label\n");
	if (JAMMO_IS_MIDITRACK_VIEW(track->track_view))
	{
		printf("It is instrument track\n");
		//We can start realtime jamming or midi-editor.
		//Midi-editor backs always to sequencer.
		//From Jamming we can go back to sequencer or reach editor
		//TODO: some way to go straight to editor.

		//jamming
		instrument_gui_start(JAMMO_MIDITRACK_VIEW(track->track_view),sequencer, NULL); //NULL means no slider

		//editor
		//midi_editor_start_with_miditrack_view(JAMMO_MIDITRACK_VIEW(track->track_view));
	}
	else if (TANGLE_IS_WIDGET(track->track_view))
	{
		printf("It is slider track\n");
		instrument_gui_start(NULL,sequencer, TANGLE_WIDGET(track->track_view)); //NULL means no keyboard, but slider
	}

	return TRUE;
}



static gboolean pressed_sequencer_button_mute(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	SequencerTrackView *track_view = (SequencerTrackView*) data;
	JammoTrack* jammotrack;
	g_object_get(track_view->track_view, "track", &jammotrack, NULL);
	if (!jammotrack)
	{
		printf("JammoTrack false!\n");
		return FALSE;
	}
	//if mute is off, set it on
	if (!track_view->muted) {
		animate_buttonpress_down(actor);
		jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(jammotrack), TRUE);
		//Shade button
	}

	//if mute is on, set it off
	if (track_view->muted) {
		animate_buttonpress_release(actor);
		jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(jammotrack), FALSE);
	}

	//Change mute state to opposite
	track_view->muted = !track_view->muted;

	return TRUE;
}



/**
This syncs scrollers offset to tracks volume.
*/
gboolean sequencer_volumebar_used (ClutterActor *volume_scroller, gpointer none){
	//-80 is height of the visible marker.
	gfloat height = clutter_actor_get_height(CLUTTER_ACTOR(volume_scroller))-80;
	gfloat offset;
	g_object_get(volume_scroller, "scrolling-offset-y", &offset,NULL);
	//printf("height %f, offset :%f, ratio %f\n",height, offset,offset/height);

	JammoPlayingTrack* jammotrack = g_object_get_data(G_OBJECT(volume_scroller),"track");

	jammo_playing_track_set_volume(jammotrack,offset/height);
	//printf("new volume:%f\n",offset/height);
	return FALSE;
}



/**
Each tracks share one volumebar.
*/
static gboolean pressed_sequencer_button_volume(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	SequencerTrackView *track_view = (SequencerTrackView*) data;

	ClutterActor* volume_container = jammo_get_actor_by_id("fullsequencer-volumebar-container");
	ClutterActor* volume_scroller = jammo_get_actor_by_id("fullsequencer-volumebar-scroller");
	JammoPlayingTrack* jammotrack;

	//slider track is inside tangle_widget, midi- and loop-tracks are inside of jammo-*track-view
	if (TANGLE_IS_WIDGET(track_view->track_view))
		jammotrack = JAMMO_PLAYING_TRACK(g_object_get_data(G_OBJECT(track_view->track_view),"track"));
	else
		g_object_get(track_view->track_view, "track", &jammotrack, NULL);

	gfloat current_volume = jammo_playing_track_get_volume(jammotrack);
	//printf("volume of this track is :%f\n",current_volume);


	//When pressing already selected track, hide volume bar
	JammoPlayingTrack* jammotrack_in_use = g_object_get_data(G_OBJECT(volume_scroller),"track");
	if (jammotrack == jammotrack_in_use) {
		clutter_actor_hide(volume_container);
		g_object_set_data(G_OBJECT(volume_scroller),"track",NULL);
	}


	else {
		clutter_actor_show(volume_container);
		//-80 is height of the visible marker.
		gfloat new_place = (clutter_actor_get_height(CLUTTER_ACTOR(volume_scroller))-80)*current_volume;
		//printf("new place: %f\n",new_place);
		clutter_actor_animate(CLUTTER_ACTOR(volume_scroller), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-y", new_place, NULL);
		g_object_set_data(G_OBJECT(volume_scroller),"track",jammotrack);
		}

	return FALSE;
}



bool save_sequencer_state(gchar* outputFilename) {

	printf("Saving composition\n");
	FILE *ofp;

	ofp = fopen(outputFilename, "w");
	g_free(outputFilename);
	g_return_val_if_fail(ofp != NULL, FALSE);

	GList* children;
	TangleWidget* widget;
	JammoSampleButton *jsb;

	int i;

	for(i = 0; i < tracks_in_use; i++) {
		fprintf(ofp, "Track-type: %s\n", track_views[i]->track_type);
		if(strcmp(track_views[i]->track_type, "KEYBOARD_TRACK_VIEW") == 0)
		{
		//TODO
			continue;
		}
		printf("%s\n", track_views[i]->track_type);

		widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_views[i]->track_view)));

		for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {

			jsb = JAMMO_SAMPLE_BUTTON(children->data);
			//printf("x: '%lf' \n",clutter_actor_get_x(CLUTTER_ACTOR(jsb)));
			printf("slot: '%d' \n", (gint) clutter_actor_get_x(CLUTTER_ACTOR(jsb))/width_of_slot);
			fprintf(ofp,"%d\n", (gint) clutter_actor_get_x(CLUTTER_ACTOR(jsb))/width_of_slot);
			gchar* sound;
			gchar* image;
			g_object_get(jsb, "sample-filename", &sound, NULL);
			g_object_get(jsb, "image-filename", &image, NULL);
			printf("sound: '%s' \n",sound);
			printf("image: '%s'\n", image);
			fprintf(ofp,"%s\n",sound);
			fprintf(ofp,"%s\n",image);
		}
	}

	fclose(ofp);
	return FALSE;
}


static void on_cursor_notify_x(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleVault* vault;
	TangleScrollingActor* scrolling_actor;
	JammoCursor* cursor;
	gfloat x, visible, offset, max;

	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 2, TANGLE_TYPE_SCROLLING_ACTOR, &scrolling_actor, JAMMO_TYPE_CURSOR, &cursor);
	g_object_get(object, "x", &x, NULL);
	g_object_get(scrolling_actor, "width", &visible, "scrolling-offset-x", &offset, "scrolling-width", &max, NULL);
	if (x < offset) {
		offset = x - visible + 80.0;
		if (offset < 0.0) {
			offset = 0.0;
		}
		clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-x", offset, NULL);
	} else if (x > offset + visible - 80.0) {
		offset += 160.0;
		if (offset > max) {
			offset = max;
		}
		clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-x", offset, NULL);
	}
}

/***************************
Functions for json
**************************/



void fullsequencer_play_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Play-Stop-Button pressed\n");

	//tanglebutton is toggle-button.
	gboolean selected;
	g_object_get(tanglebutton, "selected", &selected,  NULL);

	if (selected) {
		jammo_sequencer_play(sequencer);
	}
	else {
		jammo_sequencer_stop(sequencer);
	}
}



//TODO
void fullsequencer_record_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Record pressed\n");
}


//TODO. use metronome-track
//is METRONOME_STATE used?
void fullsequencer_metronome_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Metronome pressed\n");

		//tanglebutton is toggle-button.
	gboolean selected;
	g_object_get(tanglebutton, "selected", &selected,  NULL);

	if (selected) {
		METRONOME_STATE = FALSE;
	}
	else {
		METRONOME_STATE = TRUE;
	}
}


void fullsequencer_community_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Community pressed. Doesn't do anything\n");
}

void fullsequencer_home_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Home pressed Doesn't do anything\n");

	/*
	jammo_sequencer_stop(sequencer);
	save_sequencer_state();
	leave_sequencer();

	*/
}



/*
Will make container (TangleWidget) and put it inside scroller.
Also prepare cursor.
and make volumesliders FIXME. they are bugging!
*/
void sequencer_track_view_create_area_for_tracks(JammoSequencer* seq, guint n_slots, guint64 total_duration, gfloat slot_width, gfloat height) {
	//Set statics
	sequencer = seq;
	number_of_slots = n_slots;
	duration_of_tracks = total_duration;
	width_of_slot = (int)slot_width;
	track_height =  height; //track_height is slot_height


	gfloat width_of_area = number_of_slots*width_of_slot;


	ClutterActor* track_container = jammo_get_actor_by_id("fullsequencer-container-for-tracks");
	//Tune width of track_container
	//Height will be 0 and when track is added it will be raised
	clutter_actor_set_size(track_container,width_of_area, 0.0);


	ClutterActor* scrolling_actor = jammo_get_actor_by_id("fullsequencer-scroller-for-tracks");

	//tune X-position of scroller
	gfloat x_place_scroller=194.0; //TODO //width_of: label+mute_button+volume_slider
	clutter_actor_set_x(scrolling_actor, x_place_scroller );
	//Height will be 0 and when track is added it will be raised
	clutter_actor_set_size(scrolling_actor, 800-x_place_scroller, 0.0);


	//Cursor:
	ClutterColor red = { 255, 0, 0, 128 };
	ClutterActor* cursor;

	texture_of_cursor = clutter_rectangle_new_with_color(&red); //this is static, because we want change it's height
	clutter_actor_set_size(texture_of_cursor, 10.0, 0.0 );
	cursor = jammo_cursor_new(sequencer, texture_of_cursor);
	tangle_actor_set_depth_position(TANGLE_ACTOR(cursor), 1);
	clutter_actor_set_position(cursor, 0.0, 0.0);
	clutter_actor_set_size(cursor, width_of_area, 80.0);
	tangle_widget_add(TANGLE_WIDGET(track_container), cursor,NULL);

	//For automatic scrolling on cursor movement
	TangleVault* vault;

	vault = tangle_vault_new(2, TANGLE_TYPE_SCROLLING_ACTOR, scrolling_actor, JAMMO_TYPE_CURSOR, cursor);
	tangle_signal_connect_vault(texture_of_cursor, "notify::x", G_CALLBACK(on_cursor_notify_x), vault);


	//Hide volumebar
	clutter_actor_hide(jammo_get_actor_by_id("fullsequencer-volumebar-container"));
}
