/*
 * jammo-track.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */
 
#include "jammo-track.h"
#include "jammo-meam-private.h"

#include "jammo-track.h"
#include "jammo-meam.h"

/**
 * SECTION:jammo-track
 * @Short_description: One track in a sequencer
 * @Title: JammoTrack
 *
 * A #JammoTrack consists of samples, notes or similar played
 * sequentially. An user of this class must make sure
 * that content does not overlap.
 *
 * A new #JammoTrack has floating reference after instantiation.
 * The reference will be sunk by a #JammoSequencer. Thus,
 * an application does not need to handle reference counting for
 * the #JammoTrack object.
 */
 
G_DEFINE_ABSTRACT_TYPE(JammoTrack, jammo_track, G_TYPE_INITIALLY_UNOWNED);

enum {
	PROP_0,
	PROP_DURATION
};

/**
 * jammo_track_get_duration:
 * @track: a #JammoTrack
 *
 * The total duration of the track in nanoseconds, or
 * JAMMO_DURATION_INVALID if the duration is not known.
 *
 * Implemented in subclasses.
 *
 * Return value: duration in nanoseconds.
 */
guint64 jammo_track_get_duration(JammoTrack* track) {
	guint64 duration = JAMMO_DURATION_INVALID;
	JammoTrackClass* track_class;

	track_class = JAMMO_TRACK_GET_CLASS(track);
	if (track_class->get_duration) {
		duration = track_class->get_duration(track);
	}
	
	return duration;
}

/**
 * _jammo_track_get_element:
 * @track: a #JammoTrack
 *
 * Internally used by a #JammoSequencer.
 *
 * Return value: a #GstElement of this track.
 */
GstElement* _jammo_track_get_element(JammoTrack* track) {
	GstElement* element = NULL;
	JammoTrackClass* track_class;

	track_class = JAMMO_TRACK_GET_CLASS(track);
	if (track_class->get_element) {
		element = track_class->get_element(track);
	}

	return element;
}

/**
 * _jammo_track_receive_message:
 * @track: a #JammoTrack
 * @message: a received #GstMessage
 *
 * Internally called by a #JammoSequencer when it receives
 * a new message.
 */
void _jammo_track_receive_message(JammoTrack* track, GstMessage* message) {
	JammoTrackClass* track_class;

	track_class = JAMMO_TRACK_GET_CLASS(track);
	if (track_class->receive_message) {
		track_class->receive_message(track, message);
	}
}

static void jammo_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoTrack* track;

	track = JAMMO_TRACK(object);

        switch (prop_id) {
		case PROP_DURATION:
			g_value_set_uint64(value, jammo_track_get_duration(track));
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_track_class_init(JammoTrackClass* track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(track_class);

	gobject_class->get_property = jammo_track_get_property;

	/**
	 * JammoTrack:duration:
	 *
	 * The total duration of the track in nanoseconds.
	 */
	g_object_class_install_property(gobject_class, PROP_DURATION,
	                                g_param_spec_uint64("duration",
	                                "Duration",
	                                "The duration of the track",
	                                0, G_MAXUINT64, JAMMO_DURATION_INVALID,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
}

static void jammo_track_init(JammoTrack* track) {
}
