/*
 * jammo-sample-button.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-sample-button.h"
#include "jammo-track-view.h"
#include <string.h>

G_DEFINE_TYPE(JammoSampleButton, jammo_sample_button, TANGLE_TYPE_DRAGGABLE_ACTOR);

enum {
	PROP_0,
	PROP_IMAGE_FILENAME,
	PROP_SAMPLE_FILENAME,
	PROP_SAMPLE,
	PROP_SAMPLE_TYPE,
	PROP_TEXT
};

struct _JammoSampleButtonPrivate {
	gchar* image_filename;
	JammoSample* sample;
	JammoSampleType sample_type;
};

static const GEnumValue sample_type_values[] = {
	{ JAMMO_SAMPLE_UNTYPED, "JAMMO_SAMPLE_UNTYPED", "untyped" },
	{ JAMMO_SAMPLE_RHYTMICAL, "JAMMO_SAMPLE_RHYTHMICAL", "rhytmical" },
	{ JAMMO_SAMPLE_MELODICAL, "JAMMO_SAMPLE_MELODICAL", "melodical" },
	{ JAMMO_SAMPLE_HARMONICAL, "JAMMO_SAMPLE_HARMONICAL", "harmonical" },
	{ JAMMO_SAMPLE_EFFECT, "JAMMO_SAMPLE_EFFECT", "effect" },
	{ 0, NULL, NULL }
};

GType jammo_sample_type_get_type(void) {
	static GType type = 0;
	
	if (!type) {
		type = g_enum_register_static("JammoSampleType", sample_type_values);
	}
	
	return type;
}

static gboolean on_clicked(TangleButton* button, gpointer user_data);
static void on_size_change(ClutterTexture* texture, gint width, gint height, gpointer user_data);

ClutterActor* jammo_sample_button_new() {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, NULL));
}

ClutterActor* jammo_sample_button_new_from_files(const gchar* image_filename, const gchar* sample_filename) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, "image-filename", image_filename, "sample-filename", sample_filename, NULL));
}

ClutterActor* jammo_sample_button_new_with_type(JammoSampleType sample_type) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, "sample-type", sample_type, NULL));
}

ClutterActor* jammo_sample_button_new_with_type_from_files(JammoSampleType sample_type, const gchar* image_filename, const gchar* sample_filename) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, "sample-type", sample_type, "image-filename", image_filename, "sample-filename", sample_filename, NULL));
}

ClutterActor* jammo_sample_button_new_from_existing(JammoSampleButton* sample_button) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, "image-filename", sample_button->priv->image_filename, "sample", sample_button->priv->sample, NULL));
}

const gchar* jammo_sample_button_get_image_filename(JammoSampleButton* sample_button) {

	return sample_button->priv->image_filename;
}

const gchar* jammo_sample_button_get_sample_filename(JammoSampleButton* sample_button) {

	return jammo_sample_get_filename(sample_button->priv->sample);
}

JammoSample* jammo_sample_button_get_sample(JammoSampleButton* sample_button) {

	return sample_button->priv->sample;
}

void jammo_sample_button_set_image_filename(JammoSampleButton* sample_button, const gchar* image_filename) {
	ClutterActor* texture;
	
	if (!sample_button->priv->image_filename ||
	    strcmp(sample_button->priv->image_filename, image_filename)) {
		g_free(sample_button->priv->image_filename);
		sample_button->priv->image_filename = g_strdup(image_filename);

		texture = clutter_texture_new_from_file(image_filename, NULL);
		tangle_button_set_normal_background_actor(TANGLE_BUTTON(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button))), texture);
		g_signal_connect(texture, "size-change", G_CALLBACK(on_size_change), sample_button);

		g_object_notify(G_OBJECT(sample_button), "image-filename");
	}
}

void jammo_sample_button_set_sample_filename(JammoSampleButton* sample_button, const gchar* sample_filename) {
	if (!sample_button->priv->sample ||
	    !jammo_sample_get_filename(sample_button->priv->sample) ||
	    strcmp(jammo_sample_get_filename(sample_button->priv->sample), sample_filename)) {
		if (sample_button->priv->sample) {
			g_object_unref(sample_button->priv->sample);
		}
		sample_button->priv->sample = jammo_sample_new_from_file(sample_filename);
		g_object_ref(sample_button->priv->sample);

		g_object_notify(G_OBJECT(sample_button), "sample-filename");
		g_object_notify(G_OBJECT(sample_button), "sample");
	}
}

void jammo_sample_button_set_sample(JammoSampleButton* sample_button, JammoSample* sample) {
	if (sample_button->priv->sample != sample) {
		if (sample_button->priv->sample) {
			g_object_unref(sample_button->priv->sample);
		}
		sample_button->priv->sample = jammo_sample_new_from_existing(sample);
		g_object_ref(sample_button->priv->sample);
		
		g_object_notify(G_OBJECT(sample_button), "sample-filename");
		g_object_notify(G_OBJECT(sample_button), "sample");
	
	}
}

JammoSampleType jammo_sample_button_get_sample_type(JammoSampleButton* sample_button) {

	return sample_button->priv->sample_type;
}

void jammo_sample_button_set_sample_type(JammoSampleButton* sample_button, JammoSampleType sample_type) {
	if (sample_button->priv->sample_type != sample_type) {
		sample_button->priv->sample_type = sample_type;
		g_object_notify(G_OBJECT(sample_button), "sample-type");
	}
}

void jammo_sample_button_get_background_color(JammoSampleButton* sample_button, ClutterColor* background_color) {
	tangle_widget_get_background_color(TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button))), background_color);
}

gboolean jammo_sample_button_get_background_color_set(JammoSampleButton* sample_button) {

	return tangle_widget_get_background_color_set(TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button))));
}

void jammo_sample_button_set_background_color(JammoSampleButton* sample_button, const ClutterColor* background_color) {
	tangle_widget_set_background_color(TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button))), background_color);
}

const gchar* jammo_sample_button_get_text(JammoSampleButton* sample_button) {
	const gchar* text = NULL;
	ClutterActor* actor;
	
	if ((actor = tangle_widget_get_foreground_actor(TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button))))) &&
	    CLUTTER_IS_TEXT(actor)) {
		text = clutter_text_get_text(CLUTTER_TEXT(actor));
	}
	
	return text;
}
	
void jammo_sample_button_set_text(JammoSampleButton* sample_button, const gchar* text) {
	TangleWidget* widget;
	ClutterActor* actor;

	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button)));
	if ((actor = tangle_widget_get_foreground_actor(widget))) {
		g_return_if_fail(CLUTTER_IS_TEXT(actor));
		
		if (g_strcmp0(clutter_text_get_text(CLUTTER_TEXT(actor)), text)) {
			if (text) {
				clutter_text_set_text(CLUTTER_TEXT(actor), text);
				g_object_notify(G_OBJECT(sample_button), "text");
			} else {
				tangle_widget_set_foreground_actor(widget, NULL);
				g_object_notify(G_OBJECT(sample_button), "text");
			}
		}
	} else if (text) {
		tangle_widget_set_foreground_actor(widget, clutter_text_new_with_text("Sans 12", text));
		g_object_notify(G_OBJECT(sample_button), "text");
	}
}

static gboolean jammo_sample_button_end_dragging(TangleDraggableActor* draggable_actor, TangleDragging* dragging) {
	gboolean retvalue = FALSE;
	JammoSampleButton* sample_button;
	ClutterActor* parent;
	ClutterActor* grand_parent;
	
	sample_button = JAMMO_SAMPLE_BUTTON(draggable_actor);
	
	parent = clutter_actor_get_parent(CLUTTER_ACTOR(sample_button));
	grand_parent = clutter_actor_get_parent(parent);
	if (JAMMO_IS_TRACK_VIEW(grand_parent)) {
		clutter_container_remove_actor(CLUTTER_CONTAINER(parent), CLUTTER_ACTOR(sample_button));
	} else {
		retvalue = TANGLE_DRAGGABLE_ACTOR_CLASS(jammo_sample_button_parent_class)->end_dragging(draggable_actor, dragging);
	}
	
	return retvalue;
}

static GObject* jammo_sample_button_constructor(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties) {
	GObject* object;
	ClutterActor* button;

	button = tangle_button_new();
	tangle_widget_set_prefer_background_size(TANGLE_WIDGET(button), TRUE);

	object = tangle_construct_with_extra_properties(G_OBJECT_CLASS(jammo_sample_button_parent_class)->constructor, type, n_construct_properties, construct_properties, "wrapped", button, NULL);

	g_signal_connect(button, "clicked", G_CALLBACK(on_clicked), object);

	return object;
}

static void jammo_sample_button_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoSampleButton* sample_button;
	
	sample_button = JAMMO_SAMPLE_BUTTON(object);

	switch (prop_id) {
		case PROP_IMAGE_FILENAME:
			jammo_sample_button_set_image_filename(sample_button, g_value_get_string(value));
			break;
		case PROP_SAMPLE_FILENAME:
			jammo_sample_button_set_sample_filename(sample_button, g_value_get_string(value));
			break;
		case PROP_SAMPLE:
			jammo_sample_button_set_sample(sample_button, JAMMO_SAMPLE(g_value_get_object(value)));
			break;
		case PROP_SAMPLE_TYPE:
			sample_button->priv->sample_type = g_value_get_enum(value);
			break;
		case PROP_TEXT:
			jammo_sample_button_set_text(sample_button, g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_sample_button_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoSampleButton* sample_button;

	sample_button = JAMMO_SAMPLE_BUTTON(object);

        switch (prop_id) {
		case PROP_IMAGE_FILENAME:
			g_value_set_string(value, jammo_sample_button_get_image_filename(sample_button));
			break;
		case PROP_SAMPLE_FILENAME:
			g_value_set_string(value, jammo_sample_button_get_sample_filename(sample_button));
			break;
		case PROP_SAMPLE:
			g_value_set_object(value, jammo_sample_button_get_sample(sample_button));
			break;
		case PROP_SAMPLE_TYPE:
			g_value_set_enum(value, sample_button->priv->sample_type);
			break;
		case PROP_TEXT:
			g_value_set_string(value, jammo_sample_button_get_text(sample_button));
			break;
		default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_sample_button_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_sample_button_parent_class)->finalize(object);
}

static void jammo_sample_button_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_sample_button_parent_class)->dispose(object);
}

static void jammo_sample_button_class_init(JammoSampleButtonClass* sample_button_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(sample_button_class);
	TangleDraggableActorClass* draggable_actor_class = TANGLE_DRAGGABLE_ACTOR_CLASS(sample_button_class);

	gobject_class->constructor = jammo_sample_button_constructor;
	gobject_class->finalize = jammo_sample_button_finalize;
	gobject_class->dispose = jammo_sample_button_dispose;
	gobject_class->set_property = jammo_sample_button_set_property;
	gobject_class->get_property = jammo_sample_button_get_property;

	draggable_actor_class->end_dragging = jammo_sample_button_end_dragging;

	/**
	 * JammoSampleButton:image-filename:
	 */
	g_object_class_install_property(gobject_class, PROP_IMAGE_FILENAME,
	                                g_param_spec_string("image-filename",
	                                "Image filename",
	                                "The filename of the displayed image",
					NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSampleButton:sample-filename:
	 */
	g_object_class_install_property(gobject_class, PROP_SAMPLE_FILENAME,
	                                g_param_spec_string("sample-filename",
	                                "Sample filename",
	                                "The filename of the played sample",
					NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSampleButton:sample:
	 */
	g_object_class_install_property(gobject_class, PROP_SAMPLE,
	                                g_param_spec_object ("sample",
	                                "Sample",
	                                "The JammoSample that is played when the button is interacting",
	                                JAMMO_TYPE_SAMPLE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSampleButton:sample-type:
	 *
	 * The type of the sample (not used internally).
	 */
	g_object_class_install_property(gobject_class, PROP_SAMPLE_TYPE,
	                                g_param_spec_enum("sample-type",
	                                "Sample type",
	                                "The type of the sample (not used internally)",
	                                JAMMO_TYPE_SAMPLE_TYPE, JAMMO_SAMPLE_UNTYPED,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSampleButton:text:
	 *
	 * A text that is displayed on top of the button.
	 */
	g_object_class_install_property(gobject_class, PROP_TEXT,
	                                g_param_spec_string("text",
	                                "Text",
	                                "A text that is displayed on top of the button",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(JammoSampleButtonPrivate));
}

static void jammo_sample_button_init(JammoSampleButton* sample_button) {
	sample_button->priv = G_TYPE_INSTANCE_GET_PRIVATE(sample_button, JAMMO_TYPE_SAMPLE_BUTTON, JammoSampleButtonPrivate);
}

static gboolean on_clicked(TangleButton* button, gpointer user_data) {
	JammoSampleButton* sample_button;
	
	sample_button = JAMMO_SAMPLE_BUTTON(user_data);

	jammo_sample_stop_all();
	jammo_sample_play(sample_button->priv->sample);
	
	return FALSE;
}

static void on_size_change(ClutterTexture* texture, gint width, gint height, gpointer user_data) {
	JammoSampleButton* sample_button;
	
	sample_button = JAMMO_SAMPLE_BUTTON(user_data);
	
	tangle_actor_set_aspect_ratio(TANGLE_ACTOR(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(sample_button))), width / height);
}
