/*
 * jammo-slider-track.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009 University of Oulu
 *
 * Authors: Aapo Rantalainen
 */
 
#include "jammo-slider-track.h"
#include "jammo-meam.h"
#include "jammo-meam-private.h"
#include <sys/time.h>

G_DEFINE_TYPE(JammoSliderTrack, jammo_slider_track, JAMMO_TYPE_PLAYING_TRACK);

enum {
	PROP_0,
	PROP_SLIDER_TYPE,
};

struct _JammoSliderTrackPrivate {
	GstElement* bin;
	GstElement* slider;
	GstElement* filter;
	int slider_type;
	struct timeval creation_time; //Not used yet
};


JammoSliderTrack* jammo_slider_track_new(JammoSliderType slider_type) {
	return JAMMO_SLIDER_TRACK(g_object_new(JAMMO_TYPE_SLIDER_TRACK,"slider-type",slider_type,NULL));
}

static guint64 jammo_slider_track_get_duration(JammoTrack* track) {
	guint64 duration = 0;

	/* TODO: Get real duration. */

	return duration;
}

static GstElement* jammo_slider_track_get_playing_element(JammoPlayingTrack* playing_track) {

	return JAMMO_SLIDER_TRACK(playing_track)->priv->bin;
}

static void jammo_slider_track_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoSliderTrack* slider_track;
	
	slider_track = JAMMO_SLIDER_TRACK(object);

	switch (prop_id) {
		case PROP_SLIDER_TYPE:
			slider_track->priv->slider_type = g_value_get_int(value);
			printf("set_property: slider=%d \n",g_value_get_int(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_slider_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
    JammoSliderTrack* slider_track;
		slider_track = JAMMO_SLIDER_TRACK(object);

        switch (prop_id) {
					case PROP_SLIDER_TYPE:
						g_value_set_int(value, slider_track->priv->slider_type);
						break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_slider_track_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_slider_track_parent_class)->finalize(object);
}

static void jammo_slider_track_dispose(GObject* object) {
	JammoSliderTrack* slider_track;

	slider_track = JAMMO_SLIDER_TRACK(object);
	G_OBJECT_CLASS(jammo_slider_track_parent_class)->dispose(object);
}


static GObject* jammo_slider_track_constructor(GType type, guint n_properties, GObjectConstructParam* properties) {
	GObject* object;
	JammoSliderTrack* slider_track;

	object = G_OBJECT_CLASS(jammo_slider_track_parent_class)->constructor(type, n_properties, properties);

	slider_track = JAMMO_SLIDER_TRACK(object);
	slider_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(slider_track, JAMMO_TYPE_SLIDER_TRACK, JammoSliderTrackPrivate);

	int slider_nro = slider_track->priv->slider_type;
	printf ("Type of slider is: %d\n",slider_nro);
	
	slider_track->priv->bin = gst_bin_new("slider-track");

	slider_track->priv->slider = gst_element_factory_make("jammoslider", NULL);
	slider_track->priv->filter = gst_element_factory_make("audiocheblimit", NULL);
	g_object_set(G_OBJECT(slider_track->priv->slider), "instrument", slider_nro, NULL);

	if (slider_nro == JAMMO_SLIDER_TYPE_KARPLUS) {
			g_object_set(G_OBJECT(slider_track->priv->slider), "attack", 0.002, NULL);
			g_object_set(G_OBJECT(slider_track->priv->slider), "decay", 0.002, NULL);
			g_object_set(G_OBJECT(slider_track->priv->slider), "sustain", 1.0, NULL);
			g_object_set(G_OBJECT(slider_track->priv->slider), "release", 0.002, NULL);
			// filter is low-pass by default
			g_object_set(slider_track->priv->filter, "cutoff", 10000.0, NULL);
	}
	
	if (slider_nro == JAMMO_SLIDER_TYPE_FM_MODULATION) {
			g_object_set(G_OBJECT(slider_track->priv->slider), "attack", 0.34, NULL);
			g_object_set(G_OBJECT(slider_track->priv->slider), "decay", 0.045, NULL);
			g_object_set(G_OBJECT(slider_track->priv->slider), "sustain", 0.8, NULL);
			g_object_set(G_OBJECT(slider_track->priv->slider), "release", 0.11, NULL);
			// filter low-pass by default
			g_object_set(slider_track->priv->filter, "cutoff", 1000.0, NULL);
	}
	
	g_object_set(G_OBJECT(slider_track->priv->slider), "recording", TRUE, NULL);
	
	gst_bin_add_many(GST_BIN(slider_track->priv->bin), slider_track->priv->slider, slider_track->priv->filter, NULL);
	gst_element_link(slider_track->priv->slider, slider_track->priv->filter);
	
	GstPad * pad = gst_element_get_pad(slider_track->priv->filter, "src");
	gst_element_add_pad(slider_track->priv->bin, gst_ghost_pad_new("src", pad));
	gst_object_unref(pad);

	gettimeofday(&slider_track->priv->creation_time, NULL);

	return object;
}


static void jammo_slider_track_class_init(JammoSliderTrackClass* slider_track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(slider_track_class);
	JammoTrackClass* track_class = JAMMO_TRACK_CLASS(slider_track_class);
	JammoPlayingTrackClass* playing_track_class = JAMMO_PLAYING_TRACK_CLASS(slider_track_class);

	track_class->get_duration = jammo_slider_track_get_duration;
	playing_track_class->get_playing_element = jammo_slider_track_get_playing_element;
	gobject_class->constructor = jammo_slider_track_constructor;
	gobject_class->finalize = jammo_slider_track_finalize;
	gobject_class->dispose = jammo_slider_track_dispose;
	gobject_class->set_property = jammo_slider_track_set_property;
	gobject_class->get_property = jammo_slider_track_get_property;

	/**
	 * JammoSliderTrack:slider-type:
	 */
	g_object_class_install_property(gobject_class, PROP_SLIDER_TYPE,
	                                g_param_spec_int("slider-type",
	                                "Slider type",
	                                "Specifies type of slider",
	                                0,JAMMO_SLIDER_TYPE_DUMMY-1,0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	
	g_type_class_add_private(gobject_class, sizeof(JammoSliderTrackPrivate));
}

void jammo_slider_track_add_event(JammoSliderTrack* slider_track,float freq, JammoSliderEventType type, GstClockTime timestamp) {
	JammoSliderEvent* new_event = malloc(sizeof(JammoSliderEvent));
	new_event->freq = freq;
	new_event->timestamp = timestamp;
	new_event->type = type;

	g_object_set(G_OBJECT(slider_track->priv->slider), "add-event", new_event, NULL);
	g_free(new_event);
}

void jammo_slider_track_remove_event(JammoSliderTrack* slider_track,float freq, JammoSliderEventType type, GstClockTime timestamp) {
	JammoSliderEvent* new_event = malloc(sizeof(JammoSliderEvent));
	new_event->freq = freq;
	new_event->type = type;
	new_event->timestamp = timestamp;

	g_object_set(G_OBJECT(slider_track->priv->slider), "remove-event", new_event, NULL);
	g_free(new_event);
}

void jammo_slider_track_set_frequency_realtime(JammoSliderTrack* slider_track,float frequency){
	jammo_slider_track_add_event(slider_track, frequency, JAMMO_SLIDER_EVENT_MOTION, 18446744073709551615ULL); //MAX
}

void jammo_slider_track_set_on_realtime(JammoSliderTrack* slider_track, float frequency){
	jammo_slider_track_add_event(slider_track, frequency, JAMMO_SLIDER_EVENT_ON, 18446744073709551615ULL); //MAX
}

void jammo_slider_track_set_off_realtime(JammoSliderTrack* slider_track, float frequency){
	jammo_slider_track_add_event(slider_track, frequency, JAMMO_SLIDER_EVENT_OFF, 18446744073709551615ULL); //MAX
}

static void jammo_slider_track_init(JammoSliderTrack* slider_track) {
	slider_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(slider_track, JAMMO_TYPE_SLIDER_TRACK, JammoSliderTrackPrivate);
}

GList * jammo_slider_track_get_event_list(JammoSliderTrack* slider_track) {
	GList * eventlist;
	g_object_get(G_OBJECT(slider_track->priv->slider),"eventlist", &eventlist, NULL);

	return eventlist;
}

void jammo_slider_track_set_event_list(JammoSliderTrack* slider_track, GList* eventlist) {
	g_object_set(G_OBJECT(slider_track->priv->slider),"eventlist", eventlist, NULL);
}

void jammo_slider_track_set_recording(JammoSliderTrack* slider_track, gboolean recording) {
	g_object_set(G_OBJECT(slider_track->priv->slider),"recording", recording, NULL);
}

void jammo_slider_track_set_live(JammoSliderTrack* slider_track, gboolean state){
	g_object_set(G_OBJECT(slider_track->priv->slider), "is-live", state, NULL);
}

/*
Debug. Prints all events in this track.
*/
void jammo_slider_track_dump_events(JammoSliderTrack* slider_track){
	GList* list= jammo_slider_track_get_event_list(JAMMO_SLIDER_TRACK(slider_track));
	JammoSliderEvent* n;

	printf("Track:\n");
	for (; list; list = list->next) {
			n = (list->data);
			printf(" %llu ns: freq:%f ,type: %d \n", (unsigned long long)n->timestamp,n->freq, n->type);
	}
	printf("/Track\n\n");
}

/**
	Clears event list of sampler. All recorded and added events are lost.
**/
void jammo_slider_track_clear_events(JammoSliderTrack* slider_track) {
	g_object_set(slider_track->priv->slider, "clear-events", TRUE, NULL);
}
