/*
 * communication.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Tommi Kallonen <tommi.kallonen@lut.fi>
 *          Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#include "communication.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>
#include <glib.h>
#include <peerhood/IteratorWrapper.h>

#include "gems_message_functions.h"
#include "gems_security.h"
#include "groupmanager.h"
#include "../cem/cem.h"
#include "gems_service_collaboration.h"

#define SOCKET_ERROR        -1
#define HOST_NAME_SIZE      255
#define QUEUE_SIZE          5
#define BUFFER_SIZE         1000
#define JAMMOPORT		5555

GTimer* testtimer;

/*int connected=0;
int hServerSocket; //Listening socket

//FDSet stuff for select(serverthread)
fd_set master;    // master file descriptor list
fd_set read_fds;  // temp file descriptor list for select()
int fdmax;        // maximum file descriptor number
int noconnections;
struct connection* connectionlist;
int serverrunning=0;*/

void * serverthread(void *);

int communication_create_connection(char *address, int port, ip_communication* communicationdata)
{
	struct hostent* pHostInfo;   /* holds info about a machine */
	struct sockaddr_in Address;  /* Internet socket address stuct */
	char buffer[100];
    	long nHostAddress;
	char strHostName[HOST_NAME_SIZE];
	int rc;
    	pthread_t threadid;

	if(strlen(address)<HOST_NAME_SIZE)
	{
		strcpy(strHostName,address);
	}
     	else
	{
		printf("\nHost name too long\n");
		return -1;
	}
	
	communicationdata->connectionlist = (struct connection *)realloc(communicationdata->connectionlist, (communicationdata->noconnections +1) * sizeof(struct connection));

	communicationdata->connectionlist[communicationdata->noconnections].userid=communicationdata->noconnections;
	
	FD_ZERO(&communicationdata->master);    // clear the master and temp sets
    	FD_ZERO(&communicationdata->read_fds);

    	printf("\nMaking a socket");
    	/* make a socket */
    	communicationdata->connectionlist[communicationdata->noconnections].sockethandle=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);

    	if(communicationdata->connectionlist[communicationdata->noconnections].sockethandle == SOCKET_ERROR)
    	{
        	printf("\nCould not make a socket\n");
        	return 0;
    	}


    	/* get IP address from name */
    	pHostInfo=gethostbyname(strHostName);
   	/* copy address into long */
    	memcpy(&nHostAddress,pHostInfo->h_addr,pHostInfo->h_length);

    	/* fill address struct */
    	Address.sin_addr.s_addr=nHostAddress;
    	Address.sin_port=htons(port);
    	Address.sin_family=AF_INET;

    	printf("\nConnecting to %s on port %d",strHostName,port);

    	/* connect to host */
	

    	if(connect(communicationdata->connectionlist[communicationdata->noconnections].sockethandle,(struct sockaddr*)&Address,sizeof(Address))== SOCKET_ERROR)
    	{
        	printf("\nCould not connect to host\n");
        	return 0;
    	}

 	FD_SET(communicationdata->connectionlist[communicationdata->noconnections].sockethandle, &communicationdata->master);
	communicationdata->fdmax = communicationdata->connectionlist[communicationdata->noconnections].sockethandle;
	rc = pthread_create(&threadid, NULL, serverthread, NULL);
	communicationdata->connected=1;
	communicationdata->noconnections++;
	return 1;
}
int communication_send_data(int id, char *message, int size, ip_communication* communicationdata)
{
	gint32 messagesize;
	gint32 nomessages;
	int i, location;
	char* data;

	nomessages=size/(BUFFER_SIZE-2*sizeof(gint32)); //Send in chunks of BUFFER_SIZE bytes
	if(size%(BUFFER_SIZE-2*sizeof(int))!=0)
		nomessages++;
	printf("Sending %d messages.\n",nomessages);
	location=0;
	for (i=0;i<nomessages;i++)
	{
		if(location+(BUFFER_SIZE-2*sizeof(gint32))>size)
			messagesize=size-location+2*sizeof(gint32);
		else
			messagesize=BUFFER_SIZE;
			
		data=(char *)malloc(messagesize);
		memcpy(data,&messagesize,sizeof(gint32));
		memcpy(data+sizeof(gint32),&nomessages,sizeof(gint32));
		memcpy(data+2*sizeof(gint32),message+location,messagesize-2*sizeof(gint32));


		if(communicationdata->connected)
		{
			printf("Sending %d bytes.",size);
			write(communicationdata->connectionlist[id].sockethandle,data,messagesize);
		}
		else
		{
			printf("No connection available");
			return -1;
		}
		free(data);
	}
	return nomessages;
}


int communication_start_server(int id, int port, ip_communication* communicationdata)
{
    struct hostent* pHostInfo;   /* holds info about a machine */
    struct sockaddr_in Address; /* Internet socket address stuct */
    int nAddressSize=sizeof(struct sockaddr_in);
    int rc,i;
    pthread_t threadid;

    if(communicationdata->serverrunning==1)
    {
	printf("\nServer allready running\n");
	return 0;
    }

    printf("\nStarting server");

    FD_ZERO(&communicationdata->master);    // clear the master and temp sets
    FD_ZERO(&communicationdata->read_fds);

    printf("\nMaking socket");
    /* make a socket */
    communicationdata->hServerSocket=socket(AF_INET,SOCK_STREAM,0);

    if(communicationdata->hServerSocket == SOCKET_ERROR)
    {
        printf("\nCould not make a socket\n");
        return 0;
    }

    /* fill address struct */
    Address.sin_addr.s_addr=INADDR_ANY;
    Address.sin_port=htons(port);
    Address.sin_family=AF_INET;

    printf("\nBinding to port %d\n",port);

    /* bind to a port */
    if(bind(communicationdata->hServerSocket,(struct sockaddr*)&Address,sizeof(Address)) 
                        == SOCKET_ERROR)
    {
        printf("\nCould not bind\n");
        return 0;
    }
 /*  get port number */
    getsockname( communicationdata->hServerSocket, (struct sockaddr *) &Address,(socklen_t *)&nAddressSize);
    printf("opened socket as fd (%d) on port (%d) for stream i/o\n",communicationdata->hServerSocket, ntohs(Address.sin_port) );

        printf("Server\n\
              sin_family        = %d\n\
              sin_addr.s_addr   = %d\n\
              sin_port          = %d\n"
              , Address.sin_family
              , Address.sin_addr.s_addr
              , ntohs(Address.sin_port)
            );
     printf("\nMaking a listen queue of %d elements",QUEUE_SIZE);
    /* establish listen queue */
    if(listen(communicationdata->hServerSocket,QUEUE_SIZE) == SOCKET_ERROR)
    {
        printf("\nCould not listen\n");
        return 0;
    }

    printf("\nWaiting for a connection\n");
    FD_SET(communicationdata->hServerSocket, &communicationdata->master);     
    communicationdata->fdmax = communicationdata->hServerSocket;
     
     communicationdata->connected=0;
     communicationdata->serverrunning=1;
     communicationdata->noconnections=0;
     rc = pthread_create(&threadid, NULL, serverthread, (void *) communicationdata);//The listening thread is started     
     return 1;
   
}

void * serverthread(void * arg)//Communication thread, creates listening socket and listens to open sockets for incoming messages
{
	gint32 size,i;
	int id=0;
	char buffer[BUFFER_SIZE];
	FILE *logfile;
	char* packetinfo;
	gint32 messagesize;
	gint32 nomessages;
	struct sockaddr_in Address; /* Internet socket address stuct */
	int nAddressSize=sizeof(struct sockaddr_in);
	ip_communication* communicationdata;

	packetinfo=(char *)malloc(2*sizeof(int));
	communicationdata=(ip_communication*)arg;


	for(;;)
    	{	
		communicationdata->read_fds = communicationdata->master; // copy it
		
        	if (select(communicationdata->fdmax+1, &communicationdata->read_fds, NULL, NULL, NULL) == -1) {
			if (errno != EINTR) {
            			perror("select");
            			exit(4);
			}
        	}
		for(i = 0; i <= communicationdata->fdmax; i++) {
			if (FD_ISSET(i, &communicationdata->read_fds)) {
				if (i == communicationdata->hServerSocket) {//New connection
					communicationdata->connectionlist = (struct connection *)realloc(communicationdata->connectionlist, (communicationdata->noconnections+1) * sizeof(struct connection));

					communicationdata->connectionlist[communicationdata->noconnections].sockethandle=accept(communicationdata->hServerSocket,(struct sockaddr*)&Address,(socklen_t *)&nAddressSize);

				     	printf("\nGot a connection");   
				     	printf("Server\n\
					      sin_family        = %d\n\
					      sin_addr.s_addr   = %d\n\
					      sin_port          = %d\n"
					      , Address.sin_family
					      , Address.sin_addr.s_addr
					      , ntohs(Address.sin_port)
					    );  
				     	communicationdata->connectionlist[communicationdata->noconnections].userid=id;

				     	FD_SET(communicationdata->connectionlist[communicationdata->noconnections].sockethandle, &communicationdata->master);     
				     	communicationdata->fdmax = communicationdata->connectionlist[communicationdata->noconnections].sockethandle;
				     	communicationdata->connected=1;
					communicationdata->noconnections++;
				}
				else //incoming data
				{
					int j;
					//What connections is sending?
					while(j<communicationdata->noconnections)
					{
						if (i == communicationdata->connectionlist[j].sockethandle)
						break;
					}					


					// read packetinfo 
					size=read(communicationdata->connectionlist[j].sockethandle,packetinfo,2*sizeof(gint32));
					memcpy(&messagesize,packetinfo,sizeof(gint32));
					memcpy(&nomessages,packetinfo+sizeof(gint32),sizeof(gint32));
					printf("\nReceived %d bytes. Messagesize: %d, NoMessages: %d",size,messagesize,nomessages);

					size=read(communicationdata->connectionlist[j].sockethandle,buffer,BUFFER_SIZE);
					printf("\nReceived %d bytes.",size);
					if(size<=0)//connection breaks
					{
						printf("\nLost connection.");
						communication_close_connection();
					}
					collaboration_handle_message(buffer,size,id);
		
				}
			}


		}

		
	}
}

int communication_close_connection(int id, ip_communication* communicationdata)
{
	printf("\nClosing socket\n");
    	/* close socket */                       
    	if(close(communicationdata->connectionlist[id].sockethandle) == SOCKET_ERROR)
    	{
        	printf("\nCould not close socket\n");
		exit(-1);
    	}
	communicationdata->noconnections--;
	if(communicationdata->noconnections==0)
		communicationdata->connected=0;
	
	return 1;
}

gems_communication_data* gems_communication_init()
{
	// Communication data struct
	gems_communication_data* data = (gems_communication_data*)g_malloc(sizeof(gems_communication_data));
	
	// Communication list
	data->connections = NULL;
	
	// Connecting list
	data->connecting = NULL;
	
	// Rejected list
	data->rejected = NULL;
	
	testtimer = g_timer_new();
	
	return data;
}

void gems_communication_cleanup()
{	
	gems_communication_data* data = gems_get_data()->communication;
	g_timer_destroy(testtimer);
	
	if(data == NULL) return;
	
	gems_communication_cleanup_lists();
	
	g_free(data);
	data = NULL;
}

void gems_communication_cleanup_lists()
{
	gems_communication_data* data = gems_get_data()->communication;
	
	if(data->connections != NULL)
	{
		g_list_foreach(data->connections, (GFunc)gems_clear_gems_connection, NULL);
		g_list_free(data->connections);
		data->connections = NULL;
	}
	
	if(data->connecting != NULL)
	{
		g_list_foreach(data->connecting, (GFunc)gems_clear_gems_connection, NULL);
		g_list_free(data->connecting);
		data->connecting = NULL;
	}
	
	if(data->rejected != NULL)
	{
		g_list_foreach(data->rejected, (GFunc)gems_clear_gems_connection, NULL);
		g_list_free(data->rejected);
		data->rejected = NULL;
	}
}

gint gems_communication_search_jammos()
{
	gchar* logmsg = NULL;
	gems_components* data = gems_get_data();
	g_timer_start(testtimer);	
	
	if(data == NULL) return FALSE;
	if(data->ph == NULL) return FALSE; // no PeerHood = no network
	
	// Get surrounding devices with service JAMMO_SERVICE_NAME 
	TDeviceList* list = ph_c_get_devicelist_with_services(data->ph,JAMMO_SERVICE_NAME);
	
	// Got a NULL reference - error with PeerHood -> stop it
	if(list == NULL)
	{
		logmsg = g_strdup_printf("gems_communication_search_jammos: connection to PeerHood daemon was lost.");
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
		gems_communication_peerhood_close_detected();
		return FALSE;
	}
	
	if(ph_c_devicelist_is_empty(list) == FALSE)
	{
		// Initialize wrapper for iterator for list
		DeviceIterator* iterator = ph_c_new_device_iterator(list);
		
		if(iterator == NULL)
		{
			ph_c_delete_devicelist(list);
			return TRUE;
		}

		// Go through list
		for(iterator = ph_c_new_device_iterator(list); ph_c_device_iterator_is_last(iterator,list) != TRUE; ph_c_device_iterator_next_iterator(iterator))
		{
			// Get the device
			MAbstractDevice* dev = ph_c_device_iterator_get_device(iterator);
			
			// Through WLAN and has peerhood? - all devices in the list should have ph enabled
			if((strcmp(ph_c_device_get_prototype(dev),"wlan-base") == 0) && (ph_c_device_has_peerhood(dev) == TRUE))
			{
				if(gems_communication_already_connected(ph_c_device_get_checksum(dev)) == NOT_IN_LIST)
				{
					if(gems_communication_establish_connection(dev) == TRUE)
					{
							logmsg = g_strdup_printf("gems_communication_search_jammos: Connected to new JamMo enabled device %s/%d",
								ph_c_device_get_name(dev),ph_c_device_get_checksum(dev));
							cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
							g_free(logmsg);
					}
				}
			}
		}
		// Delete the iterator
		ph_c_delete_device_iterator(iterator);
	}
	// Delete the list of devices
	ph_c_delete_devicelist(list);
	
	return TRUE;
	
	if(g_timer_elapsed(testtimer, NULL) > 0.020)
	{
		logmsg = g_strdup_printf("search_jammos %f",g_timer_elapsed(testtimer,NULL));
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
}

gboolean gems_communication_establish_connection(MAbstractDevice* device)
{
	gchar* logmsg = NULL;
	gems_components* data = gems_get_data();
	
	MAbstractConnection* conn = NULL;

	if(gems_connection_error_attempt_connection(data->connection_errorlist,ph_c_device_get_checksum(device)) == TRUE)
	{
		if((conn = ph_c_connect_remoteservice(data->ph, device, JAMMO_SERVICE_NAME)) != NULL)
		{
			// Connection success, remove from errors. If no such device in the list, nothing is done
			data->connection_errorlist = gems_connection_error_remove(data->connection_errorlist,ph_c_device_get_checksum(device)); 
			
			gems_message* msg = gems_create_message_service_jammo(CONNECT_REQ);

			if(gems_communication_write_data(ph_c_connection_get_fd(conn),msg) != msg->length)
			{
				logmsg = g_strdup_printf("gems_communication_gems_communication_establish_connection: Error writing CONNECT_REQ to JamMo service");
				cem_add_to_log(logmsg, LOG_ERROR);
				g_free(logmsg);
				
				data->connection_errorlist = gems_connection_error_add(data->connection_errorlist,ph_c_device_get_checksum(device), ERROR_CONNECTION);
				ph_c_connection_delete(conn); // disconnects and deletes
				conn = NULL;
				return FALSE;
			}
			else
			{
				logmsg = g_strdup_printf("gems_communication_gems_communication_establish_connection: Wrote CONNECT_REQ to JamMo service in %s",
					ph_c_connection_get_remote_address(conn));
				cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
				g_free(logmsg);
				
				logmsg = g_strdup_printf("bytes: %d [%d|%d|%d]", msg->length, ntohs(*(gint16*)&msg->message[0]),ntohl(*(gint32*)&msg->message[2]),ntohs(*(gint16*)&msg->message[6]));
				cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
				g_free(logmsg);
				
				// New item in JAMMO_WAITING_REQUEST state
				gems_connection* new_item = gems_new_gems_connection(conn, 0, 0, JAMMO_WAITING_REQUEST, 0);
				data->communication->connecting = g_list_append(data->communication->connecting, new_item);
				return TRUE;
			}
		}
		else
		{
			data->connection_errorlist = gems_connection_error_add(data->connection_errorlist,ph_c_device_get_checksum(device), ERROR_CONNECTION);
			return FALSE;
		}
	}
	else
	{
		//printf("gems_communication_establish_connection: timer has not passed");
		return FALSE;
	}
}

gint gems_communication_retrieve_profiles()
{
	gems_components* data = gems_get_data();
	g_timer_start(testtimer);
	
	if(data == NULL) return FALSE;
	if(data->ph == NULL) return FALSE;
	if(data->communication == NULL) return FALSE;
	
	gboolean cleanup_required = FALSE;
	gems_connection* element = NULL;
	gchar* logmsg = NULL;
	
	if (data->communication->connections != NULL)	
	{
		GList* iter = NULL;
		// Go through list
		for(iter = g_list_first(data->communication->connections); iter; iter = g_list_next(iter))
		{
			element = (gems_connection*)iter->data;
			// If we're already connected we can send profile request
			if(element != NULL)
			{
				// Connected and authenticated & no profile requested
				if(element->connection_state == JAMMO_CONNECTED_AUTH && element->profile == NULL)
				{
					// Request already done -> timer should have been set
					if((element->profile_requests > 0) && (g_timer_elapsed(element->profile_request_timer,NULL) < PROFILE_REQUEST_INTERVAL)) continue;

					//printf("gems_communication_retrieve_profiles: sending GET_PROFILE_PUB (%f)",g_timer_elapsed(element->profile_request_timer,NULL));

					gems_message* request = gems_create_message_profile_request(GET_PROFILE_PUB);

					if(gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE, element, request) == FALSE)
					{
						logmsg = g_strdup_printf("gems_communication_retrieve_profiles: cannot write GET_PROFILE_PUB");
						cem_add_to_log(logmsg, LOG_ERROR);
						g_free(logmsg);
						ph_c_connection_disconnect(element->connection);
						cleanup_required = TRUE;
					}
					else
					{
						/*printf("gems_communication_retrieve_profiles: wrote GET_PROFILE_PUB [%d|%d|%d|%d]",
							ntohs(*(gint16*)&request->message[0]),
							ntohl(*(gint32*)&request->message[2]),
							ntohs(*(gint16*)&request->message[6]),
							ntohs(*(gint16*)&request->message[8]));*/
						element->profile_requests += 1;
						g_timer_start(element->profile_request_timer);
					}

					gems_clear_message(request);
				}
			}
		}
	}
	
	if(cleanup_required == TRUE) data->communication->connections = gems_cleanup_connections_from_list(data->communication->connections);
	
	return TRUE;
	
	if(g_timer_elapsed(testtimer, NULL) > 0.020)
	{
		logmsg = g_strdup_printf("retrieve_profiles %f",g_timer_elapsed(testtimer,NULL));
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
}

gint gems_communication_request_groups()
{
	gems_components* data = gems_get_data();
	g_timer_start(testtimer);

	if(data == NULL) return FALSE;
	if(data->ph == NULL) return FALSE;
	if(data->communication == NULL) return FALSE;
	
	gboolean cleanup_required = FALSE;
	gems_connection* element = NULL;
	gchar* logmsg = NULL;
	
	// TODO change to rely on the list we have?
	// Get surrounding devices with service GROUP_SERVICE_NAME 
	TDeviceList* list = ph_c_get_devicelist_with_services(data->ph,GROUP_SERVICE_NAME);
	
	// Got a NULL reference - error with PeerHood -> stop it
	if(list == NULL)
	{
		logmsg = g_strdup_printf("gems_communication_request_groups: connection to PeerHood daemon was lost.");
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
		gems_communication_peerhood_close_detected();
		return FALSE;
	}
	
	if(ph_c_devicelist_is_empty(list) == FALSE)
	{
		// Initialize wrapper for iterator for list
		DeviceIterator* iterator = NULL;

		if(iterator == NULL)
		{
			ph_c_delete_devicelist(list);
			return TRUE;
		}
		
		// Go through list
		for(iterator = ph_c_new_device_iterator(list); ph_c_device_iterator_is_last(iterator,list) != TRUE; ph_c_device_iterator_next_iterator(iterator))
		{
			// Get the device
			MAbstractDevice* dev = ph_c_device_iterator_get_device(iterator);
			
			// Through WLAN and has peerhood? - all devices in the list should have ph enabled
			if((strcmp(ph_c_device_get_prototype(dev),"wlan-base") == 0) && (ph_c_device_has_peerhood(dev) == TRUE))
			{
				switch(gems_communication_already_connected(ph_c_device_get_checksum(dev)))
				{
					// Not connected ?
					case NOT_IN_LIST:
						// Try to establish connection
						if(gems_communication_establish_connection(dev) == TRUE)
						{
							logmsg = g_strdup_printf("gems_communication_request_groups: Connected to new JamMo enabled device %s/%d",
								ph_c_device_get_name(dev),ph_c_device_get_checksum(dev));
							cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
							g_free(logmsg);
						}
						break;
					case IN_CONNECTED_LIST:
						element = gems_communication_get_connection(data->communication->connections, ph_c_device_get_checksum(dev));

						// If we're already connected we can send group request
						if(element != NULL)
						{
							// Connected and authenticated & profile is requested
							if(element->connection_state == JAMMO_CONNECTED_AUTH && element->profile != NULL)
							{
								// Do not send the request to group members
								if(gems_group_is_in_group(element->profile->id) == FALSE)
								{
									// Request already done -> timer should have been set
									if(g_timer_elapsed(element->group_request_timer,NULL) < GROUP_REQUEST_INTERVAL) continue;

									//printf("gems_communication_request_groups: sending REQUEST_GROUP_INFO (%f)",g_timer_elapsed(element->group_request_timer,NULL));

									gems_message* request = gems_create_message_group_management_notify(REQUEST_GROUP_INFO,0,0);

									if(gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE, element, request) == FALSE)
									{
										logmsg = g_strdup_printf("gems_communication_request_groups: cannot write REQUEST_GROUP_INFO");
										cem_add_to_log(logmsg, LOG_ERROR);
										g_free(logmsg);
										ph_c_connection_disconnect(element->connection);
										cleanup_required = TRUE;
									}
									else
									{
										/*printf("gems_communication_request_groups: wrote REQUEST_GROUP_INFO [%d|%d|%d]",
											ntohs(*(gint16*)&request->message[0]),
											ntohl(*(gint32*)&request->message[2]),
											ntohs(*(gint16*)&request->message[6]));*/
										g_timer_start(element->group_request_timer);
									}

									gems_clear_message(request);
								}
							}
						}
						break;
					default:
						break;
				} // switch
			} // for
		}
		
		// Delete the iterator
		ph_c_delete_device_iterator(iterator);
	}
	// Delete the list of devices
	ph_c_delete_devicelist(list);
	
	if(cleanup_required == TRUE) data->communication->connections = gems_cleanup_connections_from_list(data->communication->connections);
	
	if(g_timer_elapsed(testtimer, NULL) > 0.020)
	{
		logmsg = g_strdup_printf("retrieve_groups %f",g_timer_elapsed(testtimer,NULL));
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
	
	return TRUE;
}

gint gems_communication_advert_group()
{
	gems_components* data = gems_get_data();
	
	if(data == NULL) return FALSE;
	if(data->ph == NULL) return FALSE;
	if(data->communication == NULL) return FALSE;
	
	g_timer_start(testtimer);
	gchar* logmsg = NULL;
	
	// Group not enabled
	if(data->service_group->group_info->id == NOT_ACTIVE)
	{
		// Destroy the timer and remove self from glib main loop
		if(data->service_group->group_info->group_advert_timer != NULL)
		{
			g_timer_destroy(data->service_group->group_info->group_advert_timer);
			data->service_group->group_info->group_advert_timer = NULL;
		}
		return FALSE;
	}
	// Group locked - do not advert but do not remove this since group is still active
	else if(data->service_group->group_info->is_locked) return TRUE;
	else
	{
		// Was set up
		if(data->service_group->group_info->group_advert_timer != NULL)
		{
			// Enough time has not passed from previous advert -> return
			if(g_timer_elapsed(data->service_group->group_info->group_advert_timer,NULL) < GROUP_ADVERT_INTERVAL) return TRUE;
		}
		else data->service_group->group_info->group_advert_timer = g_timer_new(); // Start the timer
		
		gems_message* msg = gems_create_message_group_management_group_info(OFFER_GROUP);
		
		// TODO after applying GObject to get the necessary data (gems_components) change to use g_list_foreach
		// Send message to all
		GList* iterator = NULL;
		
		for(iterator = g_list_first(data->communication->connections); iterator ; iterator = g_list_next(iterator))
		{
			gems_connection* element = (gems_connection*)iterator->data;
			
			if(element->profile != NULL)
			{
				if(gems_group_is_in_group(element->profile->id) == FALSE)
				{					
					if(gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE, element, msg) == FALSE)
					{
						gchar* logmsg = g_strdup_printf("gems_communication_advert_group: cannot send OFFER_GROUP to device %d with user %d",
							ph_c_connection_get_device_checksum(element->connection), element->profile->id);
						cem_add_to_log(logmsg, LOG_ERROR);
						g_free(logmsg);
						// TODO add error handling - add connection error to errorlist
					}
					// TODO add last action for this user, timestamp etc?
				}
			}
		}
		
		g_timer_start(data->service_group->group_info->group_advert_timer); // Set new time
		
		gems_clear_message(msg);
	}
	
	if(g_timer_elapsed(testtimer, NULL) > 0.020)
	{
		logmsg = g_strdup_printf("advert_group %f",g_timer_elapsed(testtimer,NULL));
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
	
	return TRUE;
}

gint gems_communication_sanitize_grouplist()
{
	gems_components* data = gems_get_data();
	if(data == NULL) return FALSE;
	if(data->ph == NULL) return FALSE;
	if(data->communication == NULL) return FALSE;
	if(data->service_group == NULL) return FALSE;
	
	GList* iterator = g_list_first(data->service_group->other_groups);
	
	while(iterator)
	{
		gems_group_info* info = (gems_group_info*)iterator->data;
		
		if(info == NULL) continue;
		
		if(info->group_advert_timer != NULL)
		{
			if(g_timer_elapsed(info->group_advert_timer,NULL) > GROUP_REMOVE_LIMIT)
			{
				gchar* logmsg = g_strdup_printf("Group %u is past remove limit (%f elapsed), removing.",info->id, g_timer_elapsed(info->group_advert_timer,NULL));
				cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
				g_free(logmsg);
				
				GList* tempiter = iterator;
				iterator = g_list_next(iterator);
				
				data->service_group->other_groups = g_list_remove_link(data->service_group->other_groups,tempiter);
				gems_clear_group_info(info);
				g_list_free_1(tempiter);
				
				continue;
			}
		}
		
		iterator = g_list_next(iterator);
	}

	return TRUE;
}

gint gems_communication_process_connections()
{
	gems_components* data = gems_get_data();
	if(data == NULL) return FALSE;
	if(data->ph == NULL) return FALSE;
	if(data->communication == NULL) return FALSE;
	
	// TODO after applying GObject to get the necessary data (gems_components) change to use g_list_foreach
	// Check connecting list
	data->communication->connecting = gems_service_jammo_process_list(data->communication->connecting, FALSE);
	
	// TODO data->communication->connections
	
	gems_communication_process_connected();
	
	data->communication->rejected = gems_process_rejected(data->communication->rejected);
	
	return TRUE;
}

void print_list(GList* list)
{
	GList* iterator = NULL;
	
	if(list == NULL)
	{
		printf("LIST NULL\n");
		return;
	}
	
	if(g_list_length(list) == 0)
	{
		printf("LIST EMPTY\n");
	}
	// Search the list
	for(iterator = g_list_first(list); iterator; iterator = g_list_next(iterator))
	{
		gems_connection* element = (gems_connection*)iterator->data;
		
		if(element == NULL) continue;
		
		printf("%d: state: %d (data in buffer: %d bytes) | %s / %d\n",
			g_list_position(list,iterator), element->connection_state, gems_connection_partial_data_amount(element),
			ph_c_connection_get_remote_address(element->connection), ph_c_connection_get_device_checksum(element->connection));
		if(element->profile != NULL) printf("\tProfile: %u:%s, %dyo avatarid: %d\n",
			element->profile->id,element->profile->username,element->profile->age,element->profile->avatarid);
	}
}

void print_group(gems_group_info* info)
{
	printf("gid:\t%u\nowner:\t%u\ntype:\t%d\ntheme:\t%d\nsize:\t%d\nspaces\t%d\nlocked:\t%s\n",
		info->id,
		info->owner,
		info->type,
		info->theme,
		info->size,
		info->spaces,
		(info->is_locked ? "yes" : "no" ));
	if(info->group_advert_timer != NULL) printf("advert:\t%f s ago\n",g_timer_elapsed(info->group_advert_timer,NULL));
	switch(info->own_state)
	{
		case NOT_ACTIVE:
			printf("state:\tNOT_ACTIVE\n");
			break;
		case IN_GROUP:
			printf("state:\tIN_GROUP\n");
			break;
		case JOINING:
			printf("state:\tJOINING\n");
			break;
		case LEAVING:
			printf("state:\tLEAVING\n");
			break;
		default:
			break;
	}	
	guint32* members = gems_group_get_group_members();
	gint i = 0;
	printf("Peers:\n");
	for(i = 0; i < (GROUP_MAX_SIZE - 1); i++)	printf("(%d)\t%u\n",i,members[i]);
}

gint gems_communication_already_connected(int checksum)
{
	gems_components* data = gems_get_data();
	
	if(data == NULL)
	{
		gchar* logmsg = g_strdup_printf("gems_communication_already_connected: data == NULL");
		cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
		g_free(logmsg);
		return NOT_IN_LIST;	
	}

	//printf("gems_communication_already_connected: going through lists to search %d", checksum);
	
	if(gems_communication_search_from_list(data->initialized_connections, checksum) == TRUE) return IN_INITIALIZED_LIST;
	if(gems_communication_search_from_list(data->service_jammo->connections, checksum) == TRUE) return IN_JAMMO_SERVICE_LIST;
	if(gems_communication_search_from_list(data->communication->connecting, checksum) == TRUE) return IN_CONNECTING_LIST;
	if(gems_communication_search_from_list(data->communication->connections, checksum) == TRUE) return IN_CONNECTED_LIST;
	if(gems_communication_search_from_list(data->communication->rejected, checksum) == TRUE) return IN_REJECTED_LIST;
	
	//printf("gems_communication_already_connected: No match %d", checksum);
	return NOT_IN_LIST;
}

gboolean gems_communication_search_from_list(GList* list, int checksum)
{
	GList* iterator = NULL;
	if(g_list_length(list) == 0) return FALSE;
	// Search the list
	for(iterator = g_list_first(list); iterator; iterator = g_list_next(iterator))
	{
		gems_connection* element = (gems_connection*)iterator->data;
		
		if(element == NULL) continue;
		
		// Match - return TRUE
		if(ph_c_connection_get_device_checksum(element->connection) == checksum )
		{
			//printf("gems_communication_search_from_list: match found for %d!", checksum);
			return TRUE;
		}
	}
	return FALSE;
}

gems_connection* gems_communication_get_connection(GList* list, int checksum)
{
	GList* iterator = NULL;
	if(g_list_length(list) == 0) return NULL; // empty
	
	// Search the list
	for(iterator = g_list_first(list); iterator; iterator = g_list_next(iterator))
	{
		gems_connection* element = (gems_connection*)iterator->data;
		
		if(element == NULL) continue;
		
		// Match - return element
		if(ph_c_connection_get_device_checksum(element->connection) == checksum ) return element;
	}
	return NULL; // Not found
}

gems_connection* gems_communication_get_connection_with_userid(GList* list, guint32 userid)
{
	GList* iterator = NULL;
	if(g_list_length(list) == 0) return NULL; // empty
	
	// Search the list
	for(iterator = g_list_first(list); iterator; iterator = g_list_next(iterator))
	{
		gems_connection* element = (gems_connection*)iterator->data;
		
		if(element == NULL) continue;
		
		// Profile is requested
		if(element->profile != NULL )
		{
			// Matching id
			if(element->profile->id == userid) return element;
		}
	}
	return NULL; // Not found
}

void gems_communication_process_new(gems_connection* element)
{
	gems_communication_data* data = gems_get_data()->communication;
	data->connections = g_list_append(data->connections, element);
	gchar* logmsg = g_strdup_printf("gems_communication_process_new: added new");
	cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
	g_free(logmsg);
}

void gems_communication_process_new_rejected(gems_connection* element)
{
	gems_communication_data* data = gems_get_data()->communication;
	gems_connection_set_action(element); // Record the time this connection was rejected
	data->rejected = g_list_append(data->rejected, element);
	gchar* logmsg = g_strdup_printf("gems_communication_process_new_rejected: added new element to rejected list");
	cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
	g_free(logmsg);
}

gboolean gems_communication_write_encrypted_data(gint16 type, gems_connection* element, gems_message* data)
{
	gboolean success = TRUE;
	gems_message* envelope = gems_security_create_envelope(type, element, data);
	
	if((gems_communication_write_data(ph_c_connection_get_fd(element->connection), envelope)) != envelope->length)
	{
		success = FALSE;
	}
	
	gems_clear_message(envelope);
	envelope = NULL;
	
	return success;
}

// Write message to socket
gint gems_communication_write_data(gint socket, gems_message* data)
{
	return write(socket, (gchar*)data->message, data->length);
}

// Read data from socket, place data in gpointer* data
gint gems_communication_read_data(gint socket, void* data, gint length)
{
	return read(socket, (gchar*)data, length);
}

void gems_communication_process_connected()
{
	gems_components* data = gems_get_data();
	gboolean cleanup_required = FALSE;
	gems_connection* element = NULL;
	GList* iterator = NULL;
	gchar* logmsg = NULL;
	gint bytesread = -1;
	iterator = g_list_first(data->communication->connections);
	while(iterator)
	{
		element = (gems_connection*)iterator->data;
		
		if(!element)
		{
			iterator = g_list_next(iterator);
			continue;
		}
		
		// TODO implement AA service
		if(element->connection_state == JAMMO_CONNECTED_NO_AUTH) element->connection_state = JAMMO_CONNECTED_AUTH;
		
		/* TODO modify to read the rest of the packet IF there is data available,
		   which might speed up the handling process */
		if((ph_c_connection_is_connected(element->connection)) == TRUE && 
			(ph_c_connection_has_data(element->connection) == TRUE))
		{		
			// Partial data
			if(gems_connection_partial_data_amount(element) > 0)
			{
				// If 2 bytes = Service id/packet type read -> read length
				if(gems_connection_partial_data_amount(element) == sizeof(gint16))
				{
					gint32 packetlength = -1;
					
					switch ((bytesread = gems_communication_read_data(ph_c_connection_get_fd(element->connection), &packetlength, sizeof(gint32))))
					{
						/* CORRECT AMOUNT */
						case sizeof(gint32):
							//TODO check that length is in limits
							//printf("gems_communication_process_list: read length: %d", ntohl(packetlength));
							gems_connection_add_32(element,packetlength); // Add length
							break;
						case 0:
						case -1:
						default:
							logmsg = g_strdup_printf("gems_communication_process_list: length not read correctly (bytesread = %d). Disconnecting", bytesread);
							cem_add_to_log(logmsg, LOG_ERROR);
							g_free(logmsg);
							ph_c_connection_disconnect(element->connection);
							cleanup_required = TRUE;
							break;
					}
				}
				
				// If 6 bytes read = Service id and length read && Not authenticated -> read command
				else if(gems_connection_partial_data_amount(element) == (sizeof(gint16) + sizeof(gint32)) &&
					(element->connection_state == JAMMO_CONNECTED_NO_AUTH))
				{
					gint16 command = -1;
					
					switch((bytesread = gems_communication_read_data(ph_c_connection_get_fd(element->connection), &command, sizeof(gint16))))
					{
						case sizeof(gint16):
							//printf("gems_communication_process_list : command: %d, user in state: %d", ntohs(command), element->connection_state);
							gems_connection_add_16(element,command);
							break;

						case 0:
						case -1:
						default:
							logmsg = g_strdup_printf("gems_communication_process_list : command not read correctly (bytesread = %d). Disconnecting", bytesread);
							cem_add_to_log(logmsg, LOG_ERROR);
							g_free(logmsg);
							
							ph_c_connection_disconnect(element->connection);
							cleanup_required = TRUE;
							break;
					}
				}
				
				// Authenticated, read 6 or more bytes -> read rest
				else if(element->connection_state == JAMMO_CONNECTED_AUTH &&
					(gems_connection_partial_data_amount(element) >= (sizeof(gint16) + sizeof(gint32))))
				{
					// remaining = length - read bytes
					gint remaining = ntohl(gems_connection_get_32(element, sizeof(gint16))) - gems_connection_partial_data_amount(element);
					gchar remainingdata[remaining];
					memset(&remainingdata,'\0', remaining);
					
					// All read?
					if((bytesread = gems_communication_read_data(ph_c_connection_get_fd(element->connection), &remainingdata, remaining)) == remaining)
					{
						//printf("gems_communication_process_list : data fully read, read %d bytes.", bytesread);
						gems_connection_add_data(element,remainingdata,bytesread);
					}
					// Partial read
					else if((bytesread > 0) && (bytesread < remaining))
					{
						//printf("gems_communication_process_list : hash not fully read, read %d bytes.", bytesread);
						gems_connection_add_data(element,remainingdata,bytesread);
						/*printf("gems_communication_process_list : bytes remaining = %d",
							ntohl(gems_connection_get_32(element,sizeof(gint16))) - gems_connection_partial_data_amount(element));*/
					}
					// Errorous read
					else
					{
						logmsg = g_strdup_printf("gems_communication_process_list : error reading rest of data (%d), disconnecting", bytesread);
						cem_add_to_log(logmsg, LOG_ERROR);
						g_free(logmsg);
						
						ph_c_connection_disconnect(element->connection);
						cleanup_required = TRUE;
					}
				}
				
				// All read? Data read = length parameter in buffer[2]
				if(gems_connection_partial_data_amount(element) == ntohl(gems_connection_get_32(element,sizeof(gint16))))
				{
					//printf("gems_communication_process_list: full packet received (%d bytes).", ntohl(gems_connection_get_32(element,sizeof(gint16))));
					if(gems_communication_process_data(element) == FALSE)
					{
						ph_c_connection_disconnect(element->connection);
						cleanup_required = TRUE;
					}
				}
				
				if(gems_connection_partial_data_amount(element) > ntohl(gems_connection_get_32(element,sizeof(gint16))))
				{					
					logmsg = g_strdup_printf("gems_communication_process_list: more than full packet! ERROR?");
					cem_add_to_log(logmsg, LOG_ERROR);
					g_free(logmsg);
					
					ph_c_connection_disconnect(element->connection);
					cleanup_required = TRUE;
				}
			}
			
			// No partial data - read first 16 bits
			else
			{
				gint16 identifier = -1;
			
				switch((bytesread = gems_communication_read_data(ph_c_connection_get_fd(element->connection), &identifier, sizeof(gint16))))
				{
					/* CORRECT AMOUNT */
					case sizeof(gint16):
						// Authenticated - expect a encrypted envelope
						if(element->connection_state == JAMMO_CONNECTED_AUTH)
						{
							switch(ntohs(identifier))
							{
								// "Private" message for all other than group work
								case JAMMO_PACKET_PRIVATE:
									gems_connection_add_16(element,identifier);
									break;
								// Group/pair work message
								case JAMMO_PACKET_GROUP:
									gems_connection_add_16(element,identifier);
									break;
								//case JAMMO_PACKET_ADMIN:
								case ERROR_MSG:
									gems_connection_add_16(element,identifier);
									break;
								default:
									// TODO handle errorous packet type
									// After a device is authenticated everything should be sent as encrypted data inside envelope
									break;
							}
						}
						else if(element->connection_state == JAMMO_CONNECTED_NO_AUTH)
						{
							if(ntohs(identifier) != AA_SERVICE_ID)
							{
								logmsg = g_strdup_printf("gems_communication_process_list: invalid service id \"%d\" user in state %d",
									ntohs(identifier), element->connection_state);
								cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
								g_free(logmsg);
								
								gems_message* err = gems_create_error_message(ERROR_INVALID_SERVICE_ID_TYPE, AA_SERVICE_ID);
								if(gems_communication_write_data(ph_c_connection_get_fd(element->connection),err) != err->length)
								{
									logmsg = g_strdup_printf("gems_service_jammo_process_connections: cannot write ERROR_INVALID_SERVICE_ID");
									cem_add_to_log(logmsg, LOG_ERROR);
									g_free(logmsg);
								}
								gems_clear_message(err);
								ph_c_connection_disconnect(element->connection);
								cleanup_required = TRUE;
							}
							// TODO check that AA service is running
						}
						break;
					
					case 0:
					case -1:
					default:
						logmsg = g_strdup_printf("gems_communication_process_list: service id read error(%d)", bytesread);
						cem_add_to_log(logmsg, LOG_ERROR);
						g_free(logmsg);
						ph_c_connection_disconnect(element->connection);
						cleanup_required = TRUE;
						break;
				}
			}
		}
		
		iterator = g_list_next(iterator);
	}
	if(cleanup_required == TRUE) data->communication->connections = gems_cleanup_connections_from_list(data->communication->connections);
}

gboolean gems_communication_process_data(gems_connection* element)
{
	gems_components* data = gems_get_data();
	gboolean success = TRUE;
	gems_message* err = NULL;
	gchar* logmsg = NULL;
	
	if(data == NULL) return FALSE;
	if(data->service_profile == NULL) return FALSE;
	if(element == NULL) return FALSE;
	
	// If not for AA_SERVICE_ID (state == JAMMO_CONNECTED_NO_AUTH) or ERROR, message is in envelope
	if(element->connection_state == JAMMO_CONNECTED_AUTH)
	{
		switch(ntohs(gems_connection_get_16(element,0)))
		{
			case ERROR_MSG:
				break;
			case JAMMO_PACKET_PRIVATE:
				/*printf("gems_communication_process_data: buffer contains: [%d|%d|<-hash->|%d|%d|%d]",
					ntohs(gems_connection_get_16(element,0)),
					ntohl(gems_connection_get_32(element,2)),
					ntohs(gems_connection_get_16(element,6+JAMMO_MESSAGE_DIGEST_SIZE+1)),
					ntohl(gems_connection_get_32(element,6+JAMMO_MESSAGE_DIGEST_SIZE+1+2)),
					ntohs(gems_connection_get_16(element,6+JAMMO_MESSAGE_DIGEST_SIZE+1+6)));*/
		
				// TODO react to security rval
				switch(gems_security_extract_envelope(element))
				{
					case SECURITY_OK:
						break;
					case SECURITY_VERIFY_FAILED:
						break;
					case SECURITY_CORRUPT_MESSAGE:
						break;
					default:
						break;
				}
			// Decrypt content
			// Verify hash
			// Check content service id
			// Check connection state?
				break;
			case JAMMO_PACKET_GROUP:
				break;
			case JAMMO_PACKET_ADMIN:
				break;
			default:
				logmsg = g_strdup_printf("gems_communication_process_data: unknown envelope id (%d) from authenticated user",ntohs(gems_connection_get_16(element,0)));
				cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
				g_free(logmsg);
				// TODO create error message and send
				break;
		}
	}
	
	gems_connection_localize_data_in_buffer(element);
	
	logmsg = g_strdup_printf("gems_communication_process_data: element contains: [%d|%d|%d]",
		gems_connection_get_16(element,0),
		gems_connection_get_32(element,2),
		gems_connection_get_16(element,6));
	cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
	g_free(logmsg);
	
	switch(gems_connection_get_16(element,0))
	{
		case AA_SERVICE_ID:
			// Pass element to authentication for key exchange
			logmsg = g_strdup_printf("gems_communication_process_data: AA");
			cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
			g_free(logmsg);
			
			err = gems_create_error_message(ERROR_SERVICE_NOT_RUNNING_TYPE, AA_SERVICE_ID);
			if(gems_communication_write_data(ph_c_connection_get_fd(element->connection),err) != err->length)
			{
				logmsg = g_strdup_printf("gems_service_jammo_process_connections: cannot write ERROR_SERVICE_NOT_RUNNING");
				cem_add_to_log(logmsg, LOG_ERROR);
				g_free(logmsg);
			}
			gems_clear_message(err);
			success = FALSE; // TODO change to add a "invalid message" to errors
			break;
		case PROFILE_SERVICE_ID:
			// Check that service is running
			if(data->service_profile->enabled == TRUE) success = gems_service_profile_process_request(element);
			else
			{
				err = gems_create_error_message(ERROR_SERVICE_NOT_RUNNING_TYPE, PROFILE_SERVICE_ID);
				if(gems_communication_write_data(ph_c_connection_get_fd(element->connection),err) != err->length)
				{
					logmsg = g_strdup_printf("gems_service_jammo_process_connections: cannot write ERROR_SERVICE_NOT_RUNNING");
					cem_add_to_log(logmsg, LOG_ERROR);
					g_free(logmsg);
				}
				gems_clear_message(err);
				success = FALSE; // TODO change to add a "invalid message" to errors
			}
			break;
		case GROUP_SERVICE_ID:
			// check that profile is requested!
			// TODO create a better system - request profile directly
			if(element->profile == NULL) break;
			// Check that service is running
			if(data->service_group->enabled == TRUE) success = gems_service_group_process_request(element);
			else
			{
				err = gems_create_error_message(ERROR_SERVICE_NOT_RUNNING_TYPE, GROUP_SERVICE_ID);
				if(gems_communication_write_data(ph_c_connection_get_fd(element->connection),err) != err->length)
				{
					logmsg = g_strdup_printf("gems_service_jammo_process_connections: cannot write ERROR_SERVICE_NOT_RUNNING");
					cem_add_to_log(logmsg, LOG_ERROR);
					g_free(logmsg);
				}
				gems_clear_message(err);
				success = FALSE; // TODO change to add a "invalid message" to errors
			}
			break;
		case COLLABORATION_SERVICE_ID:
			// Check that service is running
			if(data->service_collaboration->enabled == TRUE) success = gems_service_collaboration_process_request(element);
			else
			{
				err = gems_create_error_message(ERROR_SERVICE_NOT_RUNNING_TYPE, COLLABORATION_SERVICE_ID);
				if(gems_communication_write_data(ph_c_connection_get_fd(element->connection),err) != err->length)
				{
					logmsg = g_strdup_printf("gems_service_jammo_process_connections: cannot write ERROR_SERVICE_NOT_RUNNING");
					cem_add_to_log(logmsg, LOG_ERROR);
					g_free(logmsg);
				}
				gems_clear_message(err);
				success = FALSE; // TODO change to add a "invalid message" to errors
			}
			break;
			
			break;
		/*case SONGBANK_SERVICE_ID:
			break;
		case DATATRANSFER_SERVICE_ID:
			break;
		case MENTOR_SERVICE_ID:
			break;
		case CONTROL_SERVICE_ID:
			break;	
		case PROBE_SERVICE_ID:
			break;*/
		case ERROR_MSG:
			logmsg = g_strdup_printf("gems_communication_process_data: error message, type = %d, message: %s",
				gems_connection_get_16(element,8), gems_connection_get_char(element,10));
			cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
			g_free(logmsg);
			gems_communication_process_error(element);
			break;
		default:
			logmsg = g_strdup_printf("gems_communication_process_data: invalid service id \"%d\" user in state %d",
				gems_connection_get_16(element,0), element->connection_state);
			cem_add_to_log(logmsg, LOG_NETWORK_DEBUG);
			g_free(logmsg);
			
			err = gems_create_error_message(ERROR_INVALID_SERVICE_ID_TYPE, JAMMO_SERVICE_ID);
			if(gems_communication_write_data(ph_c_connection_get_fd(element->connection),err) != err->length)
			{
				logmsg = g_strdup_printf("gems_service_jammo_process_connections: cannot write ERROR_INVALID_SERVICE_ID");
				cem_add_to_log(logmsg, LOG_ERROR);
				g_free(logmsg);
			}
			gems_clear_message(err);
			//success = FALSE;
			break;
	}

	// Clear buffer
	gems_connection_clear_buffer(element);
	
	return success;
}

// TODO implement error checking in different states
void gems_communication_process_error(gems_connection* element)
{
	gems_message* msg = NULL;
	gchar* logmsg = NULL;
	gems_connection* owner_elem = NULL;
	
	switch(element->connection_state)
	{
		case JAMMO_CONNECTED_AUTH:
			// Check the sending service
			switch(gems_connection_get_16(element,6))
			{
				case PROFILE_SERVICE_ID:
					// Check the error type
					switch(gems_connection_get_16(element,8))
					{
						case ERROR_SERVICE_NOT_RUNNING_TYPE:
							break;
						case ERROR_INVALID_ACCESS_TYPE:
							break;
						default:
							break;
					}
					break;
					
				case GROUP_SERVICE_ID:
					switch(gems_connection_get_16(element,8))
					{
						case ERROR_SERVICE_NOT_RUNNING_TYPE:
							break;
						case ERROR_NOT_IN_GROUP_TYPE:
							/* if this is sent by a member in the active group the group owner should 
							synchronize group information with all members ! -> send MEMBER_LIST  */
							break;
						case ERROR_GROUP_NOT_ACTIVE_TYPE:
							/* Add tag for gems_connection* to indicate this kind of response,
							-> increase the interval between group requests and remove the tag
							when response or offer is received */
							
							/* If trying to join or leave check that this is sent by group owner,
							if not send the previous request to owner - if owner sent this, the
							group info might be corrupt -> reset */
							
							// Sender is in our group
							if(gems_group_is_in_group(element->profile->id))
							{
								switch(gems_get_data()->service_group->group_info->own_state)
								{
									case JOINING:
										// Fallthrough to next case
									case LEAVING:
										// Sent by owner?
										if(element->profile->id == gems_group_get_owner(NULL))
										{
											logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_GROUP_NOT_ACTIVE, sent by owner, group is being reset.");
											cem_add_to_log(logmsg,LOG_NETWORK_DEBUG);
											g_free(logmsg);
										
											// TODO notify chum through callback
											
											gems_reset_group_info();
										}
										// Sent by some other
										else
										{
											owner_elem = gems_communication_get_connection_with_userid(gems_get_data()->communication->connections,gems_group_get_owner(NULL));
										
											if(owner_elem != NULL)
											{
												logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_GROUP_NOT_ACTIVE, LAST ACTION NOT IMPLEMENTED, nothing done.");
												cem_add_to_log(logmsg,LOG_NETWORK_DEBUG);
												g_free(logmsg);
											}
											else
											{
												logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_GROUP_NOT_ACTIVE, no connection to group owner");
												cem_add_to_log(logmsg,LOG_ERROR);
												g_free(logmsg);
											}
										}
										break;
									
									case IN_GROUP:
										// sent by owner?
										if(element->profile->id == gems_group_get_owner(NULL))
										{
											// TODO notify chum through callback
											
											gems_reset_group_info();
										}
										// Otherwise ignore
										break;
									
									case NOT_IN_GROUP:
										// IGNORE
										break;
									
									default:
										break;
								}
							}
							else
							{
								// TODO increase interval for requests to this connection
							}
							break;
							
						case ERROR_INVALID_GROUP_ID_TYPE:
							/* Try to request group info from group owner */
							break;
							
						case ERROR_ALREADY_IN_GROUP_TYPE:
							/* Request member list from group owner */
							
							// Are we in a active group and the sender is in our group
							if(gems_group_active() && gems_group_is_in_group(element->profile->id))
							{
								owner_elem = element;
								
								// Msg is not from owner
								if(owner_elem->profile->id != gems_group_get_owner(NULL)) 
								{
									// Get the group owner
									owner_elem = gems_communication_get_connection_with_userid(gems_get_data()->communication->connections, gems_group_get_owner(NULL));
									
									if(owner_elem == NULL) // Not found
									{
										logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_ALREADY_IN_GROUP, no connection to group owner");
										cem_add_to_log(logmsg,LOG_ERROR);
										g_free(logmsg);
										break;
									}
									else
									{
										// Create request
										msg = gems_create_message_group_management_notify(REQUEST_MEMBERS, 0, 0);
							
										// Send request
										if(gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE, owner_elem, msg) == FALSE)
										{
											logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_ALREADY_IN_GROUP, cannot write REQUEST_MEMBERS to group owner");
											cem_add_to_log(logmsg,LOG_ERROR);
											g_free(logmsg);
										}
						
										gems_clear_message(msg);
									}
								}	
							}
							else
							{
								logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_ALREADY_IN_GROUP (group %s), got from %u - %s group", 
									gems_group_active() ? "active" : "not active", element->profile->id, gems_group_is_in_group(element->profile->id) ? "in" : "not in");
								cem_add_to_log(logmsg,LOG_NETWORK_DEBUG);
								g_free(logmsg);
							}
							break;
							
						case ERROR_GROUP_FULL_TYPE:
							// From owner
							if(element->profile->id == gems_group_get_owner(NULL))
							{
								/* Remove self from the group and reset the group info */
								if(gems_get_data()->service_group->group_info->own_state == JOINING) gems_reset_group_info();
								// TODO notify chum through callback
							}
							// From other
							else
							{
								logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_GROUP_FULL, from other than owner (uid: %u)",element->profile->id);
								cem_add_to_log(logmsg,LOG_ERROR);
								g_free(logmsg);
							}
							break;
						
						/* TODO implement after last action recording is made -> resend to correct owner */
						case ERROR_NOT_GROUP_OWNER_TYPE:
							// In our group
							if(gems_group_is_in_group(element->profile->id))
							{
								// Not from owner
								if(element->profile->id != gems_group_get_owner(NULL))
								{
									owner_elem = gems_communication_get_connection_with_userid(gems_get_data()->communication->connections,gems_group_get_owner(NULL));
							
									if(owner_elem == NULL)
									{
										logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_NOT_GROUP_OWNER, no connection to group owner");
										cem_add_to_log(logmsg,LOG_NETWORK);
										g_free(logmsg);
									}
									else
									{
										logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_NOT_GROUP_OWNER, LAST ACTION NOT YET IMPLEMENTED, doing nothing.");
										cem_add_to_log(logmsg,LOG_NETWORK_DEBUG);
										g_free(logmsg);
									}
								}
								// This shouldn't happen
								else
								{
									logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_NOT_GROUP_OWNER returned by owner!");
									cem_add_to_log(logmsg,LOG_NETWORK_DEBUG);
									g_free(logmsg);
								}
							}
							break;
							
						case ERROR_GROUP_LOCKED_TYPE:
							// TODO add last actions to check that last message was join request
							if(element->profile->id == gems_group_get_owner(NULL))
							{
								if(gems_get_data()->service_group->group_info->own_state == JOINING) gems_reset_group_info();
								// TODO notify chum through callback
							}
							else
							{
								logmsg = g_strdup_printf("gems_communication_process_error: processing ERROR_GROUP_LOCKED returned by other than owner! (uid: %u)", element->profile->id);
								cem_add_to_log(logmsg,LOG_ERROR);
								g_free(logmsg);
							}
							break;
						default:
							break;
					}
					break;
					
				//case COLLABORATION_SERVICE_ID:
				//	break;
				//case SONGBANK_SERVICE_ID:
				//	break;
				//case DATATRANSFER_SERVICE_ID:
				//	break;
				//case MENTOR_SERVICE_ID:
				//	break;
				//case CONTROL_SERVICE_ID:
				//	break;	
				//case PROBE_SERVICE_ID:
				//	break;
				// Invalid sender
				default:
					break;
			}
			break;
		case JAMMO_CONNECTED_NO_AUTH:
			switch(gems_connection_get_16(element,6)) // Get sending service id
			{
				case JAMMO_SERVICE_ID:
					/* If tried to connect to some other than JamMo service, the ERROR_INVALID_SERVICE_ID
					   will be sent by JAMMO_SERVICE_ID only */
					switch(gems_connection_get_16(element,8)) // Get error
					{
						case ERROR_INVALID_SERVICE_ID_TYPE:
							break;
						default:
							break;
					}
					break;
				case AA_SERVICE_ID:
					/* Authentication errors */
					// invalid service id type
					switch(gems_connection_get_16(element,8)) // Get error
					{
						case ERROR_SERVICE_NOT_RUNNING_TYPE:
							break;
						default:
							break;
					}
					break;
				/* Other services should not send errors to user who is not authenticated.
				   If user tries to connect to service with 
				*/
				default:
					break;
			}
			break;
		default:
			break;
	}
}


gint gems_communication_probe_for_peerhood()
{
	gems_components* data = gems_get_data();
	
	if(data == NULL) return FALSE;
	
	gchar* logmsg = g_strdup_printf("gems_networking_probe_for_peerhood");
	cem_add_to_log(logmsg,LOG_INFO);
	g_free(logmsg);
	
	// Initialize callback if it was not set (PeerHood not found at start)
	if(data->ph_cb == NULL) gems_init_peerhood_callback();
	
	// Get PeerHood instance if not set
	if(data->ph == NULL) data->ph = ph_c_get_instance(data->ph_cb);
	
	// Cannot get instance - should not happen
	if(data->ph == NULL)
	{
		logmsg = g_strdup_printf("gems_networking_probe_for_peerhood: Got PeerHood NULL reference.");
		cem_add_to_log(logmsg,LOG_ERROR);
		g_free(logmsg);
		return TRUE; // Got NULL reference, try again
	}
	
	gchar** args = (gchar**)g_malloc(sizeof(gchar**));
	args[0] = "./jammo";
	if(ph_c_init(data->ph,1,args) == TRUE)
	{
		gems_register_services();
		gems_enable_network_timeout_functions();
		logmsg = g_strdup_printf("gems_networking_probe_for_peerhood: initialized and re-enabled timeout functions.");
		cem_add_to_log(logmsg,LOG_INFO);
		g_free(logmsg);
		g_free(args);
		return FALSE; // Got connection, remove
	}
	else
	{
		logmsg = g_strdup_printf("gems_networking_probe_for_peerhood: cannot initialize peerhood.");
		cem_add_to_log(logmsg,LOG_INFO);
		g_free(logmsg);
		g_free(args);
		return TRUE;
	}
}


void gems_communication_peerhood_close_detected()
{
	gchar* logmsg = NULL;
	gems_components* data = gems_get_data();
	
	if(data == NULL) return;
	
	logmsg = g_strdup_printf("gems_networking_peerhood_close_detected.");
	cem_add_to_log(logmsg,LOG_INFO);
	g_free(logmsg);
	
	if(data->ph == NULL)
	{
		logmsg = g_strdup_printf("gems_networking_peerhood_close_detected: PeerHood NULL.");
		cem_add_to_log(logmsg,LOG_NETWORK_DEBUG);
		g_free(logmsg);
	}
	else
	{
		ph_c_delete_peerhood(data->ph);
		data->ph = NULL;
		
		//ph_c_delete_callback(data->ph_cb);
		//data->ph_cb = NULL;
	}
	
	data->service_jammo->enabled = FALSE;
	data->service_profile->enabled = FALSE;
	data->service_group->enabled = FALSE;
	data->service_collaboration->enabled = FALSE;
		
	
	/*jammo_gems_cleanup_lists();
	printf("jammo_gems_cleanup_lists\n");

	gems_communication_cleanup_lists();
	printf("gems_networking_cleanup_lists\n");*/
	
	// TODO create and user cleanup_lists() of each service
	/*gems_service_jammo_cleanup_lists();
	gems_service_profile_cleanup_lists();
	gems_service_group_cleanup_lists();
	gems_service_collaboration_cleanup_lists();
	
	*/
	/*
	g_timeout_add_full(G_PRIORITY_LOW,2000,(GSourceFunc)gems_communication_probe_for_peerhood, NULL, NULL);
	logmsg = g_strdup_printf("gems_networking_peerhood_close_detected: deleted, timeout for probe added.");
	cem_add_to_log(logmsg,LOG_INFO);
	g_free(logmsg);*/
	
}


