/**sequencer_track_view.c is part of JamMo.
License: GPLv2, read more from COPYING

First call this: sequencer_sequencer_create_sequencer_container(JammoSequencer*)
And then sequencer_track_view_add_with_type(sequencerTrackViewTYpe)
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>
#include <tangle.h>


#include "../../meam/jammo-track.h"
#include "../../meam/jammo-editing-track.h"
#include "../../meam/jammo-instrument-track.h"
#include "../../meam/jammo-midi.h"


#include "../jammo-track-view.h"
#include "../jammo-miditrack-view.h"
#include "../jammo.h"
#include "../jammo-cursor.h"


#include "sequencer_track_view.h"
#include "sequencer.h"
#include "midi_editor.h"
#include "instrument_gui.h"

//Some default values for track_views.
//These works now e.g with animal/140/backing_track
static int number_of_slots = 32;
static guint64 one_second = 1000000000L;
static int beats_per_minute = 140;
static int tempo_factory = 4;

static int width_of_slot = 40;



static SequencerTrackView *track_views[6] = {NULL};
static gint tracks_in_use = 0;

static ClutterActor* track_container = NULL; //contains all tracks (and nothing more) and will be in scroller_actor
static ClutterActor *sequencer_track_area;   //all tracks and buttons and everything

//All tracks share these
static ClutterActor *TEXTURE_VOLUME_BACKGROUND;
static ClutterActor *BUTTON_VOLUME_SLIDER;

static JammoSequencer* sequencer; // for playing the tracks;

static gfloat VOLUME_MAX = 100; //? FIX

static gint BUTTON_SEQUENCER_VOLUME_ACTIVE = -1; //which volume button is pressed down

static gboolean volume_slider_is_dragging = FALSE;
static gboolean volume_background_is_dragging = FALSE;

static gboolean METRONOME_STATE = FALSE;

static void get_volume_background_coords(gfloat *position_x, gfloat *position_y);
static gboolean pressed_sequencer_button_mute(ClutterActor *actor, ClutterEvent *event, gpointer data);
static ClutterAnimation * animate_buttonpress_release(ClutterActor *actor);
static gboolean pressed_sequencer_button_volume(ClutterActor *actor, ClutterEvent *event, gpointer data);
static gboolean pressed_sequencer_button_track_label(ClutterActor *actor, ClutterEvent *event, gpointer data);
static void move_volume_slider(ClutterEvent *event);

static SequencerTrackView *get_active_volume_button_track_view() {
	int i;
	for(i = 0; i < tracks_in_use; i++) {
		if(track_views[i]->volume_pressed_down == TRUE)
			return track_views[i];
	}
	return NULL;
}

static void pressed_sequencer_volume_background (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	volume_background_is_dragging = TRUE;
	clutter_grab_pointer(CLUTTER_ACTOR(actor));

	move_volume_slider(event);
}

static void pressed_sequencer_button_volume_slider (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	volume_slider_is_dragging = TRUE;

	clutter_grab_pointer(CLUTTER_ACTOR(actor));

	move_volume_slider(event);

}

static void released_sequencer_button_volume_slider (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	volume_slider_is_dragging = FALSE;
	clutter_ungrab_pointer();

}

static void released_sequencer_volume_background (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	volume_background_is_dragging = FALSE;
	clutter_ungrab_pointer();

}

static void motion_sequencer_volume_background (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	if (volume_background_is_dragging) {

		move_volume_slider(event);

	}

}

static void motion_sequencer_button_volume_slider (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	if (volume_slider_is_dragging) {

		move_volume_slider(event);

	}

}


/*
 * Creates and returns SequencerTrackView struct
 */
static SequencerTrackView *sequencer_track_view_new(ClutterActor *label,
    ClutterActor *track_view, char *track_view_type, void* track) {

	ClutterActor* mute_button = clutter_texture_new_from_file("/opt/jammo/sequencer/button_mute.png", NULL);
	ClutterActor* volume_button = clutter_texture_new_from_file("/opt/jammo/sequencer/button_volume.png", NULL);

	SequencerTrackView *view = malloc(sizeof(SequencerTrackView));
	view->sequencer_track_view_container = tangle_widget_new();
	view->label = label;
	view->mute_button = mute_button;
	view->volume_button = volume_button;
	view->track_view = track_view;
	view->position = -1;
	view->muted = FALSE;
	view->volume_pressed_down = FALSE;
	view->volume_level = VOLUME_MAX;
	view->track_type = track_view_type;

	tangle_widget_add(TANGLE_WIDGET(view->sequencer_track_view_container), view->label,NULL);
	tangle_widget_add(TANGLE_WIDGET(view->sequencer_track_view_container), view->mute_button,NULL);
	tangle_widget_add(TANGLE_WIDGET(view->sequencer_track_view_container), view->volume_button,NULL);

	gfloat x=0;
	clutter_actor_set_position(view->label, x, 0.0);
	x += 2 +  clutter_actor_get_width(view->label);

	clutter_actor_set_position(view->mute_button, x, 0.0);
	x += 2.0 + clutter_actor_get_width(view->mute_button);

	clutter_actor_set_position(view->volume_button, x, 0.0);

	g_signal_connect(CLUTTER_ACTOR(view->label), "button_press_event", G_CALLBACK(pressed_sequencer_button_track_label), view);
	g_signal_connect(CLUTTER_ACTOR(view->mute_button), "button_press_event", G_CALLBACK(pressed_sequencer_button_mute), view);
	g_signal_connect(CLUTTER_ACTOR(view->volume_button), "button_press_event", G_CALLBACK(pressed_sequencer_button_volume), view);



	clutter_actor_set_reactive(view->label, TRUE);
	clutter_actor_set_reactive(view->mute_button, TRUE);
	clutter_actor_set_reactive(view->volume_button, TRUE);

	return view;
}

/* Adds STV into array & container and positions it.
 */
static void sequencer_track_view_add(SequencerTrackView *track_view) {
	gfloat y_pos = 49.0;

	track_views[tracks_in_use] = track_view;
	tracks_in_use++;
	clutter_container_add_actor(CLUTTER_CONTAINER(sequencer_track_area), track_view->sequencer_track_view_container);

	clutter_container_add_actor(CLUTTER_CONTAINER(track_container), track_view->track_view);
	clutter_actor_set_x(track_view->track_view, clutter_actor_get_x(track_view->track_view)+1);  //unless left border is not visible
	clutter_actor_set_y(track_view->track_view, y_pos * (tracks_in_use - 1)+2);

	clutter_actor_set_y(track_view->sequencer_track_view_container, y_pos * tracks_in_use);
	track_view->position = tracks_in_use - 1;
	printf("Sequencer Track View placed in Position y: %f\n",
         clutter_actor_get_y(track_view->sequencer_track_view_container));
}



/*
 * Adds given type of track view into sequencer view
 * Returns track_view_actor (jammo_track_view) of newly created sequencer_track_view
 * Returns NULL is askes invalid type.
 */
ClutterActor* sequencer_track_view_add_with_type(gchar* track_view_type, guint64 duration) {
	char *twt = track_view_type;
	printf("Making '%s' track (duration: %" G_GUINT64_FORMAT ")\n",track_view_type, duration);

	gfloat track_height = 50.0; //Maybe parameter?

	ClutterActor *label = clutter_texture_new();
	guint64 temp=60*tempo_factory;
	guint64 big = temp * one_second;
	guint64 duration_of_one_slot = big / beats_per_minute;

	JammoTrack* track;

	ClutterActor* track_view_actor = NULL; //This will be returned
	SequencerTrackView *sequencer_track_view;

	//Choose label-image:
	if (strncmp(track_view_type,"DRUM_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_drum.png", NULL);

	else if (strncmp(track_view_type,"SLIDER_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_slider.png", NULL);

	else if (strncmp(track_view_type,"AUDIO_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_audio.png", NULL);

	else if (strncmp(track_view_type,"LOOP_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_loop.png", NULL);

	else if (strncmp(track_view_type,"BACKING_TRACK_VIEW",7)==0)
		label = clutter_texture_new_from_file("/opt/jammo/sequencer/label_backingtrack.png", NULL);

	else if (strncmp(track_view_type,"KEYBOARD_TRACK_VIEW",7)==0)
		label= clutter_texture_new_from_file("/opt/jammo/sequencer/label_keyboard.png", NULL);


	else  { //Error-case
		printf("Not detected track-view type\n");
		return NULL;
	}


	//Make instrument_track and miditrack_view
	if (strncmp(track_view_type,"KEYBOARD_TRACK_VIEW",19)==0)
		{
		track = JAMMO_TRACK(jammo_instrument_track_new(JAMMO_INSTRUMENT_TYPE_UD));
		jammo_instrument_track_set_realtime(JAMMO_INSTRUMENT_TRACK(track),FALSE);

		//TODO: we need to know highest_note and lowest_note for this instrument!
		track_view_actor = jammo_miditrack_view_new(JAMMO_INSTRUMENT_TRACK(track),
          number_of_slots, duration_of_one_slot, width_of_slot, track_height,60,24); //height can be anything, we zoom it
		jammo_miditrack_view_zoom_to_fit(JAMMO_MIDITRACK_VIEW(track_view_actor), number_of_slots*width_of_slot,track_height);
		jammo_miditrack_view_set_show_grid(JAMMO_MIDITRACK_VIEW(track_view_actor),FALSE);
		}


	//Make editing_track and track_view
	else
	{
		track = JAMMO_TRACK(jammo_editing_track_new_fixed_duration(duration));

		track_view_actor = jammo_track_view_new(JAMMO_EDITING_TRACK(track),
          number_of_slots, duration_of_one_slot, width_of_slot, track_height);
		g_object_set(track_view_actor, "line-every-nth-slot", 1, NULL);
	}

	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(track));
	sequencer_track_view = sequencer_track_view_new(label, track_view_actor, twt, track);
	sequencer_track_view_add(sequencer_track_view);
	return track_view_actor;
}

static ClutterAnimation * animate_buttonpress_release(ClutterActor *actor) {
	ClutterAnimation *anim = clutter_actor_animate (actor,          /* the actor to animate */
			CLUTTER_LINEAR, /* the easing mode */
			100,           /* the duration of the animation */
			"scale-x", 1.0, /* final horizontal scaling factor */
			"scale-y", 1.0, /* final vertical scaling factor */
			"scale-gravity",CLUTTER_GRAVITY_CENTER,
			NULL);
	return anim;
}

static ClutterAnimation * animate_buttonpress_down(ClutterActor *actor) {
	ClutterAnimation *anim = clutter_actor_animate (actor,          /* the actor to animate */
			CLUTTER_LINEAR, /* the easing mode */
			100,           /* the duration of the animation */
			"scale-x", 0.80, /* final horizontal scaling factor */
			"scale-y", 0.80, /* final vertical scaling factor */
			"scale-gravity",CLUTTER_GRAVITY_CENTER,
			NULL);
	return anim;
}


/*
 * Event for pressing label on track, data points to track structure
 */
static gboolean pressed_sequencer_button_track_label(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	SequencerTrackView* track = (SequencerTrackView*)data;

	printf("Pressed track label\n");
	if (JAMMO_IS_MIDITRACK_VIEW(track->track_view))
	{
		printf("It is instrument track\n");
		//We can start realtime jamming or midi-editor.
		//Midi-editor backs always to sequencer.
		//From Jamming we can go back to sequencer or reach editor
		//TODO: some way to go straight to editor.

		//jamming
		instrument_gui_start(JAMMO_MIDITRACK_VIEW(track->track_view));

		//editor
		//midi_editor_start_with_miditrack_view(JAMMO_MIDITRACK_VIEW(track->track_view));
	}
	return TRUE;
}



static gboolean pressed_sequencer_button_mute(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	SequencerTrackView *track_view = (SequencerTrackView*) data;
	JammoTrack* jammotrack;
	g_object_get(track_view->track_view, "track", &jammotrack, NULL);
	if (!jammotrack)
	{
		printf("JammoTrack false!\n");
		return FALSE;
	}
	//if mute is off, set it on
	if (!track_view->muted) {
		animate_buttonpress_down(actor);
		jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(jammotrack), TRUE);
		//Shade button
	}

	//if mute is on, set it off
	if (track_view->muted) {
		animate_buttonpress_release(actor);
		jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(jammotrack), FALSE);
	}

	//Change mute state to opposite
	track_view->muted = !track_view->muted;

	return TRUE;
}


static void set_volume_slider_position_by_state(gint state) {

	gfloat position_x_volume_background = 0.0;
	gfloat position_y_volume_background = 0.0;
	get_volume_background_coords(&position_x_volume_background, &position_y_volume_background);

	//get volume background size
	gfloat volume_background_width = 0;
	gfloat volume_background_height = 0;
	clutter_actor_get_size (TEXTURE_VOLUME_BACKGROUND, &volume_background_width, &volume_background_height);

	//get volume slider size
	gfloat volume_slider_width = 0;
	gfloat volume_slider_height = 0;
	clutter_actor_get_size (BUTTON_VOLUME_SLIDER, &volume_slider_width, &volume_slider_height);

	//vertical range of volume background
	gfloat v_range = volume_background_height - volume_slider_height;

	//volume range is 0..VOLUME_MAX
	//calculate slider position
	gfloat position_x_volume_slider = position_x_volume_background + 2;
	gfloat position_y_volume_slider = position_y_volume_background + (v_range - state);

	clutter_actor_set_position (BUTTON_VOLUME_SLIDER, position_x_volume_slider, position_y_volume_slider);

}

static void get_volume_background_coords(gfloat *position_x, gfloat *position_y) {

	gfloat position_x_volume_background = clutter_actor_get_x(track_views[BUTTON_SEQUENCER_VOLUME_ACTIVE]->volume_button);
	gfloat position_y_volume_background = clutter_actor_get_y(track_views[BUTTON_SEQUENCER_VOLUME_ACTIVE]->sequencer_track_view_container);
	//get the width of volume button
	gfloat volume_button_width = 0;
	gfloat volume_button_height = 0;

	clutter_actor_get_size (track_views[BUTTON_SEQUENCER_VOLUME_ACTIVE]->volume_button, &volume_button_width, &volume_button_height);

	position_x_volume_background += volume_button_width + 2;

	*position_x = position_x_volume_background;
	*position_y = position_y_volume_background;

}

static gboolean pressed_sequencer_button_volume(ClutterActor *actor, ClutterEvent *event, gpointer data) {
	SequencerTrackView *track_view = (SequencerTrackView*) data;

	//if selected button was in normal state (not pressed down)
	if (track_view->volume_pressed_down == FALSE) {

		//release already pressed button
		if(BUTTON_SEQUENCER_VOLUME_ACTIVE > -1)
			animate_buttonpress_release(track_views[BUTTON_SEQUENCER_VOLUME_ACTIVE]->volume_button);
		//set state pressed
		track_view->volume_pressed_down = TRUE;
		BUTTON_SEQUENCER_VOLUME_ACTIVE = track_view->position;
		//Push selected volume button down
		//use general animate function
		animate_buttonpress_down(track_view->volume_button);

		//Open volume slider next to selected button
		clutter_actor_show(BUTTON_VOLUME_SLIDER);

		clutter_actor_show(TEXTURE_VOLUME_BACKGROUND);
		clutter_actor_raise_top(TEXTURE_VOLUME_BACKGROUND);
		clutter_actor_raise_top(BUTTON_VOLUME_SLIDER);
		//get selected volume button coords
		gfloat position_x_volume_background = 0.0;
		gfloat position_y_volume_background = 0.0;

		get_volume_background_coords(&position_x_volume_background, &position_y_volume_background);

		//set volume background to new position
		clutter_actor_set_position (TEXTURE_VOLUME_BACKGROUND, position_x_volume_background, position_y_volume_background);

		//set the volume slider in its current position
		set_volume_slider_position_by_state(track_view->volume_level);

	} else {


		//selected volume button was down - set it to normal state
		//use general animate function
		animate_buttonpress_release(track_view->volume_button);

		track_view->volume_pressed_down = FALSE;

		clutter_actor_hide(BUTTON_VOLUME_SLIDER);
		clutter_actor_hide(TEXTURE_VOLUME_BACKGROUND);
	}

	return TRUE;
}

static void move_volume_slider(ClutterEvent *event) {

	SequencerTrackView *track_view = get_active_volume_button_track_view();
	JammoTrack* jammotrack;
	g_object_get(track_view->track_view, "track", &jammotrack, NULL);
	if (!jammotrack)
		return;

	gfloat x_mouse_f,y_mouse_f;
	clutter_event_get_coords (event, &x_mouse_f, &y_mouse_f);

	//get selected volume button coords
	gfloat position_x_volume_background_f = 0.0;
	gfloat position_y_volume_background_f = 0.0;
	get_volume_background_coords(&position_x_volume_background_f, &position_y_volume_background_f);

	gint position_x_volume_background = (gint)position_x_volume_background_f;
	gint position_y_volume_background = (gint)position_y_volume_background_f;

	gint new_x, new_y;

	new_x=position_x_volume_background + 2 ;
	new_y=(gint)y_mouse_f - clutter_actor_get_height(BUTTON_VOLUME_SLIDER) / 2;;

	//Check top and bottom edges
	gint slide_width = clutter_actor_get_height(TEXTURE_VOLUME_BACKGROUND) - clutter_actor_get_height(BUTTON_VOLUME_SLIDER);

	gint top_edge = position_y_volume_background;
	gint bottom_edge = position_y_volume_background + slide_width;

	if (new_y < top_edge) new_y = top_edge;
	if (new_y > bottom_edge) new_y = bottom_edge;

	gint new_volume_state = new_y - position_y_volume_background;
	//update volume state
	get_active_volume_button_track_view()->volume_level = VOLUME_MAX - new_volume_state;

	clutter_actor_set_position (CLUTTER_ACTOR(BUTTON_VOLUME_SLIDER), new_x, new_y);
	jammo_playing_track_set_volume(JAMMO_PLAYING_TRACK(jammotrack), (gfloat)get_active_volume_button_track_view()->volume_level/100);
}



bool save_sequencer_state(gchar* outputFilename) {

	printf("Saving composition\n");
	FILE *ofp;

	ofp = fopen(outputFilename, "w");
	g_free(outputFilename);
	g_return_val_if_fail(ofp != NULL, FALSE);

	GList* children;
	TangleWidget* widget;
	JammoSampleButton *jsb;

	int i;

	for(i = 0; i < tracks_in_use; i++) {
		fprintf(ofp, "Track-type: %s\n", track_views[i]->track_type);
		if(strcmp(track_views[i]->track_type, "KEYBOARD_TRACK_VIEW") == 0)
		{
		//TODO
			continue;
		}
		printf("%s\n", track_views[i]->track_type);

		widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_views[i]->track_view)));

		for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {

			jsb = JAMMO_SAMPLE_BUTTON(children->data);
			//printf("x: '%lf' \n",clutter_actor_get_x(CLUTTER_ACTOR(jsb)));
			printf("slot: '%d' \n", (gint) clutter_actor_get_x(CLUTTER_ACTOR(jsb))/width_of_slot);
			fprintf(ofp,"%d\n", (gint) clutter_actor_get_x(CLUTTER_ACTOR(jsb))/width_of_slot);
			gchar* sound;
			gchar* image;
			g_object_get(jsb, "sample-filename", &sound, NULL);
			g_object_get(jsb, "image-filename", &image, NULL);
			printf("sound: '%s' \n",sound);
			printf("image: '%s'\n", image);
			fprintf(ofp,"%s\n",sound);
			fprintf(ofp,"%s\n",image);
		}
	}

	fclose(ofp);
	return FALSE;
}


static void on_cursor_notify_x(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleVault* vault;
	TangleScrollingActor* scrolling_actor;
	JammoCursor* cursor;
	gfloat x, visible, offset, max;

	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 2, TANGLE_TYPE_SCROLLING_ACTOR, &scrolling_actor, JAMMO_TYPE_CURSOR, &cursor);
	g_object_get(object, "x", &x, NULL);
	g_object_get(scrolling_actor, "width", &visible, "scrolling-offset-x", &offset, "scrolling-width", &max, NULL);
	if (x < offset) {
		offset = x - visible + 80.0;
		if (offset < 0.0) {
			offset = 0.0;
		}
		clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-x", offset, NULL);
	} else if (x > offset + visible - 80.0) {
		offset += 160.0;
		if (offset > max) {
			offset = max;
		}
		clutter_actor_animate(CLUTTER_ACTOR(scrolling_actor), CLUTTER_EASE_IN_OUT_QUAD, 250, "scrolling-offset-x", offset, NULL);
	}
}

/***************************
Functions for json
**************************/



void fullsequencer_play_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Play-Stop-Button pressed\n");

	//tanglebutton is toggle-button.
	gboolean selected;
	g_object_get(tanglebutton, "selected", &selected,  NULL);

	if (selected) {
		jammo_sequencer_play(sequencer);
	}
	else {
		jammo_sequencer_stop(sequencer);
	}
}



//TODO
void fullsequencer_record_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Record pressed\n");
}


//TODO. use metronome-track
//is METRONOME_STATE used?
void fullsequencer_metronome_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Metronome pressed\n");

		//tanglebutton is toggle-button.
	gboolean selected;
	g_object_get(tanglebutton, "selected", &selected,  NULL);

	if (selected) {
		METRONOME_STATE = FALSE;
	}
	else {
		METRONOME_STATE = TRUE;
	}
}

void fullsequencer_trashbin_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Trashbin pressed. Doesn't do anything\n");
}

void fullsequencer_community_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Community pressed. Doesn't do anything\n");
}

void fullsequencer_home_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Home pressed Doesn't do anything\n");

	/*
	jammo_sequencer_stop(sequencer);
	save_sequencer_state();
	leave_sequencer();

	*/
}




ClutterActor *sequencer_track_view_create_area_for_tracks(JammoSequencer* seq) {

	sequencer=seq;

	//This is container we are making and will return
	sequencer_track_area = tangle_widget_new();

	gfloat width_of_area = number_of_slots*width_of_slot;
	//Area for tracks (will be horizontally scrollable)
	track_container = tangle_widget_new();
	clutter_actor_set_size(track_container,width_of_area, 292.0);

	ClutterActor* scrolling_actor;
	scrolling_actor = tangle_scrolling_actor_new(track_container);
	clutter_actor_set_position(scrolling_actor,  194.0, 48.0);
	clutter_actor_set_size(scrolling_actor, 602.0, 295.0);

	tangle_widget_add(TANGLE_WIDGET(sequencer_track_area),scrolling_actor,NULL);


	//Cursor:
	ClutterColor red = { 255, 0, 0, 128 };
	ClutterActor* texture;
	ClutterActor* cursor;

	texture = clutter_rectangle_new_with_color(&red);
	clutter_actor_set_size(texture, 10.0, 292.0 ); //TODO, change height this depending number of tracks
	cursor = jammo_cursor_new(sequencer, texture);
	tangle_actor_set_depth_position(TANGLE_ACTOR(cursor), 1);
	clutter_actor_set_position(cursor, 0.0, 0.0);
	clutter_actor_set_size(cursor, width_of_area, 80.0);
	tangle_widget_add(TANGLE_WIDGET(track_container), cursor,NULL);

	//For automatic scrolling on cursor movement
	TangleVault* vault;

	vault = tangle_vault_new(2, TANGLE_TYPE_SCROLLING_ACTOR, scrolling_actor, JAMMO_TYPE_CURSOR, cursor);
	tangle_signal_connect_vault(texture, "notify::x", G_CALLBACK(on_cursor_notify_x), vault);


	TEXTURE_VOLUME_BACKGROUND = clutter_texture_new_from_file("/opt/jammo/sequencer/volume_background.png", NULL);
	BUTTON_VOLUME_SLIDER =clutter_texture_new_from_file("/opt/jammo/sequencer/volume_button.png", NULL);

	g_signal_connect (TEXTURE_VOLUME_BACKGROUND, "button-press-event", G_CALLBACK (pressed_sequencer_volume_background), NULL);
	g_signal_connect (TEXTURE_VOLUME_BACKGROUND, "button-release-event", G_CALLBACK (released_sequencer_volume_background), NULL);
	g_signal_connect (TEXTURE_VOLUME_BACKGROUND, "motion-event", G_CALLBACK (motion_sequencer_volume_background), NULL);

	g_signal_connect (BUTTON_VOLUME_SLIDER, "button-press-event", G_CALLBACK (pressed_sequencer_button_volume_slider), NULL);
	g_signal_connect (BUTTON_VOLUME_SLIDER, "button-release-event", G_CALLBACK (released_sequencer_button_volume_slider), NULL);
	g_signal_connect (BUTTON_VOLUME_SLIDER, "motion-event", G_CALLBACK (motion_sequencer_button_volume_slider), NULL);


	clutter_actor_set_reactive(TEXTURE_VOLUME_BACKGROUND, TRUE);
	clutter_actor_set_reactive(BUTTON_VOLUME_SLIDER, TRUE);

	tangle_widget_add(TANGLE_WIDGET(sequencer_track_area),TEXTURE_VOLUME_BACKGROUND,NULL);
	tangle_widget_add(TANGLE_WIDGET(sequencer_track_area),BUTTON_VOLUME_SLIDER,NULL);

	clutter_actor_hide(TEXTURE_VOLUME_BACKGROUND);
	clutter_actor_hide(BUTTON_VOLUME_SLIDER);
	return sequencer_track_area;
}
