/**sequencer_loop.c is part of JamMo.
 License: GPLv2, read more from COPYING

 This file is for clutter based gui.
 This is part of the sequencer.
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>
#include <tangle.h>

#include "sequencer_loop.h"
#include "sequencer.h"
#include "../jammo-sample-button.h"
#include "../jammo.h" //get_id

#include "../../cem/cem.h"

#include "../../configure.h"
//Static pointers for loop rolls
static ClutterActor* rhytmical_sample_looper;
static ClutterActor* melodical_sample_looper;
static ClutterActor* harmonical_sample_looper;
static ClutterActor* effect_sample_looper;

gboolean sequencer_lock_pressed (TangleButton* tanglebutton, gpointer data){

	const gchar* name = clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));
	gchar* message = g_strdup_printf("Wheel_game: Lock '%s' pressed",name);
	cem_add_to_log(message,LOG_USER_ACTION);
	g_free(message);

	ClutterActor* scroller = NULL;
	if (strncmp(name,"rhytm",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-rhytmical-wheel-scroller");
	else if (strncmp(name,"melod",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-melodical-wheel-scroller");
	else if (strncmp(name,"harmo",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-harmonical-wheel-scroller");
	else if (strncmp(name,"effec",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-effect-wheel-scroller");

	else {
		cem_add_to_log("can't detect which lock pressed",LOG_WARN);
		return FALSE;
	}

	ClutterAction* scroll_action = tangle_actor_get_action_by_type(scroller,TANGLE_TYPE_SCROLL_ACTION);
	gfloat threshold_x, threshold_y;

	if (scroller)  {
		tangle_scroll_action_get_thresholds(TANGLE_SCROLL_ACTION(scroll_action), &threshold_x,&threshold_y);
		if (threshold_y==10000.0)
			tangle_scroll_action_set_thresholds(TANGLE_SCROLL_ACTION(scroll_action), threshold_x,30.0);//TODO: use original default value
		else
			tangle_scroll_action_set_thresholds(TANGLE_SCROLL_ACTION(scroll_action), threshold_x,10000.0);
	}
	return TRUE;
}


/*
ScrollingActor doesn't have clampping. This will set it.
*/
static void fine_tune_wheel (ClutterActor* sample_looper) {
	ClutterAction* action;
	
	action = tangle_actor_get_action_by_type(sample_looper, TANGLE_TYPE_SCROLL_ACTION);
	g_signal_connect_swapped(action, "clamp-offset-y", G_CALLBACK(tangle_widget_clamp_child_boundaries), sample_looper);
}


#define ABS_F(x) ((x) < 0 ? -(x) : (x))
/*
Returns JammoSample closest to center of wheel. If wheel is rolling and two sample are equal
some of them are returned.
(This function can be used when wheel is running but there are now reason to do that).
Will return NULL in error cases
*/
static JammoSample* get_centermost_sample_from_loop_roll(ClutterActor* wheel) {
	gfloat y;
	gfloat offset=0,not_used; //How much scroller is scrolled
	gfloat height;    //Height of the scroller (we take this from color_box behind samples)

	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);
	tangle_scroll_action_get_offsets(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&offset);

	//Check height of first sample-button, suppose all have same height
	offset+= clutter_actor_get_height(tangle_widget_get_nth_child(TANGLE_WIDGET(wheel),0));

	//Check height of the wheel
	height = clutter_actor_get_height(wheel);
	//printf("height = %f\n",height);

	//Check all sample button which one is closest to center (='offset')
	float best_value = height;
	JammoSampleButton* closest = NULL;

	GList* children;
	for (children = tangle_widget_get_children_readonly(TANGLE_WIDGET(wheel)); children; children = children->next) {
		if (!JAMMO_IS_SAMPLE_BUTTON(children->data)){
			//printf("this is not sample_button\n");  //Color_box triggers this at least
			continue;
		}

		y = clutter_actor_get_y (children->data);
		if (y==0) y+=height; //actor in first slot should be consider be in after very last (+height)

		if (ABS_F(y-offset) < best_value) {
			best_value = ABS_F(y-offset);
			closest = JAMMO_SAMPLE_BUTTON(children->data);
		}

	}
	if (closest==NULL) return NULL;
	return jammo_sample_button_get_sample(closest);
}


/***************************
Functions for json
**************************/


/*
 * plays all samples from centerline of loop rolls
 */
void fullsequencer_loop_arrow_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_play_all_loops-clicked",LOG_USER_ACTION);
	//get_centermost_sample_from_loop_roll can return NULL (in error cases)
	JammoSample* r = get_centermost_sample_from_loop_roll(rhytmical_sample_looper);
	JammoSample* e = get_centermost_sample_from_loop_roll(effect_sample_looper);
	JammoSample* m = get_centermost_sample_from_loop_roll(melodical_sample_looper);
	JammoSample* h = get_centermost_sample_from_loop_roll(harmonical_sample_looper);

	//These play very same time even without adhoc sequencer and tracks.
	if (r!=NULL)   jammo_sample_play(r);
	if (e!=NULL)   jammo_sample_play(e);
	if (m!=NULL)   jammo_sample_play(m);
	if (h!=NULL)   jammo_sample_play(h);
}

/*
amount=how many slots.
If loop is already rolling, this will abort it and will make own.
Loop will still snap to grid always.
*/
static void rotate_loop_for_amount(ClutterActor* wheel, gint amount) {
	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);

	//Check if lock is pressed
	gfloat not_used,threshold_y;
	tangle_scroll_action_get_thresholds(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&threshold_y);

	//printf("threshold_y '%f'\n",threshold_y);
	if (threshold_y==10000.0) //means rolling is disabled for this wheel
		return;
	gfloat offset;
	tangle_scroll_action_get_offsets(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&offset);
	//printf("offset '%f'\n",offset);

	int current_slot = (int)(offset)/92; //round to lower slot, if rolling already
	gfloat target = (current_slot+amount) * 92.0;
	//printf("current_slot = %d, target pixel='%f'\n",current_slot,target);
	tangle_object_animate(G_OBJECT(scroll_action),  CLUTTER_EASE_IN_OUT_QUAD, 550, "offset-y", target, NULL);
}


/*
There are not yet logic for 'Cannot Fail'. It just turn them one slot forward. TODO
*/
void fullsequencer_loop_cannot_fail_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_loop_cannot_fail_clicked-clicked",LOG_USER_ACTION);
	rotate_loop_for_amount(rhytmical_sample_looper, 1);
	rotate_loop_for_amount(melodical_sample_looper, 1);
	rotate_loop_for_amount(harmonical_sample_looper, 1);
	rotate_loop_for_amount(effect_sample_looper, 1);
}


//There will be some logic
void fullsequencer_loop_crazy_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_loop_crazy_clicked-clicked",LOG_USER_ACTION);
	GRand *random_gen = g_rand_new();

	gint rot_rhyt = g_rand_int_range(random_gen, -30, 30);
	gint rot_melo = g_rand_int_range(random_gen, -30, 30);
	gint rot_harm = g_rand_int_range(random_gen, -30, 30);
	gint rot_effe = g_rand_int_range(random_gen, -30, 30);

	rotate_loop_for_amount(rhytmical_sample_looper, rot_rhyt);
	rotate_loop_for_amount(melodical_sample_looper, rot_melo);
	rotate_loop_for_amount(harmonical_sample_looper, rot_harm);
	rotate_loop_for_amount(effect_sample_looper, rot_effe);

	g_rand_free(random_gen);
}


//////////////////////////////////////

//Bottom of the loop-view there are dropping-area, and when samplebutton is dragged here, view is changed.
gboolean sequencer_drag_begin(TangleDropAction* drop_action, TangleDragAction* drag_action){
	cem_add_to_log("jammo_sample_button dragged from wheel to bottom of the screen",LOG_USER_ACTION);
	sequencer_change_to_sequencer_view(NULL,NULL);
	
	return FALSE;
}


static int length_of_rhythms=0;
static int length_of_melodics=0;
static int length_of_harmonics=0;
static int length_of_effects=0;

typedef struct {
	const gchar* image;
	const gchar* sound;
} samplebutton_tuple;

static samplebutton_tuple rhythms[] = {
	{"wheel_game/electricdrums.png","wheel_game/Rh_16beat01_$t_44_1.ogg"},
	{"wheel_game/electricdrums.png","wheel_game/Rh_16beat02_$t_44_1.ogg"},
	{"wheel_game/electricdrums.png","wheel_game/Rh_8beat01_$t_44_1.ogg"},
	{"wheel_game/electricdrums.png","wheel_game/Rh_8beat02_$t_44_1.ogg"},
	{"wheel_game/electricdrums.png","wheel_game/Rh_8beat03_$t_44_1.ogg"},
	{"wheel_game/electricdrums.png","wheel_game/Rh_8beat04_$t_44_1.ogg"},
	{NULL,NULL}
};


static samplebutton_tuple melodics[] = {
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion1_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion2_$t_$p_44_1.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion3_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion4_$t_$p_44_2.ogg"},
	{"wheel_game/brass.png", "wheel_game/Me_SocaBrassRiff_$t_$p_44_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello01_$t_$p_44_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello02_$t_$p_44_1.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello05_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass01_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass02_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass03_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass04_$t_$p_44_1.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp01_$t_$p_34_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp02_$t_$p_44_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp03_$t_$p_44_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp04_$t_$p_44_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp05_$t_$p_34_2.ogg"},
	{"wheel_game/mandolin.png", "wheel_game/Me_mandolin02_$t_$p_34_4.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax01_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax02_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax03_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax04_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax05_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax06_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax07_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax08_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax09_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax10_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax11_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax12_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_ScarboroughFairSax_$t_$p_44_8.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_SiyaHambaSax1_$t_$p_44_4.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe01_$t_$p_44_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe02_$t_$p_44_1.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe03_$t_$p_44_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe04_$t_$p_44_1.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe05_$t_$p_34_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe06_$t_$p_44_1.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe07_$t_$p_34_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe08_$t_$p_34_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_LatinOrgan1_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ3_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ4_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ5_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ6_$t_$p_44_2.ogg"},
	{"wheel_game/vibraphone.png", "wheel_game/Me_vibraphone01_$t_$p_34_2.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone01_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone02_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone03_$t_$p_44_2.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone04_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone05_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone06_$t_$p_44_2.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone07_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone08_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone09_$t_$p_44_1.ogg"},
	{NULL,NULL}
};

static samplebutton_tuple harmonics[] = {
	{"wheel_game/accordion.png", "wheel_game/Ha_accordion01_$t_$p_44_1.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Ha_accordion02_$t_$p_44_1.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Ha_SocaAccordionRiff1_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Ha_SocaAccordionRiff2_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar01_$t_$p_44_1.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar02_$t_$p_44_1.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar02_$t_$p_44_1.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar04_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar05_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar08_$t_$p_34_2.ogg"},
	{"wheel_game/banjo.png", "wheel_game/Ha_banjo01_$t_$p_44_1.ogg"},
	{"wheel_game/banjo.png", "wheel_game/Ha_banjo02_$t_$p_44_1.ogg"},
	{"wheel_game/cello.png", "wheel_game/Ha_cello04_$t_$p_44_1.ogg"},
	{"wheel_game/cello.png", "wheel_game/Ha_cello09_$t_$p_44_1.ogg"},
	{"wheel_game/mandolin.png", "wheel_game/Ha_mandolin01_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ01_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ02_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ07_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ08_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ09_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ10_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ11_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ12_$t_$p_44_2.ogg"},
	{"wheel_game/trumpet.png", "wheel_game/Ha_trumpet01_$t_$p_44_2.ogg"},
	{NULL,NULL}
};


static samplebutton_tuple effects[] = {
	{"wheel_game/fx.png", "wheel_game/Fx_robot_$t_$p_$p_44_2.ogg"},
	{"wheel_game/electric_piano.png", "wheel_game/Fx_synth01_$t_44_1.ogg"},
	{"wheel_game/electric_piano.png", "wheel_game/Fx_synth02_$t_44_1.ogg"},
	{"wheel_game/fx.png", "wheel_game/Fx_antenna_$t_44_2.ogg"},
	{"wheel_game/fx.png", "wheel_game/Fx_electric_$t_44_1.ogg"},
	{"wheel_game/soundscape.png", "wheel_game/Fx_whoosh_$t_44_2.ogg"},
	{NULL,NULL}
};


/*
loop_type:
100=rhythms
200=melodics
300=harmonics
400=effects

returns number of this type of sample_buttons
*/
static int add_sample_buttons(ClutterActor* looper, samplebutton_tuple table[], int loop_type) {
	samplebutton_tuple new_button=table[0];
	int counter=0;
	while(new_button.image){
		ClutterActor* sample_button;
		gchar* full_image = tangle_lookup_filename(new_button.image);
		gchar* full_sound = g_strdup_printf("%s/%s", DATA_DIR,new_button.sound);
		//printf("new_sample: '%s', '%s'\n",full_image,full_sound);
		sample_button = jammo_sample_button_new(); //TODO: jammo_sample_button_new_with_type
		g_object_set(sample_button,"sequencer", jammo_get_object_by_id("fullsequencer-the-sequencer"),NULL);
		jammo_sample_button_set_image_filename(JAMMO_SAMPLE_BUTTON(sample_button),full_image);
		jammo_sample_button_set_sample_filename(JAMMO_SAMPLE_BUTTON(sample_button), full_sound);

		g_object_set(sample_button,"drag-threshold-y",10000,NULL); //No dragging by vertical-axis (it is for scrolling wheel)

		jammo_sample_button_set_loop_id(JAMMO_SAMPLE_BUTTON(sample_button), counter+loop_type);
		g_free(full_image);
		g_free(full_sound);

		tangle_widget_add(TANGLE_WIDGET(looper),sample_button,NULL);
		counter++;
		new_button=table[counter];
	}
	return counter;
}



void sequencer_loop_tune_wheels() {
	//Static
	rhytmical_sample_looper  =  jammo_get_actor_by_id("fullsequencer-loop-rhytmical-wheel-scroller");
	melodical_sample_looper  =  jammo_get_actor_by_id("fullsequencer-loop-melodical-wheel-scroller");
	harmonical_sample_looper =  jammo_get_actor_by_id("fullsequencer-loop-harmonical-wheel-scroller");
	effect_sample_looper     =  jammo_get_actor_by_id("fullsequencer-loop-effect-wheel-scroller");

	length_of_rhythms = add_sample_buttons(rhytmical_sample_looper,rhythms,100);
	length_of_melodics = add_sample_buttons(melodical_sample_looper,melodics,200);
	length_of_harmonics = add_sample_buttons(harmonical_sample_looper,harmonics,300);
	length_of_effects = add_sample_buttons(effect_sample_looper,effects,400);

	fine_tune_wheel(rhytmical_sample_looper);
	fine_tune_wheel(melodical_sample_looper);
	fine_tune_wheel(harmonical_sample_looper);
	fine_tune_wheel(effect_sample_looper);
}



/*
Each jammo-sample-button has loop-id.
First digit tells type of loop.

Will return NULL, if anything goes wrong.

*/
ClutterActor* sequencer_loop_give_sample_button_for_this_id(guint id){
	samplebutton_tuple* asked_table;
	int length_of_asked;
	guint id_; //Id without first digit
	//gchar* type;  //TODO: use this
	if (id>=100 && id <200){
		asked_table=rhythms;
		length_of_asked=length_of_rhythms;
		id_=id-100;
	} else if (id>=200 && id <300) {
		asked_table=melodics;
		length_of_asked=length_of_melodics;
		id_=id-200;
	} else if (id>=300 && id <400) {
		asked_table=harmonics;
		length_of_asked=length_of_harmonics;
		id_=id-300;
	}	else if (id>=400 && id <500) {
		asked_table=effects;
		length_of_asked=length_of_effects;
		id_=id-400;
	} else {
		return NULL;
	}

	if (id_>length_of_asked)
		return NULL;

	ClutterActor* sample_button;
	samplebutton_tuple new_button=asked_table[id_];
	gchar* full_image = tangle_lookup_filename(new_button.image);
	gchar* full_sound = g_strdup_printf("%s/%s", DATA_DIR,new_button.sound);
	//printf("new_sample: '%s', '%s'\n",full_image,full_sound);
	sample_button = jammo_sample_button_new(); //TODO: jammo_sample_button_new_with_type
	//g_object_set(sample_button,"sequencer", jammo_get_object_by_id("fullsequencer-the-sequencer"),NULL);
	jammo_sample_button_set_image_filename(JAMMO_SAMPLE_BUTTON(sample_button),full_image);
	jammo_sample_button_set_sample_filename(JAMMO_SAMPLE_BUTTON(sample_button), full_sound);

	jammo_sample_button_set_loop_id(JAMMO_SAMPLE_BUTTON(sample_button), id);
	g_free(full_image);
	g_free(full_sound);

	return sample_button;
}
