/**sequencer.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is center of 7-12 years full sequencer view.
View contains three part:
 *wheel-game (loop-view)
 *track-view
 *general-view (settings)

*/

#include <glib-object.h>
//#include <clutter/clutter.h>
#include <math.h>
#include <string.h>

#include <tangle.h>

#include "../../meam/jammo-meam.h"
#include "../../meam/jammo-slider-event.h"
#include "../../meam/jammo-slider-track.h"
#include "../../meam/jammo-backing-track.h"
#include "../../meam/jammo-metronome-track.h"

#include "../jammo.h"
#include "../jammo-mentor.h"
#include "../jammo-editing-track-view.h"
#include "../jammo-miditrack-view.h"
#include "../../cem/cem.h"

#include "sequencer.h"
#include "sequencer_loop.h"
#include "sequencer_track_view.h"
#include "startmenu.h"


/*
Sequencer and sequencer-track-view-area are defined in json.

When sequencer starts: it calls load_state_from_file(load_only_backingtrack=TRUE)
 It starts loading backing-track.
 When backing_track is loaded (on_duration_notify)
  Tell sequencer_track_view_area that now duration is known (used e.g. with cursor)
  Then load_state_from_file(load_only_backingtrack=FALSE) loads another tracks.
*/

//This is The sequencer of the game. All 'play'-buttons should play this.
//All new tracks are added to this
//If metronome is enabled it is added to this
JammoSequencer* static_sequencer;
JammoMetronomeTrack* static_metronome_track;

static void use_backing_track_duration(void);
static void on_duration_notify(GObject* object, GParamSpec* pspec, gpointer track);



static void load_state_from_file(gchar* filename, gboolean load_only_backingtrack) {

JsonParser *parser;
parser = json_parser_new ();
g_assert (JSON_IS_PARSER (parser));

GError *error = NULL;
if (!json_parser_load_from_file (parser, filename, &error))
	{
		g_print ("Error: %s\n", error->message);
		g_error_free (error);
		g_object_unref (parser);
		return;
	}

JsonNode *root;
JsonObject *object;
JsonNode *node;

g_assert (NULL != json_parser_get_root (parser));

root = json_parser_get_root (parser);
g_assert_cmpint (JSON_NODE_TYPE (root), ==, JSON_NODE_OBJECT);

object = json_node_get_object (root);
g_assert (object != NULL);

	gint tempo = 110; //'normal' tempo
	gchar* pitch ="D";  //'normal' pitch

	//META data: Load these only once
if (load_only_backingtrack){
	//int
	node = json_object_get_member (object, "tempo");
	if (node!=NULL && JSON_NODE_TYPE (node) == JSON_NODE_VALUE){
		tempo = json_node_get_int (node);
		printf("tempo: '%d'\n",tempo);
	}
	//string
	node = json_object_get_member (object, "pitch");
	if (node!=NULL && JSON_NODE_TYPE (node) == JSON_NODE_VALUE){
		pitch = (gchar*)  json_node_get_string (node);
		printf("pitch: '%s'\n",pitch);
	}

	jammo_sequencer_set_tempo(static_sequencer,tempo);
	jammo_sequencer_set_pitch(static_sequencer,pitch);

	//Correct GUI-elements:
	gchar* name_of_tempo_button;
	if (tempo==130) name_of_tempo_button="tempo-button1";
	else if (tempo==110) name_of_tempo_button="tempo-button2";
	else /*(tempo==90)*/ name_of_tempo_button="tempo-button3";
	tangle_button_set_selected(TANGLE_BUTTON(jammo_get_actor_by_id(name_of_tempo_button)),TRUE);

	gchar* name_of_pitch_button;
	if (pitch[0]=='G') name_of_pitch_button="pitch-button1";
	else if (pitch[0]=='D') name_of_pitch_button="pitch-button2";
	else /* (pitch[0]=='A')*/ name_of_pitch_button="pitch-button3";
	tangle_button_set_selected(TANGLE_BUTTON(jammo_get_actor_by_id(name_of_pitch_button)),TRUE);
}


	ClutterActor* track_view = NULL; //Will be jammo_trac_view or jammo_miditrack_view or TangleWidget
	JammoTrack* track = NULL;

	//Array
	node = json_object_get_member (object, "tracks");
	JsonArray* track_array;
	guint length_tracks = 0;
	if (node!=NULL && JSON_NODE_TYPE (node) == JSON_NODE_ARRAY){
		track_array =  json_node_get_array (node);
		length_tracks = json_array_get_length(track_array);
	}
	else {printf("Not any tracks\n"); }

	printf("length %d\n",length_tracks);
	int j;
	for (j=0;j<length_tracks;j++) {
		track_view = NULL;
		track = NULL;

		//Object
		JsonNode* track_node;
		track_node = json_array_get_element(track_array,j);

		if (track_node!=NULL && JSON_NODE_TYPE(track_node) == JSON_NODE_OBJECT){
			printf("Found track!\n");
			gchar* track_type="";

			JsonObject* sub_object = json_node_get_object(track_node);
			JsonNode *sub_node;
			sub_node = json_object_get_member (sub_object, "track-type");
			if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
				track_type =(gchar*)  json_node_get_string (sub_node);
				printf(" track-type: '%s'\n",track_type);
				}

			//boolean. We need this to construct track_view
			gboolean muted=FALSE; //default
			sub_node = json_object_get_member (sub_object, "muted");
			if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
				muted = json_node_get_boolean (sub_node);
				printf(" muted: '%d'\n",muted);
			}

			// double/float
			gfloat volume=1.0; //default volume is 100%
			sub_node = json_object_get_member (sub_object, "volume");
			if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
				volume = (gfloat) json_node_get_double (sub_node);
				printf("volume: '%f'\n",volume);
			}

			if(load_only_backingtrack) {
			//BACKING-TRACK
			if (strncmp(track_type, "BackingTrack", 12) == 0) {

					printf("Found mandatory BackingTrack\n");
					gchar* backing_track_filename="";
					//string
					sub_node = json_object_get_member (sub_object, "audio-file");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						backing_track_filename =(gchar*)  json_node_get_string (sub_node);
						printf("backing_track_filename: '%s'\n",backing_track_filename);

						track = JAMMO_TRACK(jammo_backing_track_new(backing_track_filename));
						// Create other tracks after the duration of the backing track is known.
						
						if (jammo_track_get_duration(track) == JAMMO_DURATION_INVALID) {
							g_signal_connect(track, "notify::duration", G_CALLBACK(on_duration_notify), NULL);
						} else {
							use_backing_track_duration();
						}
						track_view=sequencer_track_view_add_backing_track(track, muted,volume);
					}
					printf("Ending looking for BackingTrack\n");

				}
			} //End load_only_backingtrack

			else {
			//EDITING-TRACK
			if(strncmp(track_type, "EditingTrack", 12) == 0) {
				gint e_track_type=JAMMO_SAMPLE_UNTYPED;
				gchar* editing_track_type="";
				//string
				sub_node = json_object_get_member (sub_object, "editing-track-type");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					editing_track_type =(gchar*)  json_node_get_string (sub_node);
					printf("editing-track-type: '%s'\n",editing_track_type);
				}
				ClutterColor sample_color =  { 0, 0, 0, 255 };

				GEnumClass* enum_class = G_ENUM_CLASS(g_type_class_ref(JAMMO_TYPE_SAMPLE_TYPE));
				GEnumValue* enum_value = g_enum_get_value_by_nick(enum_class, editing_track_type);
				g_type_class_unref(enum_class);
				if (enum_value)
					e_track_type = enum_value->value;

				track_view=sequencer_track_view_add_with_type(EDITING_TRACK_VIEW,muted,volume,e_track_type);

				if (e_track_type == JAMMO_SAMPLE_RHYTMICAL) {
					sample_color.red   = 255;
					sample_color.green = 155;
					sample_color.blue  = 0;
				}
				else if (e_track_type == JAMMO_SAMPLE_MELODICAL) {
					sample_color.red   = 0;
					sample_color.green = 0;
					sample_color.blue  = 155;
				}
				else if (e_track_type == JAMMO_SAMPLE_HARMONICAL) {
					sample_color.red   = 200;
					sample_color.green = 0;
					sample_color.blue  = 0;
				}
				else if (e_track_type == JAMMO_SAMPLE_EFFECT) {
					sample_color.red    = 0;
					sample_color.green  = 155;
					sample_color.blue   = 0;
				}
				else {
					sample_color.alpha   = 0;
				}

				g_object_set(track_view, "sample-color", &sample_color, NULL);


				//Array
				sub_node = json_object_get_member (sub_object, "samples");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_ARRAY){
					JsonArray* sample_array =  json_node_get_array (sub_node);

					guint length = json_array_get_length(sample_array);
					printf("length %d\n",length);
					int i;
					for (i=0;i<length;i++) {
						JsonNode* sample_node;
						sample_node = json_array_get_element(sample_array,i);
						if (sample_node!=NULL && JSON_NODE_TYPE(sample_node) == JSON_NODE_OBJECT){
							JsonObject* sample_object = json_node_get_object(sample_node);
							gint loop_id=-1;
							gint slot=-1;

							//int
							sub_node = json_object_get_member (sample_object, "loop_id");
							if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
								loop_id =  json_node_get_int (sub_node);
								printf("loop_id: '%d'\n",loop_id);
							}
							//int
							sub_node = json_object_get_member (sample_object, "slot");
							if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
								slot =  json_node_get_int (sub_node);
								printf("slot: '%d'\n",slot);
							}
							ClutterActor* sample_button=sequencer_loop_give_sample_button_for_this_id(loop_id);
							if (sample_button==NULL)
								printf("Loading loop-id '%d' from file: sample_button==NULL, can't add to track.\n",loop_id);
							else
								jammo_editing_track_view_add_jammo_sample_button(JAMMO_EDITING_TRACK_VIEW(track_view),JAMMO_SAMPLE_BUTTON(sample_button),slot);

						} //This sample-object is ready

					} //Foreach in sample-array ready

				} //sample-array is ready
			} //Editing-track over

			//VirtualInstrumentTrack
			else if(strncmp(track_type, "VirtualInstrumentTrack", 22) == 0) {
				//int
				gint instrument=0; //default instrument
				sub_node = json_object_get_member (sub_object, "instrument");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						instrument = json_node_get_int (sub_node);
						printf("instrument: '%d'\n",instrument);
					}

				track_view=sequencer_track_view_add_with_type(INSTRUMENT_TRACK_VIEW,muted,volume,instrument);

				//string
				sub_node = json_object_get_member (sub_object, "note-file");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					gchar* note_file = (gchar*) json_node_get_string (sub_node);
					printf("loading notes from file '%s'\n",note_file);
					GList* events = jammomidi_file_to_glist(note_file);
					jammo_miditrack_view_add_event_list(JAMMO_MIDITRACK_VIEW(track_view),events);
				}
			} //VirtualInstrumentTrack over


			//SliderTrack
			else if(strncmp(track_type, "SliderTrack", 11) == 0) {
				//int
				gint instrument=0; //default instrument
				sub_node = json_object_get_member (sub_object, "instrument");
					if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
						instrument = json_node_get_int (sub_node);
						printf("instrument: '%d'\n",instrument);
					}

				track_view=sequencer_track_view_add_with_type(SLIDER_TRACK_VIEW,muted,volume,instrument);
				//In slider-case: track_view is TangleWidget.
				track = g_object_get_data(G_OBJECT(track_view), "track");

				//string
				sub_node = json_object_get_member (sub_object, "note-file");
				if (sub_node!=NULL && JSON_NODE_TYPE (sub_node) == JSON_NODE_VALUE){
					gchar* note_file = (gchar*) json_node_get_string (sub_node);
					printf("loading notes from file '%s'\n",note_file);
					GList* events = jammo_slider_event_file_to_glist(note_file);
					jammo_slider_track_set_event_list(JAMMO_SLIDER_TRACK(track), events); //We need access to track here!
				}
			} //SliderTrack over

			} //End else- load_only_backingtrack


			//Tune volume, mute, etc.
			//In case of slider and backing_track, we already have 'track'
			if (!track) {
				if (track_view && (JAMMO_IS_EDITING_TRACK_VIEW(track_view) || JAMMO_IS_MIDITRACK_VIEW(track_view)))
					g_object_get(track_view,"track",&track,NULL);
			}

			//If we didn't got track, skip this
			if (track) {
				//These are asked earlier
				jammo_playing_track_set_volume(JAMMO_PLAYING_TRACK(track), volume);
				jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(track), muted);
			}

		} //Track is ready

	} //Next track

g_object_unref(parser);

}



gboolean sequencer_change_to_loop_view(TangleActor *actor, gpointer data) {
	printf("changing view to loop \n");
	ClutterActor* scrolling_view = jammo_get_actor_by_id("fullsequencer-view-container");
	ClutterAction* action = tangle_actor_get_action_by_type(scrolling_view,TANGLE_TYPE_SCROLL_ACTION);
	tangle_object_animate(G_OBJECT(action), CLUTTER_EASE_IN_CIRC, 400, "offset-y", 0.0, NULL);
	return TRUE;
}

gboolean sequencer_change_to_sequencer_view(TangleActor *actor, gpointer data) {
	printf("changing view to sequencer \n");
	ClutterActor* scrolling_view = jammo_get_actor_by_id("fullsequencer-view-container");
	ClutterAction* action = tangle_actor_get_action_by_type(scrolling_view,TANGLE_TYPE_SCROLL_ACTION);
	tangle_object_animate(G_OBJECT(action), CLUTTER_EASE_IN_CIRC, 400, "offset-y", 480.0, NULL);
	return TRUE;
}

gboolean sequencer_change_to_bottom_view (TangleActor *actor, gpointer data) {
	printf("changing view to bottom \n");
	ClutterActor* scrolling_view = jammo_get_actor_by_id("fullsequencer-view-container");
	ClutterAction* action = tangle_actor_get_action_by_type(scrolling_view,TANGLE_TYPE_SCROLL_ACTION);
	tangle_object_animate(G_OBJECT(action), CLUTTER_EASE_IN_CIRC, 400, "offset-y", 960.0, NULL);
	return TRUE;
}

static void use_backing_track_duration(void) {
	//Now we know duration of backing-track, which is also duration of Sequencer
	//so we can tune width of area. (needed for cursor e.g.)
	sequencer_track_view_tune_width();


	//continue loading
	load_state_from_file(inputFilename,FALSE);
}

static void on_duration_notify(GObject* object, GParamSpec* pspec, gpointer none) {
	use_backing_track_duration();

	// Do not call this function ever again!
	g_signal_handlers_disconnect_by_func(object, G_CALLBACK(on_duration_notify), NULL);
}

//We want load saved composition only first time
static gboolean first_time = TRUE;

/* JSON callback when sequencer-view is showed*/
void start_sequencer_full_sequencer (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	if (first_time){
		gchar* filename = tangle_lookup_filename("seq.json"); //This is default
		start_sequencer(filename);
		g_free(filename);
	}
	first_time=FALSE;
}

/*
This actor is visible when starting jammo_get_actor_by_id("fullsequencer-view");
*/
void start_sequencer(gpointer filename)
{
	printf("Starting sequencer GUI\n");
	ClutterAction* action = tangle_actor_get_action_by_type(jammo_get_actor_by_id("fullsequencer-view-container"), TANGLE_TYPE_SCROLL_ACTION);
	g_signal_connect_swapped(action, "clamp-offset-y", G_CALLBACK(tangle_widget_clamp_child_boundaries), jammo_get_actor_by_id("fullsequencer-view-container"));

	static_sequencer = JAMMO_SEQUENCER(jammo_get_object_by_id("fullsequencer-the-sequencer"));
	static_metronome_track = jammo_metronome_track_new();
	jammo_sequencer_add_track(static_sequencer, JAMMO_TRACK(static_metronome_track));
	jammo_metronome_track_set_time_signature_beats(static_metronome_track,4);
	jammo_metronome_track_set_time_signature_note_value(static_metronome_track,4);
	jammo_metronome_track_set_accent(static_metronome_track,FALSE);
	jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(static_metronome_track), TRUE);

	//inputFilename is global! (from sequencer.h)
	if (filename == NULL)
		inputFilename = NULL;
	else {
		inputFilename = malloc(strlen((gchar*) filename) + 1);
		memcpy(inputFilename, (gchar*) filename, strlen((gchar*) filename) + 1);
		printf("The filename is '%s'\n",inputFilename);
	}


	tangle_actor_hide_animated(TANGLE_ACTOR(jammo_mentor_get_default()));


	//Tune some things that can't be done with JSON only ("clamp-scrolling-offset-y")
	sequencer_loop_tune_wheels();

	//Loading tracks from file:
	if(inputFilename) {
		FILE *ifp;
		ifp = fopen(inputFilename, "r"); //currently open only for reading

		if(ifp != NULL) {
			load_state_from_file(inputFilename,TRUE);
		}
	} else {
		printf("No file specified for sequencer.\n");
	}


	sequencer_change_to_sequencer_view(NULL,NULL); //Start to middle view.
}

/*
void leave_sequencer()
{
	clutter_actor_hide(CLUTTER_ACTOR(sequencer_view));

	start_startmenu();
}
*/



/*
 * CallBack for JSON
 */
void fullsequencer_general_arrow_clicked (TangleButton *tanglebutton, gpointer user_data){
printf("up\n");
sequencer_change_to_sequencer_view(NULL,NULL);
}


void fullsequencer_general_tempo_clicked(TangleButton *tanglebutton, gpointer user_data) {
	printf("tempo clicked '%s'\n",clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)));

	int new_tempo = 110;
	if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"fast",4)==0)
			new_tempo=130;
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"normal",6)==0)
			new_tempo=110;
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"slow",4)==0)
			new_tempo=90;
	printf("new tempo:%d\n",new_tempo);
	jammo_sequencer_set_tempo(static_sequencer, new_tempo);
}


void fullsequencer_general_pitch_clicked(TangleButton *tanglebutton, gpointer user_data) {
	printf("pitch clicked '%s'\n",clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)));

	gchar* new_pitch = "G";
	if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"high",4)==0)
			new_pitch = "G";
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"normal",6)==0)
			new_pitch = "D";
	else if (strncmp(clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton)),"low",3)==0)
			new_pitch = "A";
	printf("new pitch:'%s'\n",new_pitch);

	jammo_sequencer_set_pitch(static_sequencer, new_pitch);
}


void sequencer_render_button_pressed (TangleButton* tanglebutton, gpointer none){
	cem_add_to_log("sequencer_render_button_pressed",LOG_USER_ACTION);
}

void fullsequencer_metronome_clicked (TangleButton* tanglebutton, gpointer none){
	printf("fullsequencer: Metronome pressed\n");

	jammo_playing_track_set_muted(JAMMO_PLAYING_TRACK(static_metronome_track), !tangle_button_get_selected(tanglebutton));
}
