/**sequencer_loop.c is part of JamMo.
 License: GPLv2, read more from COPYING

 This file is for clutter based gui.
 This is part of the sequencer.
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>
#include <tangle.h>

#include "sequencer_loop.h"
#include "sequencer.h"
#include "../jammo-sample-button.h"
#include "../jammo.h" //get_id

#include "../../cem/cem.h"

//Static pointers for loop rolls
static ClutterActor* rhytmical_sample_looper;
static ClutterActor* melodical_sample_looper;
static ClutterActor* harmonical_sample_looper;
static ClutterActor* effect_sample_looper;

gboolean sequencer_lock_pressed (TangleButton* tanglebutton, gpointer data){

	const gchar* name = clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));
	printf("lock '%s' pressed\n",name);
	ClutterActor* scroller = NULL;
	if (strncmp(name,"rhytm",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-rhytmical-wheel-scroller");
	else if (strncmp(name,"melod",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-melodical-wheel-scroller");
	else if (strncmp(name,"harmo",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-harmonical-wheel-scroller");
	else if (strncmp(name,"effec",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-effect-wheel-scroller");

	else {
		printf("can't detect which lock pressed\n");
		return FALSE;
	}

	ClutterAction* scroll_action = tangle_actor_get_action_by_type(scroller,TANGLE_TYPE_SCROLL_ACTION);
	gfloat threshold_x, threshold_y;

	if (scroller)  {
		tangle_scroll_action_get_thresholds(TANGLE_SCROLL_ACTION(scroll_action), &threshold_x,&threshold_y);
		if (threshold_y==10000.0)
			tangle_scroll_action_set_thresholds(TANGLE_SCROLL_ACTION(scroll_action), threshold_x,30.0);//TODO: use original default value
		else
			tangle_scroll_action_set_thresholds(TANGLE_SCROLL_ACTION(scroll_action), threshold_x,10000.0);
	}
	return TRUE;
}


/*
ScrollingActor doesn't have clampping. This will set it.
*/
static void fine_tune_wheel (ClutterActor* sample_looper) {
	ClutterAction* action;
	
	action = tangle_actor_get_action_by_type(sample_looper, TANGLE_TYPE_SCROLL_ACTION);
	g_signal_connect_swapped(action, "clamp-offset-y", G_CALLBACK(tangle_widget_clamp_child_boundaries), sample_looper);
}


#define ABS_F(x) ((x) < 0 ? -(x) : (x))
/*
Returns JammoSample closest to center of wheel. If wheel is rolling and two sample are equal
some of them are returned.
(This function can be used when wheel is running but there are now reason to do that).
Will return NULL in error cases
*/
static JammoSample* get_centermost_sample_from_loop_roll(ClutterActor* wheel) {
	gfloat y;
	gfloat offset=0,not_used; //How much scroller is scrolled
	gfloat height;    //Height of the scroller (we take this from color_box behind samples)

	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);
	tangle_scroll_action_get_offsets(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&offset);

	//Check height of first sample-button, suppose all have same height
	offset+= clutter_actor_get_height(tangle_widget_get_nth_child(TANGLE_WIDGET(wheel),0));

	//Check height of the wheel
	height = clutter_actor_get_height(wheel);
	//printf("height = %f\n",height);

	//Check all sample button which one is closest to center (='offset')
	float best_value = height;
	JammoSampleButton* closest = NULL;

	GList* children;
	for (children = tangle_widget_get_children_readonly(TANGLE_WIDGET(wheel)); children; children = children->next) {
		if (!JAMMO_IS_SAMPLE_BUTTON(children->data)){
			//printf("this is not sample_button\n");  //Color_box triggers this at least
			continue;
		}

		y = clutter_actor_get_y (children->data);
		if (y==0) y+=height; //actor in first slot should be consider be in after very last (+height)

		if (ABS_F(y-offset) < best_value) {
			best_value = ABS_F(y-offset);
			closest = JAMMO_SAMPLE_BUTTON(children->data);
		}

	}
	if (closest==NULL) return NULL;
	return jammo_sample_button_get_sample(closest);
}


/***************************
Functions for json
**************************/


/*
 * plays all samples from centerline of loop rolls
 */
void fullsequencer_loop_arrow_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_play_all_loops-clicked",LOG_USER_ACTION);
	//get_centermost_sample_from_loop_roll can return NULL (in error cases)
	JammoSample* r = get_centermost_sample_from_loop_roll(rhytmical_sample_looper);
	JammoSample* e = get_centermost_sample_from_loop_roll(effect_sample_looper);
	JammoSample* m = get_centermost_sample_from_loop_roll(melodical_sample_looper);
	JammoSample* h = get_centermost_sample_from_loop_roll(harmonical_sample_looper);

	//These play very same time even without adhoc sequencer and tracks.
	if (r!=NULL)   jammo_sample_play(r);
	if (e!=NULL)   jammo_sample_play(e);
	if (m!=NULL)   jammo_sample_play(m);
	if (h!=NULL)   jammo_sample_play(h);
}

/*
amount=how many slots.
If loop is already rolling, this will abort it and will make own.
Loop will still snap to grid always.
*/
static void rotate_loop_for_amount(ClutterActor* wheel, gint amount) {
	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);

	//Check if lock is pressed
	gfloat not_used,threshold_y;
	tangle_scroll_action_get_thresholds(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&threshold_y);

	//printf("threshold_y '%f'\n",threshold_y);
	if (threshold_y==10000.0) //means rolling is disabled for this wheel
		return;
	gfloat offset;
	tangle_scroll_action_get_offsets(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&offset);
	//printf("offset '%f'\n",offset);

	int current_slot = (int)(offset)/92; //round to lower slot, if rolling already
	gfloat target = (current_slot+amount) * 92.0;
	//printf("current_slot = %d, target pixel='%f'\n",current_slot,target);
	tangle_object_animate(G_OBJECT(scroll_action),  CLUTTER_EASE_IN_OUT_QUAD, 550, "offset-y", target, NULL);
}


/*
There are not yet logic for 'Cannot Fail'. It just turn them one slot forward. TODO
*/
void fullsequencer_loop_cannot_fail_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	rotate_loop_for_amount(rhytmical_sample_looper, 1);
	rotate_loop_for_amount(melodical_sample_looper, 1);
	rotate_loop_for_amount(harmonical_sample_looper, 1);
	rotate_loop_for_amount(effect_sample_looper, 1);
}


//There will be some logic
void fullsequencer_loop_crazy_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	GRand *random_gen = g_rand_new();

	gint rot_rhyt = g_rand_int_range(random_gen, -30, 30);
	gint rot_melo = g_rand_int_range(random_gen, -30, 30);
	gint rot_harm = g_rand_int_range(random_gen, -30, 30);
	gint rot_effe = g_rand_int_range(random_gen, -30, 30);

	rotate_loop_for_amount(rhytmical_sample_looper, rot_rhyt);
	rotate_loop_for_amount(melodical_sample_looper, rot_melo);
	rotate_loop_for_amount(harmonical_sample_looper, rot_harm);
	rotate_loop_for_amount(effect_sample_looper, rot_effe);

	g_rand_free(random_gen);
}


//////////////////////////////////////

//Bottom of the loop-view there are dropping-area, and when samplebutton is dragged here, view is changed.
gboolean sequencer_drag_begin(TangleDropAction* drop_action, TangleDragAction* drag_action){
	printf("jammo_sample_button dragged from wheel to bottom of the screen\n");
	sequencer_change_to_sequencer_view(NULL,NULL);
	
	return FALSE;
}


void sequencer_loop_tune_wheels() {

	//Static
	rhytmical_sample_looper  =  jammo_get_actor_by_id("fullsequencer-loop-rhytmical-wheel-scroller");
	melodical_sample_looper  =  jammo_get_actor_by_id("fullsequencer-loop-melodical-wheel-scroller");
	harmonical_sample_looper =  jammo_get_actor_by_id("fullsequencer-loop-harmonical-wheel-scroller");
	effect_sample_looper     =  jammo_get_actor_by_id("fullsequencer-loop-effect-wheel-scroller");

	fine_tune_wheel(rhytmical_sample_looper);
	fine_tune_wheel(melodical_sample_looper);
	fine_tune_wheel(harmonical_sample_looper);
	fine_tune_wheel(effect_sample_looper);
}
