
/*
 * This file is part of JamMo.
 *
 * This is center of midi-editor.
 * It contains piano_roll on left, menu on bottom and grid/note-area on center.
 *
 * It is meant to be called from fullsequencer-view or instrument_gui.
 * It needs JammoMiditrackView as parameter. Editor unparents+parents given
 * JammoMiditrackView.
 *
 * On debug purpose midi-editor can be started with filename pointing notelist.
 *   Then ending editor, will terminate program.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Aapo Rantalainen
 */


#include <tangle.h>
#include <glib-object.h>
#include <ctype.h>
#include <string.h>
#include <clutter/clutter.h>
#include "midi_editor.h"
#include "midi_editor_notes.h"
#include "../../meam/jammo-meam.h"
#include "../../meam/jammo-editing-track.h"
#include "../../meam/jammo-instrument-track.h"
#include "../../meam/jammo-midi.h"

#include "../jammo.h"
#include "../jammo-mentor.h"
#include "../jammo-cursor.h"
#include "../jammo-miditrack-view.h"
#include "midi_helper.h"

#include "midi_piano_roll.h"

//STATICS
static const ClutterColor red = { 255, 0, 0, 128 }; //for cursor
static gfloat height_of_grid_slot = 64.0;  //non-zoomed grid-size (700px/64 = 11 slots.  400px/64= 6 slots)
static gfloat width_of_grid_slot = 64.0;

static ClutterActor* piano_roll;
static JammoSequencer* sequencer;

//we got these as parameters and are returning them after use
static JammoMiditrackView* static_jammo_miditrack_view; //Editor doesn't make own miditrack-view, but uses existing!
static ClutterActor* parent;
static guint highest_note; //this comes inside static_jammo_miditrack_view
static guint lowest_note;  //this comes inside static_jammo_miditrack_view
static gfloat orig_slot_width, orig_slot_height, orig_x, orig_y;



static void start_midi_editor(void);

//This is for debugging only
void midi_editor_start_with_filename(gchar* note_file){
	//Make some default MiditracView

	sequencer = jammo_sequencer_new(); //static
	JammoInstrumentTrack* track;
	track = jammo_instrument_track_new(2); //2=UD
	jammo_instrument_track_set_realtime(track, FALSE);
	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(track));

	//limits:
	guint high = 60;
	guint low = 40;
	guint slots = 25;
	// duration_of_slot = 0,25s,
	ClutterActor* track_view = jammo_miditrack_view_new(track, slots, 250000000LL, width_of_grid_slot, height_of_grid_slot,high,low);
	parent=NULL; //This will trigger when we are quitting (we want program to end then)


	GList* events = jammomidi_file_to_glist(note_file);
	jammo_miditrack_view_add_event_list(JAMMO_MIDITRACK_VIEW(track_view),events);

	midi_editor_start_with_miditrack_view(JAMMO_MIDITRACK_VIEW(track_view));
}



/*
	Change zoom-button state. TODO
	Toggling between in/out. TODO
	Zoom track. Done.
	(hide grid?)
	Zoom piano_roll. Done.

	FIXME: if scrolled very bottom and zoomed in, there are first messy
*/
void midieditor_zoom_clicked (TangleAction* action, GObject* source, const gchar* trigger, TangleProperties* properties){

	//Track
	//Will resize total area of miditrack to 700x400. (width can vary, but height should be 400)
	jammo_miditrack_view_zoom_to_fit(static_jammo_miditrack_view,1400.0, 400.0);


	//Piano-roll. Delete previous, make new with correct size.
	ClutterActor* parent = clutter_actor_get_parent(piano_roll);
	clutter_actor_destroy(piano_roll);

	gfloat height;
	g_object_get(static_jammo_miditrack_view, "slot-height", &height,NULL);

	piano_roll = create_piano_roll(height,highest_note,lowest_note,FALSE);
	clutter_container_add_actor(CLUTTER_CONTAINER(parent), piano_roll);
}

/**
 * Plays the notes saved in the note collection
 * JSON callback

 TODO: stopping
 TODO: changing button texture
 */
void midieditor_playstop_clicked (TangleAction* action, GObject* source, const gchar* trigger, TangleProperties* properties){
	jammo_sequencer_stop(JAMMO_SEQUENCER(sequencer));
	jammo_sequencer_play(JAMMO_SEQUENCER(sequencer));
}



/**
 * This function set/unset pencil_mode.
 * JSON callback
 
 TODO: changing button texture
 */
void midieditor_pencil_clicked (TangleAction* action, GObject* source, const gchar* trigger, TangleProperties* properties){

	jammo_miditrack_view_set_pen_mode(static_jammo_miditrack_view, (!jammo_miditrack_view_get_pen_mode(static_jammo_miditrack_view)));
}



/**
 * Deletes the selected note from the view and note collection.
 * JSON callback
 *
 *TODO: Think how removing is done
  -Maybe we do not have way to 'select' notes.
  -So they are dropped to trash (trash is not button, but drag-area)
      but they are removed also when dragging anywhere?
 */
void midieditor_trash_clicked (TangleAction* action, GObject* source, const gchar* trigger, TangleProperties* properties){
	printf("trash clicked\n");
}


void midieditor_exit_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	printf("midieditor_exit_clicked\n");
	if (parent==NULL) { //This happens when started debug-mode
		printf("parent is NULL, quitting\n\n");
		exit(0);
	}

	//return miditrack-view to original parent and original attributes
	ClutterActor* actor = CLUTTER_ACTOR(static_jammo_miditrack_view);
	g_object_ref(actor);
	clutter_actor_unparent(actor);                                 //remove
	clutter_container_add(CLUTTER_CONTAINER(parent),actor,NULL);   //add
	clutter_actor_set_position(actor, orig_x, orig_y);
	jammo_miditrack_view_set_show_grid(static_jammo_miditrack_view,FALSE);  //hide grid
	guint slots;
	g_object_get(static_jammo_miditrack_view,"n-slots", &slots,NULL);
	jammo_miditrack_view_zoom_to_fit(static_jammo_miditrack_view,slots*orig_slot_width, (highest_note-lowest_note+1)* orig_slot_height);
	jammo_miditrack_view_set_pen_mode(static_jammo_miditrack_view, FALSE); //pen_mode=FALSE
	g_object_unref(actor);


	//hide editor-view:
	ClutterActor* mainview;
	mainview = jammo_get_actor_by_id("main-views-widget");
	clutter_container_foreach(CLUTTER_CONTAINER(mainview), chum_hide_this_tangle_actor, NULL);

	//show sequencer
	ClutterActor* view = jammo_get_actor_by_id("fullsequencer-view");
	clutter_actor_show(view);

	//Delete piano_roll (it is generated each time)
	clutter_actor_destroy(piano_roll);
}


/*
Load original position and size so they can be returned.
*/
void midi_editor_start_with_miditrack_view(JammoMiditrackView* miditrack_view) {
	// These are static
	static_jammo_miditrack_view=miditrack_view;
	parent = clutter_actor_get_parent(CLUTTER_ACTOR(static_jammo_miditrack_view));

	orig_x = clutter_actor_get_x(CLUTTER_ACTOR(miditrack_view));
	orig_y = clutter_actor_get_y(CLUTTER_ACTOR(miditrack_view));

	g_object_get(static_jammo_miditrack_view, "slot-width", &orig_slot_width, "slot-height", &orig_slot_height ,
              "highest-note", &highest_note, "lowest-note", &lowest_note,
              NULL);

	//Make editor
	start_midi_editor();
}


/**
 * Handle to start midi editor
 */
static void start_midi_editor(void) {
	printf("Starting midi editor\n");

	ClutterActor* mainview;
	mainview = jammo_get_actor_by_id("main-views-widget");
	clutter_container_foreach(CLUTTER_CONTAINER(mainview), chum_hide_this_tangle_actor, NULL);
	tangle_actor_hide_animated(TANGLE_ACTOR(jammo_mentor_get_default()));

	ClutterActor* view = jammo_get_actor_by_id("midieditor-view");
	clutter_actor_show(view);

	//CHUM (graphical)
	ClutterActor* container_for_piano_roll;
	ClutterActor* scrolling_actor;   //for track
	ClutterActor* scrolling_actor2;  //for piano_roll
	ClutterActor* texture;
	ClutterActor* cursor;

	ClutterActor* container_for_track = tangle_widget_new(); //we want wrap track inside tangle_widget

	//Background
	ClutterActor* background_gradient;
	ClutterColor* upper_color;
	ClutterColor* bottom_color;
	upper_color = clutter_color_new(255, 0, 0, 255);
	bottom_color = clutter_color_new(0, 0, 255, 255);
	background_gradient = tangle_linear_gradient_new(upper_color, bottom_color);
	tangle_widget_set_background_actor(TANGLE_WIDGET(container_for_track), background_gradient);


	//Miditrack.
	//We have got ready-miditrack, use it.
	ClutterActor* actor = CLUTTER_ACTOR(static_jammo_miditrack_view); //make things easier
	g_object_ref(actor);

	clutter_actor_unparent(actor);                                     //remove
	tangle_widget_add(TANGLE_WIDGET(container_for_track),actor,NULL);  //add
	clutter_actor_set_position(actor, 1.0, 1.0); //1.0,1.0 unless leftmost and topmost borders are not visible
	jammo_miditrack_view_set_show_grid(static_jammo_miditrack_view,TRUE);  //show grid
	guint slots;
	g_object_get(static_jammo_miditrack_view,"n-slots", &slots, NULL);
	jammo_miditrack_view_zoom_to_fit(static_jammo_miditrack_view,slots*width_of_grid_slot, (highest_note-lowest_note+1) *height_of_grid_slot);

	g_object_unref(actor);

	scrolling_actor = tangle_scrolling_actor_new(container_for_track);
	clutter_actor_set_position(scrolling_actor, 100.0, 0.0); //100px for piano_roll on left
	clutter_actor_set_size(scrolling_actor, 700.0, 400.0);   //80px for control buttons on bottom
	clutter_container_add(CLUTTER_CONTAINER(view), scrolling_actor, NULL);

	//PianoRoll on left
	//We need container for piano roll because we want change content of piano_roll (e.g. when zooming)
	container_for_piano_roll = tangle_widget_new();
	scrolling_actor2 = tangle_scrolling_actor_new(container_for_piano_roll);
	clutter_actor_set_position(scrolling_actor2, 0.0, 0.0);
	clutter_actor_set_size(scrolling_actor2, 100.0, 400.0);
	clutter_container_add(CLUTTER_CONTAINER(view), scrolling_actor2, NULL);

	piano_roll = create_piano_roll(height_of_grid_slot,highest_note,lowest_note,TRUE);
	clutter_container_add_actor(CLUTTER_CONTAINER(container_for_piano_roll), piano_roll);


	//Bind piano_roll and tracks together.
	tangle_binding_new(G_OBJECT(scrolling_actor2), "scrolling-offset-y", G_OBJECT(scrolling_actor), "scrolling-offset-y");


	//MEAM (audio). FIXME. do not use internal sequencer, but existing one
	sequencer = jammo_sequencer_new(); //static
	JammoInstrumentTrack* track;
	track = jammo_instrument_track_new(2); //2=UD
	jammo_instrument_track_set_realtime(track, FALSE);
	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(track));


	//Cursor (doesn't work if there are only instrument-track)
	texture = clutter_rectangle_new_with_color(&red);
	clutter_actor_set_size(texture, 10.0, 78.0);
	cursor = jammo_cursor_new(sequencer, texture);
	tangle_actor_set_depth_position(TANGLE_ACTOR(cursor), 1);
	clutter_actor_set_position(cursor, 0.0, 200.0);
	clutter_actor_set_size(cursor, 64.0*25, 180.0); //width=grid_width*slot_numbers, height (not used) = grid_height*(high-low)
	clutter_container_add_actor(CLUTTER_CONTAINER(view), cursor);
}

