/*
 * jammo-backing-track.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */
 
#include "jammo-backing-track.h"
#include "jammo-meam-private.h"
#include <string.h>

G_DEFINE_TYPE(JammoBackingTrack, jammo_backing_track, JAMMO_TYPE_PLAYING_TRACK);

enum {
	PROP_0,
	PROP_FILENAME
};

struct _JammoBackingTrackPrivate {
	GstElement* bin;
	gchar* filename;
	guint64 duration;
};

typedef struct {
	JammoBackingTrack* backing_track;
	GstElement* pipeline;
} PrerollData;

static void on_new_decoded_pad(GstElement* element, GstPad* pad, gboolean last, gpointer data);
static void preroll(JammoBackingTrack* backing_track);

JammoBackingTrack* jammo_backing_track_new(const gchar* filename) {

	return JAMMO_BACKING_TRACK(g_object_new(JAMMO_TYPE_BACKING_TRACK, "filename", filename, NULL));
}


static void jammo_backing_track_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoBackingTrack* backing_track;
	
	backing_track = JAMMO_BACKING_TRACK(object);

	switch (prop_id) {
		case PROP_FILENAME:
			g_assert(!backing_track->priv->filename);
			backing_track->priv->filename = g_strdup(g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_backing_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoBackingTrack* backing_track;

	backing_track = JAMMO_BACKING_TRACK(object);

        switch (prop_id) {
		case PROP_FILENAME:
			g_value_set_string(value, backing_track->priv->filename);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static guint64 jammo_backing_track_get_duration(JammoTrack* track) {

	return JAMMO_BACKING_TRACK(track)->priv->duration;
}

static GstElement* jammo_backing_track_get_playing_element(JammoPlayingTrack* playing_track) {

	return JAMMO_BACKING_TRACK(playing_track)->priv->bin;
}

static GObject* jammo_backing_track_constructor(GType type, guint n_properties, GObjectConstructParam* properties) {
	GObject* object;
	JammoBackingTrack* backing_track;
	GstElement* source;
	GstElement* decodebin;
	GstElement* convert;
	GstPad* pad;
	
	object = G_OBJECT_CLASS(jammo_backing_track_parent_class)->constructor(type, n_properties, properties);

	backing_track = JAMMO_BACKING_TRACK(object);

	g_assert(backing_track->priv->filename);

	preroll(backing_track);

	backing_track->priv->bin = gst_bin_new(NULL);

	source = gst_element_factory_make("filesrc", NULL);
	g_object_set(source, "location", backing_track->priv->filename, NULL);

	decodebin = gst_element_factory_make("decodebin", NULL);
	convert = gst_element_factory_make("audioconvert", NULL);
	g_signal_connect(decodebin, "new-decoded-pad", G_CALLBACK(on_new_decoded_pad), convert);

	gst_bin_add_many(GST_BIN(backing_track->priv->bin), source, decodebin, convert, NULL);
	gst_element_link(source, decodebin);

	pad = gst_element_get_pad(convert, "src");
	gst_element_add_pad(backing_track->priv->bin, gst_ghost_pad_new("src", pad));
	gst_object_unref(pad);

	return object;
}


static void jammo_backing_track_finalize(GObject* object) {
	JammoBackingTrack* backing_track;
	
	backing_track = JAMMO_BACKING_TRACK(object);
	g_free(backing_track->priv->filename);

	G_OBJECT_CLASS(jammo_backing_track_parent_class)->finalize(object);
}

static void jammo_backing_track_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_backing_track_parent_class)->dispose(object);
}

static void jammo_backing_track_class_init(JammoBackingTrackClass* backing_track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(backing_track_class);
	JammoTrackClass* track_class = JAMMO_TRACK_CLASS(backing_track_class);
	JammoPlayingTrackClass* playing_track_class = JAMMO_PLAYING_TRACK_CLASS(backing_track_class);

	gobject_class->constructor = jammo_backing_track_constructor;
	gobject_class->finalize = jammo_backing_track_finalize;
	gobject_class->dispose = jammo_backing_track_dispose;
	gobject_class->set_property = jammo_backing_track_set_property;
	gobject_class->get_property = jammo_backing_track_get_property;

	track_class->get_duration = jammo_backing_track_get_duration;
	playing_track_class->get_playing_element = jammo_backing_track_get_playing_element;

	/**
	 * JammoBackingTrack:filename:
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "File name",
	                                "The file name of the played audio",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(JammoBackingTrackPrivate));
}

static void jammo_backing_track_init(JammoBackingTrack* backing_track) {
	backing_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(backing_track, JAMMO_TYPE_BACKING_TRACK, JammoBackingTrackPrivate);
}

static void on_new_decoded_pad(GstElement* element, GstPad* pad, gboolean last, gpointer data) {
	GstElement* sink;
	GstPad* sinkpad;
	GstCaps* caps;

	sink = GST_ELEMENT(data);
	sinkpad = gst_element_get_pad(sink, "sink");
	if (!GST_PAD_IS_LINKED(sinkpad)) {
		caps = gst_pad_get_caps(pad);
		if (g_strrstr(gst_structure_get_name(gst_caps_get_structure(caps, 0)), "audio")) {
			gst_pad_link(pad, sinkpad);
		}
		gst_caps_unref(caps);
	}
	g_object_unref(sinkpad);
}

static gboolean preroll_bus_callback(GstBus* bus, GstMessage* message, gpointer data) {
	PrerollData* preroll_data;
	gboolean cleanup;
	GstFormat format;
	gint64 duration;
	
	preroll_data = (PrerollData*)data;
	cleanup = FALSE;
	if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ASYNC_DONE || GST_MESSAGE_TYPE(message) == GST_MESSAGE_EOS) {
		format = GST_FORMAT_TIME;
		if (gst_element_query_duration(preroll_data->pipeline, &format, &duration)) {
			preroll_data->backing_track->priv->duration = (guint64)duration;
			g_object_notify(G_OBJECT(preroll_data->backing_track), "duration");
		}
		cleanup = TRUE;
	} else if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR) {
		preroll_data->backing_track->priv->duration = 0;
		cleanup = TRUE;
	}

	if (cleanup) {
		g_object_unref(preroll_data->backing_track);
		gst_element_set_state(preroll_data->pipeline, GST_STATE_NULL);
		gst_object_unref(preroll_data->pipeline);
		g_free(preroll_data);
	}

	return !cleanup;
}

static void preroll(JammoBackingTrack* backing_track) {
	PrerollData* preroll_data;
	GstBus* bus;
	GstElement* filesrc;
	GstElement* decodebin;
	GstElement* sink;

	preroll_data = g_malloc0(sizeof(PrerollData));
	preroll_data->backing_track = backing_track;
	g_object_ref(preroll_data->backing_track);
	preroll_data->pipeline = gst_element_factory_make("pipeline", NULL);
	bus = gst_pipeline_get_bus(GST_PIPELINE(preroll_data->pipeline));
	gst_bus_add_watch(bus, preroll_bus_callback, preroll_data);
	gst_object_unref(bus);

	filesrc = gst_element_factory_make("filesrc", NULL);
	g_object_set(filesrc, "location", backing_track->priv->filename, NULL);
	decodebin = gst_element_factory_make("decodebin", NULL);
	sink = gst_element_factory_make("fakesink", NULL);
	gst_bin_add_many(GST_BIN(preroll_data->pipeline), filesrc, decodebin, sink, NULL);
	gst_element_link(filesrc, decodebin);
	g_signal_connect(decodebin, "new-decoded-pad", G_CALLBACK(on_new_decoded_pad), sink);

	gst_element_set_state(preroll_data->pipeline, GST_STATE_PLAYING); 
}
