/*
 * jammo.c (=> main.c)
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-chum.h"
#include "../meam/jammo-meam.h"
#include "jammo-mentor.h"
#include "jammo-mentor-action.h"
#include <tangle.h>
#include <clutter/clutter.h>
#include <stdlib.h> 

#include "jammo-mentor.h"
#include "chum.h" //e.g set_easy_game
#include <fcntl.h> //we check some files

#include <string.h>


#include "../gui_clutter/midi_editor.h"

#include "../gui_clutter/sequencer.h"
#include "../gui_clutter/startmenu.h"
#include "../gui_clutter/communitymenu.h"
#include "../gui_clutter/gamesmenu.h"

#include "../gems/gems.h"

#ifdef N900
#include <libosso.h>
#include <clutter/x11/clutter-x11.h>
#endif

//This is defined in welcome.c and will executed if nothing happens after 10 seconds
void welcome_door_clicked(TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties);
static gboolean door_timeouts(gpointer data);
static const char* search_path = "/opt/jammo";

/*
 * Note! You need Tangle Toolkit that is available here:
 *
 * http://gitorious.org/tangle
 *
 * To compile:
 * gcc -rdynamic -o jammo jammo.c jammo-*.c $(pkg-config --cflags --libs clutter-1.0 tangle gstreamer-0.10 gstreamer-base-0.10 gstreamer-controller-0.10) ../meam/libmeam.a
 *
 * Note that -rdynamic is important in order to dynamic signal handler binding to work!
 */

static ClutterScript* script;

ClutterActor* jammo_get_actor_by_id(const char* id) {
	ClutterActor* actor = NULL;

	if (!(actor = CLUTTER_ACTOR(clutter_script_get_object(script, id)))) {
		g_warning("Actor '%s' not found.", id);
	}
	
	return actor;
}

/* A TangleStylesheet directory is tried to load from the following locations:
 * 	/opt/jammo/stylesheet
 * 	<current_working_directory>/data/stylesheet
 */
static TangleStylesheet* get_jammo_stylesheet(void) {
	TangleStylesheet* stylesheet;
	gchar* cwd;
	gchar* filename;
	
	cwd = g_get_current_dir();
	filename = g_build_filename(cwd, "data", "stylesheet", NULL);

	if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
		stylesheet = tangle_stylesheet_new_from_file(filename);
	} else {
		stylesheet = tangle_stylesheet_new_from_file("/opt/jammo/stylesheet");
	}

	g_free(cwd);
	g_free(filename);

	return stylesheet;
}

/* JSON files and content within those are searched from the following locations:
 * 	/opt/jammo
 * 	<current_working_directory>/data/jammo
 */
static void add_search_paths(ClutterScript* script) {
	const gchar* default_search_path = "/opt/jammo";
	gchar* development_search_path;
	gchar* cwd;
	const gchar* search_paths[2];

	cwd = g_get_current_dir();
	development_search_path = g_build_filename(cwd, "data", "jammo", NULL);

	search_paths[0] = development_search_path;
	search_paths[1] = default_search_path;
	clutter_script_add_search_paths(script, search_paths, G_N_ELEMENTS(search_paths));

	g_free(cwd);
	g_free(development_search_path);
}

#ifdef N900

static gboolean pause_display_blanking(gpointer user_data) {
	osso_context_t* osso_context;
	
	osso_context = (osso_context_t*)user_data;
	
	g_return_val_if_fail(osso_display_blanking_pause(osso_context) == OSSO_OK, FALSE);
	
	return TRUE;
}

static gint rpc_callback(const gchar* interface, const gchar* method, GArray* arguments, gpointer user_data, osso_rpc_t* retval) {
	ClutterStage* stage;
	Display* display;
	Window window;
	XEvent event;

	stage = CLUTTER_STAGE(user_data);
	
	if (!strcmp(interface, "org.umsic.jammo") && !strcmp(method, "top_application")) {
		display = clutter_x11_get_default_display();
		window = clutter_x11_get_stage_window(stage);

		XRaiseWindow(display, window);
		/* Fremantle window manager does not obey XRaiseWindow, but reacts on _NET_ACTIVE_WINDOW messages. */
		event.xclient.type = ClientMessage;
		event.xclient.message_type = XInternAtom(display, "_NET_ACTIVE_WINDOW", False);
		event.xclient.display = display;
		event.xclient.window = window;
		event.xclient.format = 32;
		event.xclient.data.l[0] = 1;
		event.xclient.data.l[1] = clutter_x11_get_current_event_time();
		event.xclient.data.l[2] = None;
		event.xclient.data.l[3] = 0;
		event.xclient.data.l[4] = 0;
		XSendEvent(display, clutter_x11_get_root_window(), False, SubstructureNotifyMask | SubstructureRedirectMask, &event);
	}
	
	retval->type = DBUS_TYPE_INVALID;
	
	return OSSO_OK;
}
#endif

/* Be sure there are needed place to save compositions and singings.
 */
static void create_directories(void) {
	gchar* cmd; 
	
	cmd= g_strdup_printf("mkdir -p %s",COMPOSITIONS_DIRECTORY);
	if (system(cmd)) {
		printf("Error, can't call '%s'\n", cmd);
	}
	g_free(cmd);
	
	cmd = g_strdup_printf("mkdir -p %s",SINGINGS_DIRECTORY);
	if (system(cmd)) {
		printf("Error, can't call '%s'\n", cmd);
	}
	g_free(cmd);
}

int main(int argc, char** argv) {
#ifdef N900
	osso_context_t* osso_context;
#endif

	TangleStylesheet* stylesheet;
	gchar* filename;
	ClutterActor* stage;
	GError* error = NULL;
	GObject* object;
	ClutterActor* actor;
	ClutterColor black = { 0, 0, 0, 255 };
	
	jammo_meam_init(&argc, &argv);
	jammo_chum_init(&argc, &argv);
	
	/* GEMS initialization and profile manager test */
	/*
	gems_components* gems_comp = jammo_gems_init(&argc, &argv);
	gems_profilemanager_login(gems_comp->pm,1);
	gems_list_profile_parameters(gems_comp->pm);
	gems_profilemanager_logout(gems_comp->pm);
	*/
	
	srand (time (NULL)); //We will use rand()

#ifdef N900
	osso_context = osso_initialize("org.umsic.jammo", VERSION, 0, NULL);
	g_assert(osso_context != NULL);

	osso_display_state_on(osso_context);
	osso_display_blanking_pause(osso_context);
	
	g_timeout_add(55000, pause_display_blanking, osso_context);
#endif

	create_directories();

	/* TODO: Temporary: make sure Mentor classes exists. */
	printf("%ld %ld\n", (long)JAMMO_TYPE_MENTOR, (long)JAMMO_TYPE_MENTOR_ACTION);

	if ((stylesheet = get_jammo_stylesheet())) {
		tangle_stylesheet_set_default(stylesheet);
	}

	script = clutter_script_new();
	add_search_paths(script);
	if (!(filename = clutter_script_lookup_filename(script, "jammo.json"))) {
		g_critical("File 'jammo.json' not found.");
		
		return 1;
	}
	if (!clutter_script_load_from_file(script, filename, &error)) {
		g_warning("Could not load file '%s': %s", filename, error->message);

		return 2;
	}
	g_free(filename);
	
	if (!(object = clutter_script_get_object(script, "main"))) {
		g_warning("Object 'main' not found from the file 'jammo.json'.");
		
		return 3;
	}
	
	if (!CLUTTER_IS_ACTOR(object)) {
		g_warning("Object 'main' is not ClutterActor in the file 'jammo.json'.");
		
		return 4;
	}
	
	actor = CLUTTER_ACTOR(object);
	
	stage = clutter_stage_get_default();
	clutter_stage_set_title(CLUTTER_STAGE(stage), "JamMo");
	clutter_stage_set_color(CLUTTER_STAGE(stage), &black);
	clutter_stage_set_key_focus(CLUTTER_STAGE(stage), NULL);

#ifdef N900
	g_assert(osso_rpc_set_default_cb_f(osso_context, rpc_callback, stage) == OSSO_OK);
	
	clutter_stage_set_fullscreen(CLUTTER_STAGE(stage), TRUE);
#else
	clutter_actor_set_size(stage, 800, 480);
#endif

	clutter_container_add_actor(CLUTTER_CONTAINER(stage), actor);
	
	clutter_actor_show(stage);

	//do not show mentor on startup
	clutter_actor_hide(CLUTTER_ACTOR(jammo_mentor_get_default()));
	
 if (argc==2 && strcmp(argv[1],"editor")==0) { 
	start_midi_editor();
	}
	else if (argc==2 && strcmp(argv[1],"sequencer")==0) {
		start_sequencer();
	}
	else if (argc==2 && strcmp(argv[1],"startmenu")==0) {
		start_startmenu();
	}
	else if (argc==2 && strcmp(argv[1],"gamesmenu")==0) {
		start_gamesmenu();
	}
	else if (argc==2 && strcmp(argv[1],"communitymenu")==0) {
		start_communitymenu();
	}
	
	g_timeout_add(10000, door_timeouts,NULL); //10 seconds and door will be opened automatically
	clutter_main();

	g_object_unref(stage);
	g_object_unref(script);
	g_object_unref(stylesheet);
	
	jammo_meam_cleanup();

	/* GEMS cleanup */
	//jammo_gems_cleanup(gems_comp);

	return 0;
}
static gboolean door_timeouts(gpointer data){
printf("door_timeouts\n");
welcome_door_clicked(NULL,NULL,NULL,NULL);
return FALSE;
}


//FIXME: this is copy-pasted from singing-game (solo/duetto and language-selection uses same kind of mechanism)
static gulong handler_for_mentor_say_selected;

static void mentor_say_what_is_selected(TangleActor *actor, ClutterActorBox *arg1, gpointer user_data){
	//printf("got '%s'\n",(gchar*)user_data);
	gchar *filepath1;
	filepath1 =  g_strdup_printf("/opt/jammo/mentor_speech/mentor%s_selected_fi.ogg", (gchar*)user_data);

	jammo_mentor_speak(jammo_mentor_get_default(), filepath1);
	g_free(filepath1);
	g_signal_handler_disconnect (actor,handler_for_mentor_say_selected);
}



void jammo_easy_changed(TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties) {
	ClutterActor* ct =CLUTTER_ACTOR(source);
	const gchar* name = clutter_actor_get_name(ct);
	printf("easy/advanced clicked. get_name: '%s'\n",name);


	ClutterActor* button;
	ClutterActor* texture;
	gchar *new_name = g_strdup_printf("main-menu-easy-selection-%s-texture", name);
 
	button = jammo_get_actor_by_id("main-menu-easy-selection");
	texture = jammo_get_actor_by_id(new_name);
	tangle_button_set_normal_background_actor(TANGLE_BUTTON(button), texture);

	ClutterActor* view = jammo_get_actor_by_id("main-menu-view");
	//clutter_actor_show(view); //no need to call show, it is handled via json
	
	if (strcmp(name,"easy")==0)
		{
		chum_set_easy_game(TRUE);
		handler_for_mentor_say_selected=g_signal_connect(view, "show-completed", G_CALLBACK(mentor_say_what_is_selected), "_easy");
		}
	else if (strcmp(name,"advanced")==0)
		{
		chum_set_easy_game(FALSE);
		handler_for_mentor_say_selected=g_signal_connect(view, "show-completed", G_CALLBACK(mentor_say_what_is_selected), "_advanced");
		}


}


