/*
 * jammo-track-view.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-track-view.h"
#include "jammo-sample-button.h"
#include "../meam/jammo-sample.h"
#include "jammo-mentor.h"

G_DEFINE_TYPE(JammoTrackView, jammo_track_view, TANGLE_TYPE_DROPPABLE_ACTOR);

enum {
	PROP_0,
	PROP_TRACK,
	PROP_N_SLOTS,
	PROP_SLOT_DURATION,
	PROP_SLOT_SIZE,
	PROP_LINE_EVERY_NTH_SLOT,
	PROP_DISABLED_SLOTS_BEGIN,
	PROP_DISABLED_SLOTS_END
};

struct _JammoTrackViewPrivate {
	JammoEditingTrack* track;
	guint n_slots;
	guint64 slot_duration;
	gfloat slot_size;
	guint line_every_nth_slot;
	guint disabled_slots_begin;
	guint disabled_slots_end;
	ClutterColor hilight_color;
	
	gfloat real_slot_size;
	gfloat hilighted_start;
	gfloat hilighted_end;
};

static void on_actor_added(TangleWidget* widget, ClutterActor* actor, gpointer user_data);
static void on_actor_removed(TangleWidget* widget, ClutterActor* actor, gpointer user_data);
static void move_others_right_if_overlapping(JammoTrackView* track_view, ClutterActor* sample_button, guint slot_index);

ClutterActor* jammo_track_view_new(JammoEditingTrack* track, guint n_slots, guint64 slot_duration, gfloat slot_size) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_TRACK_VIEW, "track", track, "n-slots", n_slots, "slot-duration", slot_duration, "slot-size", slot_size, NULL));
}

void jammo_track_view_add_jammo_sample_button(JammoTrackView* track_view, JammoSampleButton* sample_button, guint slot) {
	TangleWidget* widget;

	jammo_editing_track_add_sample(track_view->priv->track, jammo_sample_button_get_sample(sample_button), slot * track_view->priv->slot_duration);

	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));

	clutter_actor_set_x(CLUTTER_ACTOR(sample_button), slot * track_view->priv->real_slot_size);
	clutter_actor_set_y(CLUTTER_ACTOR(sample_button), 0.0);
	tangle_widget_add(TANGLE_WIDGET(widget), CLUTTER_ACTOR(sample_button), NULL);

	move_others_right_if_overlapping(track_view, CLUTTER_ACTOR(sample_button), slot);
}

guint jammo_track_view_get_sample_button_slot(JammoTrackView* track_view, JammoSampleButton* sample_button) {
	g_return_val_if_fail(clutter_actor_get_parent(CLUTTER_ACTOR(sample_button)) == tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)), (guint)-1);

	return (guint)(clutter_actor_get_x(CLUTTER_ACTOR(sample_button)) / track_view->priv->real_slot_size);
}

static gboolean jammo_track_view_is_dragging_droppable(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {
	gboolean droppable = FALSE;
	JammoTrackView* track_view;
	gfloat center_x, center_y;
	guint slot;
	guint n_slots;
	gfloat start;
	
	track_view = JAMMO_TRACK_VIEW(droppable_actor);
	
	if (JAMMO_IS_SAMPLE_BUTTON(dragging->draggable_actor) &&
	    clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->center_x, dragging->center_y, &center_x, &center_y)) {
		n_slots = (guint)(clutter_actor_get_width(CLUTTER_ACTOR(dragging->draggable_actor)) / track_view->priv->real_slot_size + 0.5);
		center_x -= (n_slots - 1) * track_view->priv->real_slot_size / 2;
		slot = center_x / track_view->priv->real_slot_size;

		if (slot >= track_view->priv->disabled_slots_begin &&
		    slot + n_slots - 1 < track_view->priv->n_slots - track_view->priv->disabled_slots_end) {
			droppable = TRUE;
			start = slot * track_view->priv->real_slot_size;
			if (start != track_view->priv->hilighted_start) {
				track_view->priv->hilighted_start = start;
				track_view->priv->hilighted_end = start + n_slots * track_view->priv->real_slot_size;
				clutter_actor_queue_redraw(CLUTTER_ACTOR(track_view));
			}
		}
	}
	
	if (!droppable && track_view->priv->hilighted_start != track_view->priv->hilighted_end) {
		track_view->priv->hilighted_start = track_view->priv->hilighted_end = 0.0;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(track_view));
	}
	    		
	return droppable;
}

static gboolean jammo_track_view_handle_dragging(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {

	return FALSE;
}

static void jammo_track_view_get_preferred_width(TangleActor* actor, gfloat for_height, gboolean interacting, gfloat* min_width_p, gfloat* natural_width_p, gfloat* max_width_p) {
	JammoTrackView* track_view;
	
	track_view = JAMMO_TRACK_VIEW(actor);
	
	if (min_width_p) {
		*min_width_p = track_view->priv->n_slots * track_view->priv->slot_size;
	}
	if (natural_width_p) {
		*natural_width_p = track_view->priv->n_slots * track_view->priv->slot_size;
	}
	if (max_width_p) {
		*max_width_p = 0.0;
	}
}

static void jammo_track_view_get_preferred_height(TangleActor* actor, gfloat for_width, gboolean interacting, gfloat* min_height_p, gfloat* natural_height_p, gfloat* max_height_p) {
	if (min_height_p) {
		*min_height_p = 80.0;
	}
	if (natural_height_p) {
		*natural_height_p = 80.0;
	}
	if (max_height_p) {
		*max_height_p = 0.0;
	}
}

static void jammo_track_view_allocate_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	JammoTrackView* track_view;
	ClutterActor* grid;

	track_view = JAMMO_TRACK_VIEW(wrapper_actor);

	TANGLE_WRAPPER_ACTOR_CLASS(jammo_track_view_parent_class)->allocate_wrapped(wrapper_actor, wrapped, box, flags);
	
	track_view->priv->real_slot_size = (box->x2 - box->x1) / track_view->priv->n_slots;

	if (track_view->priv->line_every_nth_slot) {
		grid = tangle_grid_new(track_view->priv->real_slot_size * track_view->priv->line_every_nth_slot - (1.0 / track_view->priv->n_slots), box->y2 - box->y1 - 1.0);
		tangle_widget_set_background_actor(TANGLE_WIDGET(wrapped), grid);
	} else {
		tangle_widget_set_background_actor(TANGLE_WIDGET(wrapped), NULL);
	}
}

static gboolean jammo_track_view_handle_dropping(TangleDroppableActor* droppable_actor, TangleDragging* dragging, gboolean accepted) {
	JammoTrackView* track_view;
	TangleWidget* widget;
	gfloat x, y, center_x, center_y;
	JammoSampleButton* sample_button;
	ClutterActor* copy_of_sample_button;
	guint slot;
	guint n_slots;
	
	track_view = JAMMO_TRACK_VIEW(droppable_actor);
	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));
	
	TANGLE_DROPPABLE_ACTOR_CLASS(jammo_track_view_parent_class)->handle_dropping(droppable_actor, dragging, accepted);

	if (accepted && clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->x, dragging->y, &x, &y) &&
	    clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->center_x, dragging->center_y, &center_x, &center_y)) {
		sample_button = JAMMO_SAMPLE_BUTTON(dragging->draggable_actor);
		copy_of_sample_button = jammo_sample_button_new_from_existing(sample_button);

		n_slots = (guint)(clutter_actor_get_width(CLUTTER_ACTOR(dragging->draggable_actor)) / track_view->priv->real_slot_size + 0.5);
		center_x -= (n_slots - 1) * track_view->priv->real_slot_size / 2;
		slot = center_x / track_view->priv->real_slot_size;

		jammo_editing_track_add_sample(track_view->priv->track, jammo_sample_button_get_sample(JAMMO_SAMPLE_BUTTON(copy_of_sample_button)), slot * track_view->priv->slot_duration);

		/* tangle_draggable_actor_set_floating_mode_dragged(TANGLE_DRAGGABLE_ACTOR(copy_of_sample_button), TANGLE_FLOATING_MODE_COLLAPSE); */
		clutter_actor_set_x(copy_of_sample_button, x - dragging->anchor_x);
		clutter_actor_set_y(copy_of_sample_button, y - dragging->anchor_y);
		tangle_widget_add(widget, copy_of_sample_button, NULL);

		clutter_actor_animate(copy_of_sample_button, CLUTTER_EASE_OUT_CUBIC, 300, "x", slot * track_view->priv->real_slot_size, "y", 0.0, NULL);

		move_others_right_if_overlapping(track_view, copy_of_sample_button, slot);
	}
	
	track_view->priv->hilighted_start = track_view->priv->hilighted_end = 0.0;

	return FALSE;
}

static void jammo_track_view_paint(ClutterActor* actor) {
	JammoTrackView* track_view;
	gdouble scale_x;
	gdouble scale_y;
	guint8 alpha;
	ClutterActorBox box;

	track_view = JAMMO_TRACK_VIEW(actor);

	if (CLUTTER_ACTOR_IS_MAPPED(actor)) {
		CLUTTER_ACTOR_CLASS(jammo_track_view_parent_class)->paint(actor);

		if (track_view->priv->hilighted_start != track_view->priv->hilighted_end) {
			cogl_push_matrix();

			g_object_get(actor, "transition-scale-x", &scale_x, "transition-scale-y", &scale_y, NULL);
			cogl_scale(scale_x, scale_y, 0);

			alpha = clutter_actor_get_paint_opacity(actor) * track_view->priv->hilight_color.alpha / 255;
			cogl_set_source_color4ub(track_view->priv->hilight_color.red,
			 			 track_view->priv->hilight_color.green,
			 			 track_view->priv->hilight_color.blue,
						 alpha);
			tangle_actor_get_aligned_allocation(TANGLE_ACTOR(actor), &box);
			cogl_rectangle(box.x1 + track_view->priv->hilighted_start, box.y1, box.x1 + track_view->priv->hilighted_end, box.y2);

			cogl_pop_matrix();
		}
	}
}

static GObject* jammo_track_view_constructor(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties) {
	GObject* object;
	JammoTrack* track;
	ClutterActor* widget;

	g_return_val_if_fail(tangle_lookup_construct_properties(type, n_construct_properties, construct_properties, "track", &track, NULL), NULL);
	g_return_val_if_fail(track != NULL, NULL);

	widget = tangle_widget_new();

	object = tangle_construct_with_extra_properties(G_OBJECT_CLASS(jammo_track_view_parent_class)->constructor, type, n_construct_properties, construct_properties, "wrapped", widget, NULL);

	g_signal_connect(widget, "actor-added", G_CALLBACK(on_actor_added), object);
	g_signal_connect(widget, "actor-removed", G_CALLBACK(on_actor_removed), object);

	return object;
}

static void jammo_track_view_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoTrackView* track_view;
	
	track_view = JAMMO_TRACK_VIEW(object);

	switch (prop_id) {
		case PROP_TRACK:
			track_view->priv->track = JAMMO_EDITING_TRACK(g_value_get_object(value));
			g_object_ref(track_view->priv->track);
			break;
		case PROP_N_SLOTS:
			track_view->priv->n_slots = g_value_get_uint(value);
			break;
		case PROP_SLOT_DURATION:
			track_view->priv->slot_duration = g_value_get_uint64(value);
			break;
		case PROP_SLOT_SIZE:
			track_view->priv->slot_size = g_value_get_float(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(track_view));
			break;
		case PROP_LINE_EVERY_NTH_SLOT:
			track_view->priv->line_every_nth_slot = g_value_get_uint(value);
			clutter_actor_queue_relayout(CLUTTER_ACTOR(track_view));
			break;
		case PROP_DISABLED_SLOTS_BEGIN:
			track_view->priv->disabled_slots_begin = g_value_get_uint(value);
			break;
		case PROP_DISABLED_SLOTS_END:
			track_view->priv->disabled_slots_end = g_value_get_uint(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_track_view_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoTrackView* track_view;

	track_view = JAMMO_TRACK_VIEW(object);

        switch (prop_id) {
		case PROP_TRACK:
			g_value_set_object(value, track_view->priv->track);
			break;
		case PROP_N_SLOTS:
			g_value_set_uint(value, track_view->priv->n_slots);
			break;
		case PROP_SLOT_DURATION:
			g_value_set_uint64(value, track_view->priv->slot_duration);
			break;
		case PROP_SLOT_SIZE:
			g_value_set_float(value, (track_view->priv->real_slot_size ? track_view->priv->real_slot_size : track_view->priv->slot_size));
			break;
		case PROP_LINE_EVERY_NTH_SLOT:
			g_value_set_uint(value, track_view->priv->line_every_nth_slot);
			break;
		case PROP_DISABLED_SLOTS_BEGIN:
			g_value_set_uint(value, track_view->priv->disabled_slots_begin);
			break;
		case PROP_DISABLED_SLOTS_END:
			g_value_set_uint(value, track_view->priv->disabled_slots_end);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_track_view_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_track_view_parent_class)->finalize(object);
}

static void jammo_track_view_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_track_view_parent_class)->dispose(object);
}

static void jammo_track_view_class_init(JammoTrackViewClass* track_view_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(track_view_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(track_view_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(track_view_class);
	TangleWrapperActorClass* wrapper_actor_class = TANGLE_WRAPPER_ACTOR_CLASS(track_view_class);
	TangleDroppableActorClass* droppable_actor_class = TANGLE_DROPPABLE_ACTOR_CLASS(track_view_class);

	gobject_class->constructor = jammo_track_view_constructor;
	gobject_class->finalize = jammo_track_view_finalize;
	gobject_class->dispose = jammo_track_view_dispose;
	gobject_class->set_property = jammo_track_view_set_property;
	gobject_class->get_property = jammo_track_view_get_property;
	
	clutter_actor_class->paint = jammo_track_view_paint;
	
	actor_class->get_preferred_width = jammo_track_view_get_preferred_width;
	actor_class->get_preferred_height = jammo_track_view_get_preferred_height;
	
	wrapper_actor_class->allocate_wrapped = jammo_track_view_allocate_wrapped;
	
	droppable_actor_class->is_dragging_droppable = jammo_track_view_is_dragging_droppable;
	droppable_actor_class->handle_dragging = jammo_track_view_handle_dragging;
	droppable_actor_class->handle_dropping = jammo_track_view_handle_dropping;

	/**
	 * JammoTrackView:track:
	 *
	 * The track of this view.
	 */
	g_object_class_install_property(gobject_class, PROP_TRACK,
	                                g_param_spec_object("track",
	                                "Track",
	                                "The track of this view",
	                                JAMMO_TYPE_EDITING_TRACK,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:n-slots:
	 *
	 * The number of slots.
	 */
	g_object_class_install_property(gobject_class, PROP_N_SLOTS,
	                                g_param_spec_uint("n-slots",
	                                "N slots",
	                                "The number of slots",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:slot-duration:
	 *
	 * The duration of a slot in nanoseconds.
	 */
	g_object_class_install_property(gobject_class, PROP_SLOT_DURATION,
	                                g_param_spec_uint64("slot-duration",
	                                "Slot duration",
	                                "The duration of a slot in nanoseconds",
	                                0, G_MAXUINT64, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:slot-size:
	 *
	 * The width of the slot in the screen (pixels, Clutter units).
	 */
	g_object_class_install_property(gobject_class, PROP_SLOT_SIZE,
	                                g_param_spec_float("slot-size",
	                                "Slot size",
	                                "The width of the slot in the screen (pixels, Clutter units)",
	                                0, G_MAXFLOAT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:slot-size:
	 *
	 * The number of slots after a separator line is drawn (0 to disable).
	 */
	g_object_class_install_property(gobject_class, PROP_LINE_EVERY_NTH_SLOT,
	                                g_param_spec_uint("line-every-nth-slot",
	                                "Line every nth slot",
	                                "The number of slots after a separator line is drawn (0 to disable)",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:disabled-slots-begin:
	 *
	 * The number of slots that are disabled (in terms of dropping) from the beginning of the track.
	 */
	g_object_class_install_property(gobject_class, PROP_DISABLED_SLOTS_BEGIN,
	                                g_param_spec_uint("disabled-slots-begin",
	                                "Disabled slots begin",
	                                "The number of slots that are disabled (in terms of dropping) from the beginning of the track",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoTrackView:disabled-slots-end:
	 *
	 * The number of slots that are disabled (in terms of dropping) from the end of the track.
	 */
	g_object_class_install_property(gobject_class, PROP_DISABLED_SLOTS_END,
	                                g_param_spec_uint("disabled-slots-end",
	                                "Disabled slots end",
	                                "The number of slots that are disabled (in terms of dropping) from the end of the track",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (JammoTrackViewPrivate));
}

static void jammo_track_view_init(JammoTrackView* track_view) {
	track_view->priv = G_TYPE_INSTANCE_GET_PRIVATE(track_view, JAMMO_TYPE_TRACK_VIEW, JammoTrackViewPrivate);

	track_view->priv->hilight_color.red = 255;
	track_view->priv->hilight_color.green = 255;
	track_view->priv->hilight_color.blue = 255;
	track_view->priv->hilight_color.alpha = 100;
}

static void on_actor_added(TangleWidget* widget, ClutterActor* actor, gpointer user_data) {
	JammoTrackView* track_view;
	JammoSampleButton* sample_button;
	JammoSample* sample;

	g_return_if_fail(JAMMO_IS_SAMPLE_BUTTON(actor));
	
	track_view = JAMMO_TRACK_VIEW(user_data);
	sample_button = JAMMO_SAMPLE_BUTTON(actor);
	sample = jammo_sample_button_get_sample(sample_button);
	
	//If user makes first dragging, not clicking, we do not want hear click-message anymore
	jammo_mentor_speak_once(jammo_mentor_get_default(), "/opt/jammo/mentor_speech/mentor_composing_first_click_fi.ogg");
	jammo_mentor_speak_once(jammo_mentor_get_default(), "/opt/jammo/mentor_speech/mentor_composing_first_drag_fi.ogg");

	g_assert(jammo_sample_get_editing_track(sample));
}

static void on_actor_removed(TangleWidget* widget, ClutterActor* actor, gpointer user_data) {
	JammoTrackView* track_view;
	JammoSampleButton* sample_button;

	g_return_if_fail(JAMMO_IS_SAMPLE_BUTTON(actor));
	
	track_view = JAMMO_TRACK_VIEW(user_data);
	sample_button = JAMMO_SAMPLE_BUTTON(actor);
	
	jammo_editing_track_remove_sample(track_view->priv->track, jammo_sample_button_get_sample(sample_button));
}

static void on_animation_completed(ClutterAnimation* animation, gpointer user_data) {
	TangleVault* vault;
	ClutterContainer* container;
	ClutterActor* actor;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 2, CLUTTER_TYPE_CONTAINER, &container, CLUTTER_TYPE_ACTOR, &actor);

	clutter_container_remove_actor(container, actor);
}

static void move_others_right_if_overlapping(JammoTrackView* track_view, ClutterActor* sample_button, guint slot_index) {
	TangleWidget* widget;
	guint n_slots;
	GList* children;
	ClutterActor* actor = NULL;
	gfloat x;
	ClutterAnimation* animation;
	TangleVault* vault;
	
	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));
	n_slots = (guint)(clutter_actor_get_width(sample_button) / track_view->priv->real_slot_size + 0.5);
	for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {
		actor = CLUTTER_ACTOR(children->data);
		x = clutter_actor_get_x(actor);
		if (actor != sample_button &&
		    x < slot_index * track_view->priv->real_slot_size &&
		    x + clutter_actor_get_width(actor) > slot_index * track_view->priv->real_slot_size) {
			n_slots += slot_index - (guint)(x / track_view->priv->real_slot_size);
			slot_index = (guint)(x / track_view->priv->real_slot_size);
			break;
		}
		actor = NULL;
	}
	if (!actor) {
		for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {
			actor = CLUTTER_ACTOR(children->data);
			x = clutter_actor_get_x(actor);
			if (actor != sample_button &&
			    x >= slot_index * track_view->priv->real_slot_size &&
			    x <= (slot_index + n_slots - 1) * track_view->priv->real_slot_size) {
		    		n_slots -= (guint)(x / track_view->priv->real_slot_size) - slot_index;
				break;
			}
			actor = NULL;
		}
	}
	if (actor) {
		for (children = tangle_widget_get_children_readonly(widget); children; children = children->next) {
			actor = CLUTTER_ACTOR(children->data);
			x = clutter_actor_get_x(actor);
			if (actor != sample_button && x >= slot_index * track_view->priv->real_slot_size) {
				x += n_slots * track_view->priv->real_slot_size;
				animation = clutter_actor_animate(actor, CLUTTER_EASE_OUT_CUBIC, 500, "x", x, NULL);
				if (x + clutter_actor_get_width(actor) > (track_view->priv->n_slots - track_view->priv->disabled_slots_end) * track_view->priv->real_slot_size) {
					vault = tangle_vault_new(2, CLUTTER_TYPE_CONTAINER, widget, CLUTTER_TYPE_ACTOR, actor);
					tangle_signal_connect_vault(animation, "completed", G_CALLBACK(on_animation_completed), vault);
				} else {
					jammo_editing_track_add_sample(track_view->priv->track, jammo_sample_button_get_sample(JAMMO_SAMPLE_BUTTON(actor)),
					                               x / track_view->priv->real_slot_size * track_view->priv->slot_duration);
				}
			}
		}
	}
}
