/*
 * jammo-cursor.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-cursor.h"

G_DEFINE_TYPE(JammoCursor, jammo_cursor, TANGLE_TYPE_WIDGET);

enum {
	PROP_0,
	PROP_SEQUENCER,
	PROP_TEXTURE
};

struct _JammoCursorPrivate {
	JammoSequencer* sequencer;
	ClutterActor* texture;
	
	guint timeout_source_id;
};

static void on_sequencer_started(JammoSequencer* sequencer, gpointer user_data);
static void on_sequencer_stopped(JammoSequencer* sequencer, gpointer user_data);

ClutterActor* jammo_cursor_new(JammoSequencer* sequencer, ClutterActor* texture) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_CURSOR, "sequencer", sequencer, "texture", texture, NULL));
}

static void jammo_cursor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoCursor* cursor;
	
	cursor = JAMMO_CURSOR(object);

	switch (prop_id) {
		case PROP_SEQUENCER:
			cursor->priv->sequencer = JAMMO_SEQUENCER(g_value_get_object(value));
			g_signal_connect(cursor->priv->sequencer, "started", G_CALLBACK(on_sequencer_started), cursor);
			g_signal_connect(cursor->priv->sequencer, "stopped", G_CALLBACK(on_sequencer_stopped), cursor);
			break;
		case PROP_TEXTURE:
			if (cursor->priv->texture) {
				clutter_container_remove_actor(CLUTTER_CONTAINER(cursor), cursor->priv->texture);
			}
			cursor->priv->texture = CLUTTER_ACTOR(g_value_get_object(value));
			clutter_container_add_actor(CLUTTER_CONTAINER(cursor), cursor->priv->texture);
			clutter_actor_hide(cursor->priv->texture);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_cursor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoCursor* cursor;

	cursor = JAMMO_CURSOR(object);

        switch (prop_id) {
		case PROP_SEQUENCER:
			g_value_set_object(value, cursor->priv->sequencer);
			break;
		case PROP_TEXTURE:
			g_value_set_object(value, cursor->priv->texture);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_cursor_class_init(JammoCursorClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->set_property = jammo_cursor_set_property;
	gobject_class->get_property = jammo_cursor_get_property;

	g_type_class_add_private(gobject_class, sizeof(JammoCursorPrivate));
}

static void jammo_cursor_init(JammoCursor* cursor) {
	cursor->priv = G_TYPE_INSTANCE_GET_PRIVATE(cursor, JAMMO_TYPE_CURSOR, JammoCursorPrivate);
	
	g_object_set(G_OBJECT(cursor), "pickable", FALSE, NULL);
}

static gboolean on_timeout(gpointer user_data) {
	JammoCursor* cursor;
	
	cursor = JAMMO_CURSOR(user_data);

	return FALSE;
}

static void on_sequencer_started(JammoSequencer* sequencer, gpointer user_data) {
	JammoCursor* cursor;
	
	cursor = JAMMO_CURSOR(user_data);

	if (cursor->priv->texture) {
		clutter_actor_show(cursor->priv->texture);
	}
	if (!cursor->priv->timeout_source_id) {
		cursor->priv->timeout_source_id = g_timeout_add(40, on_timeout, cursor);
	}
}

static void on_sequencer_stopped(JammoSequencer* sequencer, gpointer user_data) {
	JammoCursor* cursor;
	
	cursor = JAMMO_CURSOR(user_data);
	
	if (cursor->priv->texture) {
		clutter_actor_hide(cursor->priv->texture);
	}
	if (cursor->priv->timeout_source_id) {
		g_source_remove(cursor->priv->timeout_source_id);
		cursor->priv->timeout_source_id = 0;
	}
}
