#ifdef OLD_TANGLE_IN_REPOSITORY
/*
 * jammo-mentor.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-mentor.h"
#include "../meam/jammo-sample.h"

G_DEFINE_TYPE(JammoMentor, jammo_mentor, CLUTTER_ACTOR);

enum {
	PROP_0,
	PROP_STANDBY_SCALE,
	PROP_IDLE_SPEECH,
	PROP_IDLE_TIMEOUT
};

struct _JammoMentorPrivate {
	gfloat standby_scale;
	gchar* idle_speech;
	guint idle_timeout;
	
	GList* spoken_speeches;
	guint idle_timeout_id;
	ClutterScript* script;
	JammoSample* active_sample;
};

static void speak(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data);
static void set_idle_timeout(JammoMentor* mentor, gboolean activate);

static GList* mentors;

JammoMentor* jammo_mentor_new(ClutterActor* actor, gfloat standby_scale) {

	return JAMMO_MENTOR(g_object_new(JAMMO_TYPE_MENTOR, "normal-background-actor", actor, "standby-scale", standby_scale, "prefer-background-size", TRUE, NULL));
}

gfloat jammo_mentor_get_standby_scale(JammoMentor* mentor) {

	return mentor->priv->standby_scale;
}

void jammo_mentor_set_standby_scale(JammoMentor* mentor, gfloat scale) {
	if (mentor->priv->standby_scale != scale) {
		mentor->priv->standby_scale = scale;
		if (!mentor->priv->active_sample) {
			clutter_actor_set_scale(CLUTTER_ACTOR(mentor), scale, scale);
		}
		g_object_notify(G_OBJECT(mentor), "standby-scale");
	}
}

void jammo_mentor_speak(JammoMentor* mentor, const gchar* speech) {
	speak(mentor, speech, NULL, NULL);
}

void jammo_mentor_speak_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	speak(mentor, speech, callback, user_data);
}

void jammo_mentor_speak_once(JammoMentor* mentor, const gchar* speech) {
	if (!g_list_find(mentor->priv->spoken_speeches, speech)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, NULL, NULL);
	}
}

void jammo_mentor_speak_once_with_callback(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	if (!g_list_find(mentor->priv->spoken_speeches, speech)) {
		mentor->priv->spoken_speeches = g_list_prepend(mentor->priv->spoken_speeches, g_strdup(speech));
		speak(mentor, speech, callback, user_data);
	}
}

const gchar* jammo_mentor_get_idle_speech(JammoMentor* mentor) {

	return mentor->priv->idle_speech;
}

void jammo_mentor_set_idle_speech(JammoMentor* mentor, const gchar* speech) {
	if (g_strcmp0(mentor->priv->idle_speech, speech)) {
		g_free(mentor->priv->idle_speech);
		mentor->priv->idle_speech = g_strdup(speech);
		g_object_notify(G_OBJECT(mentor), "idle-speech");
	}
}

guint jammo_mentor_get_idle_timeout(JammoMentor* mentor) {

	return mentor->priv->idle_timeout;
}

void jammo_mentor_set_idle_timeout(JammoMentor* mentor, guint milliseconds) {
	if (mentor->priv->idle_timeout != milliseconds) {
		mentor->priv->idle_timeout = milliseconds;
		if (!mentor->priv->active_sample) {
			set_idle_timeout(mentor, TRUE);
		}
		g_object_notify(G_OBJECT(mentor), "idle-timeout");
	}
}

JammoMentor* jammo_mentor_get_default(void) {

	return (mentors ? JAMMO_MENTOR(mentors->data) : NULL);
}

static void jammo_mentor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoMentor* mentor;
	
	mentor = JAMMO_MENTOR(object);

	switch (prop_id) {
		case PROP_STANDBY_SCALE:
			jammo_mentor_set_standby_scale(mentor, g_value_get_float(value));
			break;
		case PROP_IDLE_SPEECH:
			jammo_mentor_set_idle_speech(mentor, g_value_get_string(value));
			break;
		case PROP_IDLE_TIMEOUT:
			jammo_mentor_set_idle_timeout(mentor, g_value_get_uint(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_mentor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoMentor* mentor;

	mentor = JAMMO_MENTOR(object);

        switch (prop_id) {
		case PROP_STANDBY_SCALE:
			g_value_set_float(value, mentor->priv->standby_scale);
			break;
		case PROP_IDLE_SPEECH:
			g_value_set_string(value, mentor->priv->idle_speech);
			break;
		case PROP_IDLE_TIMEOUT:
			g_value_set_uint(value, mentor->priv->idle_timeout);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_mentor_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_parent_class)->finalize(object);
}

static void jammo_mentor_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_parent_class)->dispose(object);
}

static void jammo_mentor_class_init(JammoMentorClass* mentor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(mentor_class);
	TangleButtonClass* button_class = TANGLE_BUTTON_CLASS(mentor_class);

	gobject_class->finalize = jammo_mentor_finalize;
	gobject_class->dispose = jammo_mentor_dispose;
	gobject_class->set_property = jammo_mentor_set_property;
	gobject_class->get_property = jammo_mentor_get_property;

//	button_class->clicked = jammo_mentor_clicked;

	g_type_class_add_private (gobject_class, sizeof (JammoMentorPrivate));
}

static void jammo_mentor_init(JammoMentor* mentor) {
	mentor->priv = G_TYPE_INSTANCE_GET_PRIVATE(mentor, JAMMO_TYPE_MENTOR, JammoMentorPrivate);
}

static void on_sample_stopped(JammoSample* sample, gpointer user_data) {
	TangleVault* vault;
	JammoMentor* mentor;
	JammoMentorSpokenCallback callback;
	const gchar* speech;
	gpointer data;
	
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 4, JAMMO_TYPE_MENTOR, &mentor, G_TYPE_POINTER, &callback, G_TYPE_STRING, &speech, G_TYPE_POINTER, &data);
	
	g_object_unref(mentor->priv->active_sample);
	mentor->priv->active_sample = NULL;
	
	callback(mentor, speech, data);
}

static void speak(JammoMentor* mentor, const gchar* speech, JammoMentorSpokenCallback callback, gpointer user_data) {
	gchar* filename;
	TangleVault* vault;
	
	if (mentor->priv->active_sample) {
		jammo_sample_stop(mentor->priv->active_sample);
	}
	
	if (mentor->priv->script && (filename = clutter_script_lookup_filename(mentor->priv->script, speech))) {
		mentor->priv->active_sample = jammo_sample_new_from_file(filename);
		g_free(filename);
	} else {
		mentor->priv->active_sample = jammo_sample_new_from_file(speech);
	}
	
	vault = tangle_vault_new(4, JAMMO_TYPE_MENTOR, mentor, G_TYPE_POINTER, callback, G_TYPE_STRING, speech, G_TYPE_POINTER, user_data);
	tangle_signal_connect_vault(mentor->priv->active_sample, "stopped", G_CALLBACK(on_sample_stopped), vault);
	
	jammo_sample_play(mentor->priv->active_sample);
}

static void set_idle_timeout(JammoMentor* mentor, gboolean activate) {
	/* TOFO */
}
#endif
