/*
 * jammo-track-view.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-track-view.h"
#include "jammo-sample-button.h"
#include "../meam/jammo-sample.h"

G_DEFINE_TYPE(JammoTrackView, jammo_track_view, TANGLE_TYPE_DROPPABLE_ACTOR);

enum {
	PROP_0,
	PROP_TRACK
};

struct _JammoTrackViewPrivate {
	JammoEditingTrack* track;
	guint slots;
	gulong slot_duration;
};

static void on_actor_added(TangleWidget* widget, ClutterActor* actor, gpointer user_data);
static void on_actor_removed(TangleWidget* widget, ClutterActor* actor, gpointer user_data);

ClutterActor* jammo_track_view_new(JammoEditingTrack* track) {

	return CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_TRACK_VIEW, "track", track, NULL));
}

static gboolean tangle_track_view_is_dragging_droppable(TangleDroppableActor* droppable_actor, TangleDragging* dragging) {
	
	return JAMMO_IS_SAMPLE_BUTTON(dragging->draggable_actor);
}

static gboolean tangle_track_view_handle_dropping(TangleDroppableActor* droppable_actor, TangleDragging* dragging, gboolean accepted) {
	JammoTrackView* track_view;
	TangleWidget* widget;
	gfloat x, y;
	JammoSampleButton* sample_button;
	ClutterActor* copy_of_sample_button;
	
	track_view = JAMMO_TRACK_VIEW(droppable_actor);
	widget = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(track_view)));
	
	TANGLE_DROPPABLE_ACTOR_CLASS(jammo_track_view_parent_class)->handle_dropping(droppable_actor, dragging, accepted);

	if (accepted && clutter_actor_transform_stage_point(CLUTTER_ACTOR(track_view), dragging->x, dragging->y, &x, &y)) {
		sample_button = JAMMO_SAMPLE_BUTTON(dragging->draggable_actor);
		copy_of_sample_button = jammo_sample_button_new_from_existing(sample_button);
		/* tangle_draggable_actor_set_floating_mode_dragged(TANGLE_DRAGGABLE_ACTOR(copy_of_sample_button), TANGLE_FLOATING_MODE_COLLAPSE); */
		/* TODO: position */
		clutter_actor_set_x(copy_of_sample_button, x - dragging->anchor_x);
		tangle_widget_add_after(widget, copy_of_sample_button, NULL, NULL, NULL);
	}

	return FALSE;
}

static GObject* jammo_track_view_constructor(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties) {
	GObject* object;
	JammoTrack* track;
	ClutterActor* widget;
	ClutterActor* grid;

	g_return_val_if_fail(tangle_lookup_construct_properties(type, n_construct_properties, construct_properties, "track", &track, NULL), NULL);
	g_return_val_if_fail(track != NULL, NULL);

	widget = tangle_widget_new();
	tangle_widget_set_prefer_background_size(TANGLE_WIDGET(widget), TRUE);
	grid = tangle_grid_new(79.0, 79.0);
	clutter_actor_set_size(grid, 800.0, 80.0);
	tangle_widget_set_background_actor(TANGLE_WIDGET(widget), grid);

	object = tangle_construct_with_extra_properties(G_OBJECT_CLASS(jammo_track_view_parent_class)->constructor, type, n_construct_properties, construct_properties, "wrapped", widget, NULL);

	g_signal_connect(widget, "actor-added", G_CALLBACK(on_actor_added), object);
	g_signal_connect(widget, "actor-removed", G_CALLBACK(on_actor_removed), object);

	return object;
}

static void jammo_track_view_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoTrackView* track_view;
	
	track_view = JAMMO_TRACK_VIEW(object);

	switch (prop_id) {
		case PROP_TRACK:
			track_view->priv->track = JAMMO_EDITING_TRACK(g_value_get_object(value));
			g_object_ref(track_view->priv->track);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_track_view_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoTrackView* track_view;

	track_view = JAMMO_TRACK_VIEW(object);

        switch (prop_id) {
		case PROP_TRACK:
			g_value_set_object(value, track_view->priv->track);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_track_view_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_track_view_parent_class)->finalize(object);
}

static void jammo_track_view_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_track_view_parent_class)->dispose(object);
}

static void jammo_track_view_class_init(JammoTrackViewClass* track_view_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(track_view_class);
	TangleDroppableActorClass* droppable_actor_class = TANGLE_DROPPABLE_ACTOR_CLASS(track_view_class);

	gobject_class->constructor = jammo_track_view_constructor;
	gobject_class->finalize = jammo_track_view_finalize;
	gobject_class->dispose = jammo_track_view_dispose;
	gobject_class->set_property = jammo_track_view_set_property;
	gobject_class->get_property = jammo_track_view_get_property;
	
	droppable_actor_class->is_dragging_droppable = tangle_track_view_is_dragging_droppable;
	droppable_actor_class->handle_dropping = tangle_track_view_handle_dropping;

	/**
	 * JammoTrackView:track:
	 */
	g_object_class_install_property(gobject_class, PROP_TRACK,
	                                g_param_spec_object("track",
	                                "Track",
	                                "The track of this view",
	                                JAMMO_TYPE_EDITING_TRACK,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (JammoTrackViewPrivate));
}

static void jammo_track_view_init(JammoTrackView* track_view) {
	track_view->priv = G_TYPE_INSTANCE_GET_PRIVATE(track_view, JAMMO_TYPE_TRACK_VIEW, JammoTrackViewPrivate);
}

static void on_actor_added(TangleWidget* widget, ClutterActor* actor, gpointer user_data) {
	JammoTrackView* track_view;
	JammoSampleButton* sample_button;
	
	track_view = JAMMO_TRACK_VIEW(user_data);
	sample_button = JAMMO_SAMPLE_BUTTON(actor);
	
	jammo_editing_track_add_sample(track_view->priv->track, jammo_sample_button_get_sample(sample_button), clutter_actor_get_x(actor) * 50000000);
}

static void on_actor_removed(TangleWidget* widget, ClutterActor* actor, gpointer user_data) {
	JammoTrackView* track_view;
	JammoSampleButton* sample_button;
	
	track_view = JAMMO_TRACK_VIEW(user_data);
	sample_button = JAMMO_SAMPLE_BUTTON(actor);
	
	jammo_editing_track_remove_sample(track_view->priv->track, jammo_sample_button_get_sample(sample_button));
}
