/*
 * jammo-sequencer.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */
 
#include "jammo-sequencer.h"
#include "jammo-meam.h"
#include "jammo-meam-private.h"
#include "jammo-pipeline.h"

/**
 * SECTION:jammo-sequencer
 * @Short_description: A multitrack sequencer
 * @Title: JammoSequencer
 *
 * A #JammoSequencer consists of multiple tracks that
 * each can play own sound simultaneously.
 *
 * A #JammoSequencer must be unreferenced with g_object_unref()
 * after usage. It automatically frees
 * associated #JammoTrack objects.
 */

G_DEFINE_TYPE(JammoSequencer, jammo_sequencer, G_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_READY
};

enum {
	STOPPED,
	LAST_SIGNAL
};

struct _JammoSequencerPrivate {
	GstElement* pipeline;
	GstElement* adder;
	GList* tracks;
	guint ready : 1;
	guint playing : 1;
};

static gboolean on_bus_message(GstBus* bus, GstMessage* message, gpointer data);
static void on_duration_notify(GObject* object, GParamSpec* param_spec, gpointer user_data);

static guint signals[LAST_SIGNAL] = { 0 };

/**
 * jammo_sequencer_new:
 *
 * Creates a new sequencer.
 *
 * Return value: a #JammoSequencer
 */
JammoSequencer* jammo_sequencer_new() {

	return JAMMO_SEQUENCER(g_object_new(JAMMO_TYPE_SEQUENCER, NULL));
}

/**
 * jammo_sequencer_is_ready:
 * @sequencer: a #JammoSequencer
 *
 * Tells whether the sequencer is ready to play immediately
 * or not. A sequencer may not be ready if, for example,
 * some track or sample is still loading content or
 * calculating the duration.
 *
 * Return value: TRUE if the sequencer is ready, FALSE otherwise.
 */
gboolean jammo_sequencer_is_ready(JammoSequencer* sequencer) {
	
	return sequencer->priv->ready;
}

/**
 * jammo_sequencer_play:
 * @sequencer: a #JammoSequencer
 *
 * Starts playing all associated tracks simultaneously.
 */
void jammo_sequencer_play(JammoSequencer* sequencer) {
	if (sequencer->priv->ready) {
		gst_element_set_state(sequencer->priv->pipeline, GST_STATE_PLAYING);
	}
	sequencer->priv->playing = TRUE;
}

/**
 * jammo_sequencer_stop:
 * @sequencer: a #JammoSequencer
 *
 * Stops playing started with #jammo_sequencer_play(). The
 * next #jammo_sequencer_play() call starts from the beginning.
 */
void jammo_sequencer_stop(JammoSequencer* sequencer) {
	gst_element_set_state(sequencer->priv->pipeline, GST_STATE_NULL);
	g_signal_emit(sequencer, signals[STOPPED], 0);
	sequencer->priv->playing = FALSE;
}

/**
 * jammo_sequencer_stop:
 * @sequencer: a #JammoSequencer
 *
 * Pauses playing started with #jammo_sequencer_play().
 * The next #jammo_sequencer_play() call start from the
 * position where the sequencer was paused.
 */
void jammo_sequencer_pause(JammoSequencer* sequencer) {
	gst_element_set_state(sequencer->priv->pipeline, GST_STATE_PAUSED);
	sequencer->priv->playing = FALSE;
}

/**
 * jammo_sequencer_get_track_count:
 * @sequencer: a #JammoSequencer
 *
 * Return how many #JammoTracks are associated with the
 * sequencer.
 *
 * Return value: track count
 */
guint jammo_sequencer_get_track_count(JammoSequencer* sequencer) {

	return g_list_length(sequencer->priv->tracks);
}

/**
 * jammo_sequencer_get_track:
 * @sequencer: a #JammoSequencer
 * @index: the index of a requested track
 *
 * Returns a #JammoTrack by its index number.
 * The index must be lower than track count
 * (see #jammo_sequencer_get_track_count()).
 *
 * Return value: a #JammoTrack
 */
JammoTrack* jammo_sequencer_get_track(JammoSequencer* sequencer, guint index) {
	JammoTrack* track = NULL;
	GList* list;
	
	if ((list = g_list_nth(sequencer->priv->tracks, index))) {
		track = JAMMO_TRACK(list->data);
	}
	
	return track;
}

/**
 * jammo_sequencer_add_track:
 * @sequencer: a #JammoSequencer
 * @track: a #JammoTrack
 *
 * Associates the given track to the sequencer.
 *
 * This function sinks the floating reference of
 * a #JammoTrack.
 */
void jammo_sequencer_add_track(JammoSequencer* sequencer, JammoTrack* track) {
	GstElement* element;
	GstPad* pad;
	
	if (g_list_find(sequencer->priv->tracks, track)) {
		g_warning("A track is already added into a sequencer.");
	} else {
		sequencer->priv->tracks = g_list_prepend(sequencer->priv->tracks, g_object_ref_sink(track));
		element = _jammo_track_get_element(track);
		gst_bin_add(GST_BIN(sequencer->priv->pipeline), element);
		if ((pad = gst_element_get_static_pad(element, "src"))) {
			gst_element_link(element, sequencer->priv->adder);
			gst_object_unref(pad);
		}
		g_signal_connect(track, "notify::duration", G_CALLBACK(on_duration_notify), sequencer);
		if (sequencer->priv->ready &&
		    jammo_track_get_duration(track) == JAMMO_DURATION_INVALID) {
			if (sequencer->priv->playing) {
				gst_element_set_state(sequencer->priv->pipeline, GST_STATE_NULL);
			}
			sequencer->priv->ready = FALSE;
			g_object_notify(G_OBJECT(sequencer), "ready");
		}
	}
}

/**
 * jammo_sequencer_get_duration:
 * @sequencer: a #JammoSequencer
 *
 * Calculates the total duration of the all tracks.
 * Specifically, the total duration is the duration 
 * of the longest track.
 *
 * Return value: duration in nanoseconds.
 */
guint64 jammo_sequencer_get_duration(JammoSequencer* sequencer) {
	guint64 duration = (guint64)0;
	GList* list;
	guint64 d;
	
	for (list = sequencer->priv->tracks; list; list = list->next) {
		if ((d = jammo_track_get_duration(JAMMO_TRACK(list->data))) > duration) {
			duration = d;
		}
	}
	
	return duration;
}

static void jammo_sequencer_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoSequencer* sequencer;
	
	sequencer = JAMMO_SEQUENCER(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_sequencer_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoSequencer* sequencer;

	sequencer = JAMMO_SEQUENCER(object);

        switch (prop_id) {
		case PROP_READY:
			g_value_set_boolean(value, sequencer->priv->ready);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_sequencer_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_sequencer_parent_class)->finalize(object);
}

static void jammo_sequencer_dispose(GObject* object) {
	JammoSequencer* sequencer;

	sequencer = JAMMO_SEQUENCER(object);
	g_list_foreach(sequencer->priv->tracks, (GFunc)g_object_unref, NULL);
	g_list_free(sequencer->priv->tracks);
	sequencer->priv->tracks = NULL;

	G_OBJECT_CLASS(jammo_sequencer_parent_class)->dispose(object);
}

static void jammo_sequencer_class_init(JammoSequencerClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->finalize = jammo_sequencer_finalize;
	gobject_class->dispose = jammo_sequencer_dispose;
	gobject_class->set_property = jammo_sequencer_set_property;
	gobject_class->get_property = jammo_sequencer_get_property;

	/**
	 * JammoSequencer:ready:
	 *
	 * Whether the sequencer is ready to play or not.
	 */
	g_object_class_install_property(gobject_class, PROP_READY,
	                                g_param_spec_boolean("ready",
	                                "Ready",
	                                "Whether the sequencer is ready to play or not",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSequencer::stopped:
	 * @sequencer: the object which received the signal
	 *
	 * The ::stopped signal is emitted when the playing (of all tracks) has stopped.
	 */
	signals[STOPPED] = g_signal_new("stopped", G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST, 0,
					       NULL, NULL,
					       g_cclosure_marshal_VOID__VOID,
					       G_TYPE_NONE, 0);

	g_type_class_add_private(gobject_class, sizeof(JammoSequencerPrivate));
}

static void jammo_sequencer_init(JammoSequencer* sequencer) {
	GstBus* bus;
	GstElement* sink;

	sequencer->priv = G_TYPE_INSTANCE_GET_PRIVATE(sequencer, JAMMO_TYPE_SEQUENCER, JammoSequencerPrivate);
	sequencer->priv->pipeline = jammo_pipeline_new(NULL);
	bus = gst_pipeline_get_bus(GST_PIPELINE(sequencer->priv->pipeline));
	gst_bus_add_watch(bus, on_bus_message, sequencer);
	gst_object_unref (bus);

	if (!(sequencer->priv->adder = gst_element_factory_make("adder", NULL))) {
		g_warning("Could not create GStreamer element 'adder'.\n");
	} else if (!(sink = gst_element_factory_make("autoaudiosink", NULL))) {
		g_warning("Could not create GStreamer element 'autoaudiosink'.\n");	
	} else {
		gst_bin_add_many(GST_BIN(sequencer->priv->pipeline), sequencer->priv->adder, sink, NULL);
		gst_element_link(sequencer->priv->adder, sink);
	}
}

static gboolean on_bus_message(GstBus* bus, GstMessage* message, gpointer data) {
	JammoSequencer* sequencer;
	GError* error;
	gchar* debug;
	GstObject* source;
	GList* list;
	JammoTrack* track;
	GstObject* object;
	
	sequencer = JAMMO_SEQUENCER(data);
	switch (GST_MESSAGE_TYPE(message)) {
		case GST_MESSAGE_EOS:
			gst_element_set_state(sequencer->priv->pipeline, GST_STATE_NULL);
			g_signal_emit(sequencer, signals[STOPPED], 0);
			break;
		case GST_MESSAGE_ERROR:
			gst_message_parse_error(message, &error, &debug);
			g_print("Error: %s\n", error->message);
			g_error_free(error);
			g_free(debug);
			break;
		case GST_MESSAGE_ELEMENT:
			source = GST_MESSAGE_SRC(message);
			for (list = sequencer->priv->tracks; list; list = list->next) {
				track = JAMMO_TRACK(list->data);
				object = GST_OBJECT(_jammo_track_get_element(track));
				if (object == source || gst_object_has_ancestor(source, object)) {
					_jammo_track_receive_message(track, message);
					break;
				}
			}
			break;
		default:
			break;
	}
	
	return TRUE;
}

static void on_duration_notify(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	JammoSequencer* sequencer;
	JammoTrack* track;
	gboolean ready;
	GList* list;

	sequencer = JAMMO_SEQUENCER(user_data);
	track = JAMMO_TRACK(object);

	if (jammo_track_get_duration(track) == JAMMO_DURATION_INVALID) {
		if (sequencer->priv->ready) {
			if (sequencer->priv->playing) {
				gst_element_set_state(sequencer->priv->pipeline, GST_STATE_NULL);
			}
			sequencer->priv->ready = FALSE;
			g_object_notify(G_OBJECT(sequencer), "ready");
		}
	} else if (!sequencer->priv->ready) {
		ready = TRUE;
		for (list = sequencer->priv->tracks; list; list = list->next) {
			if (list->data != (gpointer)track &&
			    jammo_track_get_duration(JAMMO_TRACK(list->data)) == JAMMO_DURATION_INVALID) {
				ready = FALSE;
				break;
			}
		}
		if (ready) {
			sequencer->priv->ready = TRUE;
			g_object_notify(G_OBJECT(sequencer), "ready");
			if (sequencer->priv->playing) {
				gst_element_set_state(sequencer->priv->pipeline, GST_STATE_PLAYING);
			}
		}
	}	
}
